/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: TasksManager.java 21 2011-05-09 16:43:58Z sletellier $
 * $HeadURL: http://svn.chorem.org/svn/vradi/tags/vradi-0.6/vradi-services/src/main/java/org/chorem/vradi/services/tasks/TasksManager.java $
 * %%
 * Copyright (C) 2010 Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

package org.chorem.vradi.services.tasks;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.vradi.VradiConstants;
import org.chorem.vradi.VradiServiceConfigurationHelper;
import org.chorem.vradi.entities.XmlStream;
import org.chorem.vradi.services.managers.BindingManager;
import org.chorem.vradi.services.managers.FormManager;
import org.chorem.vradi.services.managers.MailingManager;
import org.chorem.vradi.services.managers.ThesaurusManager;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.services.WikittyEvent;
import org.nuiton.wikitty.services.WikittyListener;

import java.util.Calendar;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;

/**
 * This manager handle tasks management in
 * {@code org.chorem.vradi.services.tasks} package.
 *
 * @author chatellier
 * @version $Revision: 21 $
 *          <p/>
 *          Last update : $Date: 2011-05-09 18:43:58 +0200 (lun., 09 mai 2011) $
 *          By : $Author: sletellier $
 */
public class TasksManager implements WikittyListener {

    private static final Log log = LogFactory.getLog(TasksManager.class);

    protected ApplicationConfig config;

    /** Cron task to check mails every 10 minutes. */
    protected Timer mailCronTimer;

    /** Cron task to import streams. */
    protected ScheduledExecutorService xmlStreamScheduler;

    /** Cron task to enable auto send task. */
    protected ScheduledExecutorService autoSendScheduler;

    protected ScheduledFuture<?> autoSendTask;

    protected ReceiveMailTasks receiveMailTasks;

    /** Map entre les id des flux, et les taches gérant. */
    protected Map<String, ScheduledFuture<?>> xmlStreamImportTasks;

    protected WikittyProxy wikittyProxy;

    protected MailingManager mailingManager;

    protected BindingManager bindingManager;

    protected ThesaurusManager thesaurusManager;

    protected FormManager formManager;

    public TasksManager(ApplicationConfig config,
                        WikittyProxy wikittyProxy,
                        ThesaurusManager thesaurusManager,
                        FormManager formManager,
                        MailingManager mailingManager,
                        BindingManager bindingManager) {

        this.config = config;
        this.wikittyProxy = wikittyProxy;
        this.mailingManager = mailingManager;
        this.bindingManager = bindingManager;
        this.thesaurusManager = thesaurusManager;
        this.formManager = formManager;
        xmlStreamImportTasks = new HashMap<String, ScheduledFuture<?>>();
    }

    /** Init all tasks. */
    public void initTasks() {
        initMailTask();
        initXmlStreamTasks();

        // If hour is fill in properties
        String sendTaskHour = VradiServiceConfigurationHelper.getSendTaskHour(config);
        if (!StringUtils.isEmpty(sendTaskHour)) {

            // Create autoSend task
            initAutoSendTask(sendTaskHour);
        }
    }

    protected void initAutoSendTask(String sendTaskHour) {

        // les imports sont fait sequentiellement
        autoSendScheduler = Executors.newScheduledThreadPool(1);

        TimerTask task = new AutoSendTasks(formManager, mailingManager);

        autoSendTask = initDaylyTask(autoSendScheduler, task, sendTaskHour);
    }

    /** Init and start mail tasks (check every 10 minutes) */
    protected void initMailTask() {
        mailCronTimer = new Timer("vradi-mail-task");
        receiveMailTasks = new ReceiveMailTasks(mailingManager);
        mailCronTimer.schedule(receiveMailTasks, 0, 10 * 60 * 1000);
    }

    /** Init and start xml stream tasks. */
    protected void initXmlStreamTasks() {

        // les imports sont fait sequentiellement
        xmlStreamScheduler = Executors.newScheduledThreadPool(1);

        List<XmlStream> xmlStreams = bindingManager.getAllXmlStreams();

        for (XmlStream xmlStream : xmlStreams) {
            if (xmlStream != null) {
                initXmlStreamTask(xmlStream);
            }
        }
    }

    /**
     * Init single xml stream task.
     *
     * @param xmlStream xml stream
     */
    protected void initXmlStreamTask(XmlStream xmlStream) {
        String streamImportTime = xmlStream.getImportTime();

        if (StringUtils.isBlank(streamImportTime)) {
            // par defaut minuit, si pas de données
            streamImportTime = "0:0";
        }

        // TODO EC20100921 prevoir le cas où une tâche dure plus d'une journée
        TimerTask autoImportTask;
        if (xmlStream.getAutoAssign()) {
            autoImportTask = new AssignXmlStreamTasks(config, wikittyProxy, bindingManager, thesaurusManager, formManager, xmlStream.getWikittyId());
        } else {
            autoImportTask = new ImportXmlStreamTasks(config, wikittyProxy, bindingManager, xmlStream.getWikittyId());
        }
        ScheduledFuture<?> sheduledImportTask = initDaylyTask(xmlStreamScheduler, autoImportTask, streamImportTime);

        if (log.isInfoEnabled()) {
            log.info("Starting import of xml stream " + xmlStream.getName() + " at " + streamImportTime);
        }
        xmlStreamImportTasks.put(xmlStream.getWikittyId(), sheduledImportTask);
    }

    public ScheduledFuture<?> initDaylyTask(ScheduledExecutorService scheduledExecutorService, TimerTask task, String hour) {

        Matcher matchImportTime = VradiConstants.XML_STREAM_IMPORT_TIME.matcher(hour);
        if (matchImportTime.find()) {
            String hours = matchImportTime.group(1);
            String minutes = matchImportTime.group(2);

            // calcule le nombre de minutes entre maitenant
            // et l'heure d'import du flux
            Calendar nowCalendar = Calendar.getInstance();
            int nowHours = nowCalendar.get(Calendar.HOUR_OF_DAY);
            int nowMinutes = nowCalendar.get(Calendar.MINUTE);

            int toHours = Integer.parseInt(hours);
            int toMinutes = Integer.parseInt(minutes);

            int delayInMinute = toHours * 60 + toMinutes - (nowHours * 60 + nowMinutes);
            if (delayInMinute < 0) {
                delayInMinute += 24 * 60;
            }
            if (log.isInfoEnabled()) {
                log.info(task.getClass().getName() + " - in scheduler, delai : " + delayInMinute + "min, period " + 24 * 60 + "min");
            }
            return scheduledExecutorService.scheduleAtFixedRate(task, delayInMinute, 24 * 60, TimeUnit.MINUTES);

        }
        if (log.isWarnEnabled()) {
            log.warn("Can't parse " + hour);
        }
        return null;
    }

    /**
     * Cancel import task for specified id and restart a new task if requiered.
     *
     * @param xmlStreamId xml stream id
     * @param restart     restart a new import task for stream id
     */
    protected void cancelXmlImportTask(String xmlStreamId, boolean restart) {

        // just cancel task
        if (log.isDebugEnabled()) {
            log.debug("Canceling import stream task for " + xmlStreamId);
        }
        ScheduledFuture<?> sheduledImportTask = xmlStreamImportTasks.get(xmlStreamId);
        if (sheduledImportTask != null) {
            xmlStreamImportTasks.remove(xmlStreamId);
            sheduledImportTask.cancel(false);
        } else {
            // ca peut arriver lors de la creation de stream (pas de précédente)
            if (log.isDebugEnabled()) {
                log.debug("Nothing to cancel for wikitty id " + xmlStreamId);
            }
        }

        // restart when stream is updated
        if (restart) {
            XmlStream xmlStream = wikittyProxy.restore(XmlStream.class, xmlStreamId);
            if (xmlStream == null) {
                if (log.isErrorEnabled()) {
                    log.error("Can't restart import task for non existing xml stream " + xmlStreamId);
                }
            } else {
                initXmlStreamTask(xmlStream);
            }
        }
    }

    @Override
    public void putWikitty(WikittyEvent event) {

        if (log.isDebugEnabled()) {
            log.debug("Receiving wikitty event : " + event);
        }

        // ecoute les modification faites sur les formulaires
        Map<String, Wikitty> wikittiesMap = event.getWikitties();
        Collection<Wikitty> wikitties = wikittiesMap.values();
        for (Wikitty wikitty : wikitties) {
            if (wikitty instanceof XmlStream) {
                cancelXmlImportTask(((XmlStream) wikitty).getWikittyId(), true);
            }
        }
    }

    @Override
    public void removeWikitty(WikittyEvent event) {

        if (log.isDebugEnabled()) {
            log.debug("Receiving wikitty event : " + event);
        }

        // ecoute les modification faites sur les formulaires
        // on ne peut pas savoir si c'est un xmlstream
        // mais on c'est si l'id est lancé
        Map<String, Wikitty> wikittyMap = event.getWikitties();
        if (wikittyMap != null) {
            Set<String> wikittyIds = wikittyMap.keySet();
            for (String wikittyId : wikittyIds) {
                // if pas indispensable, mais ca evite des log inutiles
                if (xmlStreamImportTasks.containsKey(wikittyId)) {
                    cancelXmlImportTask(wikittyId, false);
                }
            }
        }
    }

    @Override
    public void clearWikitty(WikittyEvent event) {

    }

    @Override
    public void putExtension(WikittyEvent event) {

    }

    @Override
    public void removeExtension(WikittyEvent event) {

    }

    @Override
    public void clearExtension(WikittyEvent event) {

    }
}
