/*
 * #%L
 * Pollen :: UI (strust2)
 * 
 * $Id: AbstractPollenAuthorization.java 3394 2012-05-28 15:27:04Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3.1.1/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/security/AbstractPollenAuthorization.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui.security;

import com.google.common.base.Preconditions;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.shiro.util.StringUtils;
import org.apache.shiro.web.filter.authz.AuthorizationFilter;
import org.apache.shiro.web.util.WebUtils;
import org.chorem.pollen.bean.PollUri;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.services.DefaultPollenServiceContext;
import org.chorem.pollen.services.PollenServiceContext;
import org.chorem.pollen.services.PollenServiceFactory;
import org.chorem.pollen.ui.PollenApplicationContext;
import org.chorem.pollen.ui.PollenSession;
import org.chorem.pollen.ui.PollenUIUtils;
import org.nuiton.topia.TopiaContext;
import org.nuiton.web.filter.TopiaTransactionFilter;

import javax.servlet.ServletContext;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public abstract class AbstractPollenAuthorization extends AuthorizationFilter {

    /**
     * Pattern to detects uri from a pollen url :
     * group 1 is action namespace
     * group 2 is action name
     * group 3 is pollUri
     * group 4 is poll page (optional)
     * group 5 is method (optional)
     */
    public static final Pattern URI_PATTERN =
            Pattern.compile("/(.[^/]+)/(.[^/]+)/(.[^!/]+)(/.[^!/]*){0,1}(!.*){0,1}");

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(AbstractPollenAuthorization.class);

    protected AbstractPollenAuthorization() {
        if (log.isInfoEnabled()) {
            log.info("Init - " + this);
        }
    }

    @Override
    public ServletContext getServletContext() {
        ServletContext servletContext = super.getServletContext();
        if (servletContext == null) {
            servletContext = PollenUIUtils.getServletContext();
            Preconditions.checkNotNull(servletContext);
            // store it back
            setServletContext(servletContext);
        }
        return servletContext;
    }

    protected PollUri getPollUri(ServletRequest request) {
        String servletPath = ((HttpServletRequest) request).getServletPath();
        Matcher m = URI_PATTERN.matcher(servletPath);

        PollUri result = null;
        if (m.find()) {

            String uriId = m.group(3);
            result = PollUri.newPollUri(uriId);

        }
        return result;
    }

    protected UserAccount getPollenUSerAccount(ServletRequest request) {
        PollenSession pollenSession = PollenSession.get(request);

        return pollenSession.getUserAccount();
    }

    protected PollenServiceContext getServiceContext(ServletRequest request) {
        PollenApplicationContext applicationContext =
                PollenApplicationContext.get(getServletContext());

        PollenServiceFactory serviceFactory = new PollenServiceFactory();

        TopiaContext transaction =
                TopiaTransactionFilter.getTransaction(request);

        Locale locale = Locale.getDefault();
        return DefaultPollenServiceContext.newContext(
                locale,
                transaction,
                applicationContext.getConfiguration(),
                serviceFactory
        );
    }

    @Override
    protected final boolean onAccessDenied(ServletRequest request,
                                           ServletResponse response) throws IOException {
        String unauthorizedUrl = getUnauthorizedUrl();
        //SHIRO-142 - ensure that redirect _or_ error code occurs - both cannot happen due to response commit:
        if (StringUtils.hasText(unauthorizedUrl)) {
            WebUtils.issueRedirect(request, response, unauthorizedUrl);
        } else {
            WebUtils.toHttp(response).sendError(HttpServletResponse.SC_UNAUTHORIZED);
        }
        return false;
    }
}
