/*
 * #%L
 * Pollen :: UI (strust2)
 * 
 * $Id: VoteForPoll.java 3395 2012-05-28 15:34:46Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3.1.1/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/actions/poll/VoteForPoll.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui.actions.poll;

import com.google.common.base.Preconditions;
import com.opensymphony.xwork2.Preparable;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.bean.PollUri;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.business.persistence.VoteToChoice;
import org.chorem.pollen.common.PollType;

/**
 * Votes to a poll.
 *
 * @author tchemit <chemit@codelutin.com>
 * @author fdesbois <fdesbois@codelutin.com>
 * @since 1.3
 */
public class VoteForPoll extends AbstractVoteAction implements Preparable {

    private static final Log log = LogFactory.getLog(VoteForPoll.class);

    private static final long serialVersionUID = 1L;

    @Override
    public void prepare() throws Exception {

        prepareVotePage();
    }

    @Override
    public void validate() {

        Vote vote = getVote();
        Preconditions.checkNotNull(vote);

        PollAccount pollAccount = getVote().getPollAccount();
        Preconditions.checkNotNull(pollAccount);

        Poll poll = getPoll();

        if (!vote.isAnonymous()) {

            String name = pollAccount.getVotingId();
            if (StringUtils.isBlank(name)) {
                addFieldError("pollAccount.votingId",
                              _("pollen.error.pollAccount.votingId.required"));
            }

            // check if the new pollAccount (topiaId = null) has already voted
            if (pollAccount.getTopiaId() == null &&
                getVoteService().hasAlreadyVoted(name, poll)) {
                addFieldError("pollAccount.votingId",
                              _("pollen.error.user.alreadyVoted"));
            }
        }


        int nbVotes = 0;
        int totalValues = 0;
        for (VoteToChoice voteToChoice : vote.getChoiceVoteToChoice()) {

            Integer value = voteToChoice.getVoteValue();
            // XXX-fdesbois-2012-04-11 : for a VoteCountingType#NUMBER the difference between 0 and null value could be important
            if (value != null && value != 0) {
                nbVotes++;
                totalValues += value;
            }
        }

        // check that the number of choices is not greater than maxChoiceNb
        int maxChoice = getPoll().getMaxChoiceNb();
        if (maxChoice != 0 && nbVotes > maxChoice) {
            addFieldError("vote.choices",
                          _("pollen.error.vote.maxChoiceNb", maxChoice));
        }

        // check for percentage that the sum of all values equals 100
        if (isPercentageVoteCounting() && totalValues != 100) {
            addFieldError("vote.choices", _("pollen.error.vote.percentage"));
        }
    }

    @Override
    public String execute() throws Exception {

        // REMARQUES :
        // - pas le droit de modif si le pollAccount est rattaché a un userAccount et qu'on est pas loggé ?!?
        // - Ce serait plus simple que l'Admin ne puisse jamais voter, il ne peut que modérer les votes

        if (getVote().getTopiaId() != null) {
            getVoteService().updateVote(getVote());

        } else {
            Vote voteCreated = getVoteService().createVote(getVote());
            getPollService().addVoteToPoll(getPoll(), voteCreated);
        }

        //TODO tchemit-2012-05-18 Why clean messages and just messages here ?
        clearFlashMessages();

        // For free Poll, display the update Url (useless if user is logged)
        if (PollType.FREE == getPoll().getPollType() && !isUserLoggued()) {

            String pollId = getPoll().getPollId();
            String accountId = getPollAccount().getAccountId();
            PollUri pollUri = PollUri.newPollUri(pollId, accountId);

            String updateUrl = getVoteService().getUpdateVoteUrl(pollUri);

            if (log.isDebugEnabled()) {
                log.debug(String.format(
                        "UpdateURL for poll '%s' and account '%s' = %s",
                        pollId, accountId, updateUrl));
            }

            addFlashMessage(
                    _("pollen.information.vote.createdWithUpdateUrl", updateUrl));

        } else {

            addFlashMessage(_("pollen.information.vote.created"));
        }
        return SUCCESS;
    }
}
