/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id: ContactModelFactory.java 1424 2011-12-16 10:29:50Z bleny $
 * $HeadURL: http://svn.forge.codelutin.com/svn/wao/tags/wao-3.2.1/wao-ui/src/main/java/fr/ifremer/wao/ui/services/ContactModelFactory.java $
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

package fr.ifremer.wao.ui.services;

import fr.ifremer.wao.WaoUtils;
import fr.ifremer.wao.bean.ObsProgram;
import fr.ifremer.wao.bean.UserRole;
import fr.ifremer.wao.entity.Contact;
import fr.ifremer.wao.entity.FishingZone;
import fr.ifremer.wao.entity.TerrestrialLocation;
import fr.ifremer.wao.entity.WaoUser;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.PropertyConduit;
import org.apache.tapestry5.beaneditor.BeanModel;
import org.apache.tapestry5.services.BeanModelSource;

import java.lang.annotation.Annotation;

/**
 * This class is used in {@link fr.ifremer.wao.ui.pages.Contacts} page to fix
 * what column can be see (depending on view option, role, program), and how
 * to display the content in each column.
 *
 * Created: 18 nov. 2009
 *
 * @author fdesbois <fdesbois@codelutin.com>
 */
public class ContactModelFactory {

    protected UserRole userRole;
    protected ObsProgram obsProgram;
    protected boolean fullView;

    /** the model the factory will build */
    protected BeanModel<Contact> model;
    private PropertyConduit observationTypePropertyConduit;

    public ContactModelFactory(ObsProgram obsProgram, UserRole userRole, boolean fullView) {
        this.userRole = userRole;
        this.obsProgram = obsProgram;
        this.fullView = fullView;
    }

    /** add a property in the model and make it not sortable
     * @param property the name of the property to add
     * @param conduit a PropertyConduit to get the String representation of the
     *        value of the property, may be null if toString can be used */
    protected void addNonSortableProperty(String property, PropertyConduit conduit) {
        model.add(property, conduit);
        model.get(property).sortable(false);
    }

    /** add ALL properties to the bean model. */
    protected void initModel() {

        // order is meaningful, properties will be displayed in columnsForImport from left to right

        addNonSortableProperty(Contact.PROPERTY_CREATION_DATE, null);
        addNonSortableProperty(Contact.PROPERTY_MAIN_OBSERVER, getUserPropertyConduit());
        addNonSortableProperty(WaoUser.PROPERTY_COMPANY, getCompanyPropertyConduit());
        addNonSortableProperty(Contact.PROPERTY_SAMPLE_ROW, getSampleRowPropertyConduit());
        addNonSortableProperty("program", getProgramPropertyConduit());
        addNonSortableProperty("professionCode", getProfessionCodePropertyConduit());
        addNonSortableProperty("professionLibelle", getProfessionLibellePropertyConduit());
        addNonSortableProperty("professionSpecies", getProfessionSpeciesPropertyConduit());
        addNonSortableProperty("expectedSamplingStrategy", getExpectedSamplingStrategyPropertyConduit());
        addNonSortableProperty("fishingZoneFacade", getFishingZoneFacadePropertyConduit());
        addNonSortableProperty("fishingZoneSector", getFishingZoneSectorPropertyConduit());
        addNonSortableProperty("fishingZoneDivision", getFishingZoneDivisionPropertyConduit());
        addNonSortableProperty(Contact.PROPERTY_TERRESTRIAL_LOCATION, getTerrestrialLocationPropertyConduit());
        addNonSortableProperty(Contact.PROPERTY_BOAT, getBoatPropertyConduit());
        addNonSortableProperty("boatImmatriculation", getBoatImmatriculationPropertyConduit());
        addNonSortableProperty("boatDistrictCode", getBoatDistrictCodePropertyConduit());
        addNonSortableProperty("observationType", getObservationTypePropertyConduit());
        addNonSortableProperty(Contact.PROPERTY_STATE, null);
        addNonSortableProperty(Contact.PROPERTY_CONTACT_STATE_MOTIF, getContactStateMotifPropertyConduit());
        addNonSortableProperty(Contact.PROPERTY_LANDING_DATE, getLandingDatePropertyConduit());
        addNonSortableProperty(Contact.PROPERTY_OBSERVATION_BEGIN_DATE, null);
        addNonSortableProperty(Contact.PROPERTY_OBSERVATION_END_DATE, null);
        addNonSortableProperty(Contact.PROPERTY_NB_OBSERVANTS, null);
        addNonSortableProperty("mammals", null);
        addNonSortableProperty("actualSamplingStrategy", getActualSamplingStrategyPropertyConduit());
        addNonSortableProperty(Contact.PROPERTY_COMPLETE_SAMPLING, null);
        addNonSortableProperty(Contact.PROPERTY_COMMENT, null);
        addNonSortableProperty(Contact.PROPERTY_DATA_INPUT_DATE, null);
        addNonSortableProperty(Contact.PROPERTY_RESTITUTION, getRestitutionPropertyConduit());
        addNonSortableProperty("validation", null);
        addNonSortableProperty(Contact.PROPERTY_COMMENT_COORDINATOR, getCommentCoordinatorPropertyConduit());
        addNonSortableProperty(Contact.PROPERTY_COMMENT_ADMIN, getCommentAdminPropertyConduit());
        addNonSortableProperty("dataReliability", getDataReliabilityPropertyConduit());
        addNonSortableProperty(Contact.PROPERTY_OBS_DEB_CODE, null);
        addNonSortableProperty("actions", null);
    }

    /** remove properties from model depending on role, program, and fullView */
    protected void filterModel() {

        if (obsProgram == ObsProgram.OBSMER) {
            model.exclude(Contact.PROPERTY_TERRESTRIAL_LOCATION);
        } else {
            model.exclude(Contact.PROPERTY_CONTACT_STATE_MOTIF, "mammals",
                          "dataReliability", Contact.PROPERTY_RESTITUTION);
        }

        if (obsProgram != ObsProgram.OBSVENTE) {
            model.exclude("expectedSamplingStrategy",
                          "actualSamplingStrategy",
                          Contact.PROPERTY_COMPLETE_SAMPLING);
        }

        if (obsProgram == ObsProgram.OBSDEB) {
            model.exclude("professionLibelle", "professionSpecies","fishingZoneFacade",
                          "fishingZoneSector", "fishingZoneDivision", "professionCode",
                          "program");
        } else {
            model.exclude(Contact.PROPERTY_LANDING_DATE, "observationType",
                          Contact.PROPERTY_OBS_DEB_CODE);
        }

        if (userRole == UserRole.PROFESSIONAL) {
            // evo#344, le professionnel peut tout voir sauf 
            model.exclude(Contact.PROPERTY_COMMENT,
                          Contact.PROPERTY_COMMENT_COORDINATOR,
                          Contact.PROPERTY_COMMENT_ADMIN,
                          "dataReliability",
                          "actions");
        } else {
            if (userRole != UserRole.ADMIN || ! fullView) {
                model.exclude("observer",
                        WaoUser.PROPERTY_COMPANY,"program","professionCode", "professionLibelle",
                        "professionSpecies", "fishingZoneFacade", "fishingZoneSector",
                        "fishingZoneDivision", "boatImmatriculation", "boatDistrictCode");
            }

            if (userRole != UserRole.ADMIN && userRole != UserRole.COORDINATOR) {
                model.exclude("dataReliability");
            }
        }

    }

    public BeanModel<Contact> newModel(BeanModelSource beanModelSource, ComponentResources resources) {
        model = beanModelSource.createDisplayModel(Contact.class, resources.getMessages());

        // reset, so next adds will not throw an exception about already existing property
        model.include();

        initModel();

        filterModel();

        return model;
    }

    /** put in common some implementation of PropertyConduit interface. */
    protected static abstract class ContactReaderPropertyConduit implements PropertyConduit {

        public abstract String getValue(Contact contact);

        @Override
        public String get(Object instance) {
            // cast and delegate
            Contact contact = (Contact) instance;
            return getValue(contact);
        }

        /** all value returned are string */
        @Override
        public Class getPropertyType() {
            return String.class;
        }

        /** should never be called */
        @Override
        public void set(Object instance, Object value) {
            throw new UnsupportedOperationException();
        }

        /** should never be called */
        @Override
        public <T extends Annotation> T getAnnotation(Class<T> annotationClass) {
            throw new UnsupportedOperationException();
        }
    }

    protected PropertyConduit getUserPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return contact.getMainObserver().getFullName();
            }
        };
    }
    
    protected PropertyConduit getCompanyPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return contact.getMainObserver().getCompany().getName();
            }
        };
    }
    
    protected PropertyConduit getSampleRowPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return contact.getSampleRow().getCode();
            }
        };
    }

    protected PropertyConduit getBoatPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return contact.getBoat().getName();
            }
        };
    }

    protected PropertyConduit getProgramPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return contact.getSampleRow().getProgramName();
            }
        };
    }

    protected PropertyConduit getProfessionCodePropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return contact.getSampleRow().getProfessionDescription();
            }
        };
    }

    protected PropertyConduit getProfessionLibellePropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return contact.getSampleRow().getProfession().getLibelle();
            }
        };
    }

    protected PropertyConduit getProfessionSpeciesPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return contact.getSampleRow().getProfession().getSpecies();
            }
        };
    }

    protected PropertyConduit getFishingZoneFacadePropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return contact.getSampleRow().getFacade();
            }
        };
    }

    protected PropertyConduit getFishingZoneSectorPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return contact.getSampleRow().getSectors();
            }
        };
    }

    protected PropertyConduit getFishingZoneDivisionPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                String result = "";
                for (FishingZone zone : contact.getSampleRow().getFishingZone()) {
                    result += zone.getDistrictCode() + " ";
                }
                return result;
            }
        };
    }

    protected PropertyConduit getBoatImmatriculationPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return String.valueOf(contact.getBoat().getImmatriculation());
            }
        };
    }

    protected PropertyConduit getBoatDistrictCodePropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return contact.getBoat().getDistrictCode();
            }
        };
    }

    protected PropertyConduit getCommentCoordinatorPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return contact.getCommentCoordinator();
            }
        };
    }

    protected PropertyConduit getCommentAdminPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return contact.getCommentAdmin();
            }
        };
    }

    protected PropertyConduit getDataReliabilityPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return contact.getDataReliability().toString();
            }
        };
    }

    protected PropertyConduit getActualSamplingStrategyPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return contact.getSamplingStrategy().toString();
            }
        };
    }

    protected PropertyConduit getExpectedSamplingStrategyPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                return contact.getSampleRow().getSamplingStrategy().toString();
            }
        };
    }

    protected PropertyConduit getTerrestrialLocationPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                TerrestrialLocation terrestrialLocation = contact.getTerrestrialLocation();
                String value;
                if (terrestrialLocation == null) {
                    value = "";
                } else {
                    value = contact.getTerrestrialLocation().getDescription();
                }
                return value;
            }
        };
    }

    protected PropertyConduit getLandingDatePropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                String value = "";
                if (contact.getLandingDate() != null) {
                   value = WaoUtils.formatDateTime(contact.getLandingDate());
                }
                return value;
            }
        };
    }

    protected PropertyConduit getObservationTypePropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                String value = "";
                if (contact.getObservationType() != null) {
                   value = contact.getObservationType().toString();
                }
                return value;
            }
        };
    }

    protected PropertyConduit getContactStateMotifPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                String value = "";
                if (contact.getContactStateMotif() != null) {
                   value = contact.getContactStateMotif().toString();
                }
                return value;
            }
        };
    }

    protected PropertyConduit getRestitutionPropertyConduit() {
        return new ContactReaderPropertyConduit() {
            @Override
            public String getValue(Contact contact) {
                String value = "";
                if (contact.getRestitution() != null) {
                    value = WaoUtils.formatDate(contact.getRestitution());
                }
                return value;
            }
        };
    }

}
