/*
 * #%L
 * Wao :: Web Interface
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
/**
 * This class is used to manage comment window for contacts. Two possible cases :
 * - comment from current edition
 * - direct comment for coordinator or admin
 * The differences are in saveComment method. The openWindow method is used
 * in template using a JSONObject from Java that contains comment data.
 */
ContactComment = Class.create({
    initialize: function() {
        this.commentForm = $('commentForm');

        this.image = $('commentImage');
        this.invalidImage = $('commentInvalidImage');
        this.validImage = $('commentValidImage');

        this.window = commentWindow;
    },
    /**
     * Open the commentWindow ::
     * Use JSON to initialize data from template :
     * - data.edited : if a contact is currently in edition
     * - data.comment : comment value to initialize
     * - data.unfinished : if contact is in unfinished state
     * - data.id : id of the contact (used to save directly the comment)
     * - data.formId : id of the contactsForm in edition mode (edited = true)
     */
    openWindow: function(data) {
        this.editionMode = data.edited;
        this.oldUnfinished = data.unfinished;
        this.oldComment = data.comment;

        // Init comment from argument
        var comment = data.comment;
        if (data.formId) {
            // Init contactsForm
            this.contactForm = $(data.formId);
            comment = $F(this.contactForm.comment);
        }

        // Update commentForm
        this.commentForm.editComment.setValue(comment);
        this.commentForm.hiddenContactId.setValue(data.id);

        this.window.showCenter(true);
    },
    /**
     * Save the comment ::
     * If form is in edition, the comment will be push in existing input 'comment'
     * otherwise the commentForm will be submitted to save the comment
     */
    saveComment: function() {

        if (!this.editionMode) {
            // Execute commentForm to save directly the comment
            this.commentForm.submit();
        } else {
            // Refresh edition form with comment
            var newComment = $F(this.commentForm.editComment);
            this.refreshCommentImage(newComment);
            this.contactForm.comment.setValue(newComment);
        }

        this.window.close();
    },
    /**
     * Return true if current state has changed and becomes unfinished.
     * NewComment need to be defined.
     */
    isCommentValid: function(newComment) {

        var currentState = $F(this.contactForm.contactState);

        var isUnfinished = currentState == 'BOAT_UNAVAILABLE' ||
                                currentState == 'BOAT_REFUSED' ||
                                currentState == 'BOAT_DEFINITIVE_REFUSED';

        var becomesUnfinished = isUnfinished && !this.oldUnfinished;

        return newComment && (isUnfinished || becomesUnfinished);
    },
    /**
     * Refresh the comment image (only in edition mode) ::
     * Depends on contactUnfinished value to know if errors can occurs on comment
     * provides by return value of isCommentMandatory() function.
     * - param : commentChanged boolean, true if comment has changed
     */
    refreshCommentImage: function(newComment) {

        // VALID
        if (newComment && this.oldComment != newComment) {
            this.showImage(this.validImage);
            this.hideImage(this.invalidImage);
            this.hideImage(this.image);

        // INVALID
        } else if (!this.isCommentValid(newComment)) {
            this.showImage(this.invalidImage);
            this.hideImage(this.validImage);
            this.hideImage(this.image);

        // NO_CHANGE
        } else {
            this.showImage(this.image);
            this.hideImage(this.invalidImage);
            this.hideImage(this.validImage);
        }

    },
    showImage: function(image) {
        if (image.hasClassName('hidden')) {
            image.removeClassName('hidden');
        }
    },
    hideImage: function(image) {
        if (!image.hasClassName('hidden')) {
            image.addClassName('hidden');
        }
    }
});

ContactSendEmail = Class.create({
    initialize: function(message, contactState) {
        this.contactState = contactState;
        this.message = message;

        $('saveContact').observe('click', this.doConfirm.bind(this));
    },
    doConfirm: function(event) {

        var contactForm = event.element().form;

        if ($F(contactForm.contactState) == this.contactState && confirm(this.message)) {
            contactForm.hiddenSendEmail.setValue(true);
        }
    }
});
