/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id: WaoManager.java 591 2010-07-07 13:45:46Z fdesbois $
 * $HeadURL: svn+ssh://fdesbois@labs.libre-entreprise.org/svnroot/suiviobsmer/tags/wao-1.5/wao-ui/src/main/java/fr/ifremer/wao/ui/services/WaoManager.java $
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.wao.ui.services;

import fr.ifremer.wao.WaoBusinessException;
import fr.ifremer.wao.WaoContext;
import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.WaoProperty;
import fr.ifremer.wao.bean.ContactState;
import fr.ifremer.wao.bean.SamplingFilter;
import fr.ifremer.wao.entity.Boat;
import fr.ifremer.wao.entity.Contact;
import fr.ifremer.wao.entity.FishingZone;
import fr.ifremer.wao.entity.Profession;
import fr.ifremer.wao.entity.SampleRow;
import fr.ifremer.wao.service.ServiceUser;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.services.RegistryShutdownListener;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.util.ApplicationConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * WaoManager
 *
 * Created: 24 nov. 2009
 *
 * @author fdesbois <fdesbois@codelutin.com>
 */
public class WaoManager implements Runnable, RegistryShutdownListener {

    private static final Logger log =
            LoggerFactory.getLogger(WaoManager.class);

    private WaoContext context;

    private ServiceUser serviceUser;

    /**
     * Constructor of WaoManager. It needs the WaoContext to start and stop.
     *
     * @param context used to manage application lifecycle
     */
    public WaoManager(WaoContext context) {
        this.context = context;
    }

    /**
     * ServiceUser is needed to start the application.
     *
     * @param serviceUser
     * @see fr.ifremer.wao.WaoContext#start(fr.ifremer.wao.service.ServiceUser) 
     */
    void setServiceUser(ServiceUser serviceUser) {
        this.serviceUser = serviceUser;
    }

    /**
     * Called to start the application.
     */
    @Override
    public void run() {
        context.start(serviceUser);
    }

    /**
     * Called to stop the application
     */
    @Override
    public void registryDidShutdown() {
        context.stop();
    }

    /**
     * Context to inject in new service instances.
     *
     * @return the WaoContext of the application
     */
    WaoContext getContext() {
        return context;
    }

    public boolean isActivityCalendarImportRun() {
        return context.isActivityCalendarImportRun();
    }

    public String getProperty(WaoProperty property) {
        return context.getProperty(property);
    }

    public ApplicationConfig getConfiguration() {
        return context.getConfiguration();
    }

    public Date getCurrentDate() {
        return context.getCurrentDate();
    }

    public boolean isDevEnvironment() {
        return log.isDebugEnabled();
    }

    public String[] getErrorMessages(WaoException eee, Messages messages,
            Logger logger) {

        String[] results = null;
        String cause = null;
        // Only show cause on dev environment
        if (eee.getCause() != null && isDevEnvironment()) {
            cause = eee.getCause().getClass().getSimpleName() +
                        " : " + eee.getCause().getMessage();
            results = new String[2];
            results[1] = cause;
        } else {
            results = new String[1];
        }


        if (logger.isDebugEnabled()) {
            logger.debug("ERROR from business : " + eee.getMessage() +
                    " _ args = " + eee.getArgs());
            if (cause != null) {
                logger.debug("ERROR caused by : " + cause);
            }
        }

        if (eee.getArgs() != null) {
            results[0] = messages.format(eee.getMessage(), eee.getArgs());
        } else {
            results[0] = messages.get(eee.getMessage());
        }
        return results;
    }

    public String getErrorMessage(WaoBusinessException eee, Messages messages,
            Logger logger) {
        if (logger.isDebugEnabled()) {
            logger.debug("ERROR from business : " + eee.getMessage());
        }
        return eee.getMessage();
    }

    public String getTooltipText(String input) {
        if (input != null) {
            return input.replaceAll("\n", "<br />").replaceAll("\r", "");
        }
        return "";
    }


    public String getTooltipSampleRow(SampleRow row) {
        StringBuilder result = new StringBuilder();
        DateFormat dateFormat = new SimpleDateFormat("MM/yyyy");
        Profession profession = row.getProfession();
        result.append("<strong>Métier : </strong>").
               append(profession.getCodeDCF5());
        if (!StringUtils.isEmpty(profession.getMeshSize())) {
            result.append(' ').append(profession.getMeshSize());
        }
        if (!StringUtils.isEmpty(profession.getSize())) {
            result.append(' ').append(profession.getSize());
        }
        if (!StringUtils.isEmpty(profession.getOther())) {
            result.append(' ').append(profession.getOther());
        }
        result.append(" - ").append(profession.getLibelle());
        if (!StringUtils.isEmpty(profession.getSpecies())) {
            result.append("\n<strong>Espèces cibles : </strong>").
                   append(profession.getSpecies());
        }
        result.append("\n<strong>Programme : </strong>").
               append(row.getProgramName()).
               append("\n<strong>Période : </strong> du ").
               append(dateFormat.format(row.getPeriodBegin())).append(" au ").
               append(dateFormat.format(row.getPeriodEnd()));

        for (FishingZone zone : row.getFishingZone()) {
            result.append("\n<strong>Zone de pêche : </strong>").
                   append(zone.getCode());
        }

        return getTooltipText(result.toString());
    }

    public String getTooltipBoat(Boat boat) {
        StringBuilder result = new StringBuilder("<strong>").
                append(boat.getImmatriculation()).append("</strong>").
                append("\n<strong>Quartier : </strong>").
                append(boat.getDistrictCode()).
                append("\n<strong>Longueur : </strong>").
                append(boat.getBoatLength()).append(" m").
                append("\n<strong>Année de construction : </strong>").
                append(boat.getBuildYear());
        return getTooltipText(result.toString());
    }

    public String getDataSamplingTitle(SamplingFilter filter) {
        StringBuilder result = new StringBuilder("Données des marées");
        String separator = " : ";
        if (filter.getCompany() != null) {
            //result.append(!filtered ? " : " : ", ";
            result.append(separator).
                   append("\nSociété ").append(filter.getCompany().getName());
            separator = ", ";
        }
        if (StringUtils.isNotEmpty(filter.getProgramName())) {
            result.append(separator).
                   append("\nProgramme ").append(filter.getProgramName());
            separator = ", ";
        }
        if (filter.getSampleRow() != null) {
            Profession profession = filter.getSampleRow().getProfession();
            result.append(separator).
                   append("\nMétier ").append(profession.getCode());
            if (StringUtils.isNotEmpty(profession.getLibelle())) {
                result.append(" - ").append(profession.getLibelle());
            }
            if (StringUtils.isNotEmpty(profession.getSpecies())) {
                result.append("\nEspèces (").
                       append(profession.getSpecies()).append(")");
            }
            result.append("\nZones de pêche (");
            for (FishingZone zone : filter.getSampleRow().getFishingZone()) {
                result.append(zone.getDistrictCode()).append(' ');
            }
            result.append(")");
            separator = ", ";
        } else {
            if (filter.getFacadeName() != null) {
                result.append(separator).
                       append("\nFaçade ").append(filter.getFacadeName());
                separator = ", ";
            }
            if (filter.getSectorName() != null) {
                result.append(separator).
                       append("\nZone ").append(filter.getSectorName());
            }
        }
        return result.toString();
    }

    public String getContactStyle(Contact contact, boolean admin) {
        String result = "";
        boolean accepted = BooleanUtils.isTrue(contact.getValidationProgram()) ||
                (!admin && BooleanUtils.isTrue(contact.getValidationCompany()));

        ContactState state = contact.getContactState();
        if (BooleanUtils.isFalse(contact.getValidationProgram()) ||
                (!admin && BooleanUtils.isFalse(contact.getValidationCompany()))) {
            result = "refused";
        } else if (accepted && state.equals(ContactState.BOARDING_DONE)) {
            result = "accepted";
        } else if (accepted && state.isUnfinishedState()) {
            result = "unfinished";
        } else if (state.isFinalState()) {
            result = "even";
        } else {
            result = "odd";
        }
        return result;
    }

    

}
