/*
 * *##% 
 * Wao :: Web Interface
 * Copyright (C) 2009 - 2010 Ifremer
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package fr.ifremer.wao.ui.pages;

import fr.ifremer.wao.WaoBusinessException;
import fr.ifremer.wao.WaoBusinessException.Type;
import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.entity.Company;
import fr.ifremer.wao.entity.CompanyImpl;
import fr.ifremer.wao.entity.WaoUser;
import fr.ifremer.wao.service.ServiceReferential;
import fr.ifremer.wao.service.ServiceUser;
import fr.ifremer.wao.ui.base.GenericSelectModel;
import fr.ifremer.wao.ui.base.WaoPage;
import fr.ifremer.wao.ui.components.Layout;
import fr.ifremer.wao.ui.data.BusinessUtils;
import java.util.List;
import org.apache.commons.lang.StringUtils;
import org.apache.tapestry5.EventContext;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.apache.tapestry5.ioc.services.TypeCoercer;
import org.apache.tapestry5.upload.services.UploadedFile;
import org.slf4j.Logger;

/**
 * Administration
 *
 * Created: 9 nov. 2009
 *
 * @author fdesbois
 * @version $Revision: 425 $
 *
 * Mise a jour: $Date: 2010-04-06 12:22:43 +0200 (mar., 06 avril 2010) $
 * par : $Author: fdesbois $
 */
@IncludeStylesheet("context:css/administration.css")
public class Administration implements WaoPage {

    /**
     * The access to the page is only for admin with no readOnly right.
     *
     * @return true if the Administration page can be displayed in Layout
     */
    @Override
    public boolean canDisplay() {
        return currentUser.isAdmin() && !currentUser.getReadOnly();
    }

    @Inject
    private Logger log;

    @InjectComponent
    private Layout layout;

    @SessionState
    private WaoUser currentUser;

    @Inject
    private ServiceUser serviceUser;

    @Inject
    private ServiceReferential serviceReferential;

    @Inject
    private TypeCoercer typeCoercer;

    @Property
    private String companyId;

    @Property
    private String userId;

    @Persist
    private List<Company> companies;

    @Inject
    private PropertyAccess propertyAccess;

    private GenericSelectModel<Company> companiesSelectModel;

    private Company company;

    private GenericSelectModel<WaoUser> usersSelectModel;

    /**
     * User is persistant cause of autobuild problem ?!?
     */
    @Persist
    private WaoUser user;

    @Property
    private boolean generatePassword;

    @Property
    private String password;

    private boolean addNewCompanySelected;
    private boolean addNewUserSelected;
    private boolean deleteUserSelected;

    void setupRender() throws WaoException {
        companies = null;
        getCompanies();
        user = null;
    }

    void onActivate(EventContext ec) {
        if (ec.getCount() > 0) {
            companyId = ec.get(String.class, 0);
            if (ec.getCount() > 1) {
                userId = ec.get(String.class, 1);
            }
        }
    }

    Object[] onPassivate() {
        return new String[] {companyId, userId};
    }

   /**************************** IMPORT (ADMIN) *******************************/

    /**
     * CSV File which contains fishing zones
     */
    @Property
    private UploadedFile fishingZoneCsvFile;

    @Inject
    private Messages messages;

    @Log
    void onSuccessFromImportFishingZones() {
        try {
            int result = serviceReferential.importFishingZoneCsv(
                    fishingZoneCsvFile.getStream());
            layout.addInfo(result + " zones de pêche importées");
        } catch (WaoException eee) {
            String[] errors =
                    BusinessUtils.getErrorMessages(eee, messages, log);
            layout.addError(errors);
        }
    }

    /**************************** FORMS ****************************************/

    public List<Company> getCompanies() throws WaoException {
        if (companies == null) {
            companies = serviceUser.getCompanies(false);
        }
        return companies;
    }

    public GenericSelectModel<Company> getCompaniesSelectModel() throws WaoException {
        if (companiesSelectModel == null) {
            companiesSelectModel = new GenericSelectModel<Company>(getCompanies(), Company.class, Company.NAME, "id", propertyAccess);
        }
        return companiesSelectModel;
    }

    public Company getCompany() throws WaoException {
        if (company == null) {
            if (companyId != null) {
                company = getCompaniesSelectModel().findObject(companyId);
            } else {
                company = new CompanyImpl();
            }
        }
        return company;
    }

    public boolean isUsersAvailable() throws WaoException {
        return !StringUtils.isEmpty(getCompany().getId()) && getCompany().getActive();
    }

    void onSelectedFromAddNewCompany() {
        addNewCompanySelected = true;
    }

    void onSuccessFromActionsForm() {
        if (addNewCompanySelected) {
            companyId = null;            
        }
        userId = null;
        user = null;
    }

    void onSuccessFromCompany() throws WaoException {
        serviceUser.createUpdateCompany(company);
        companyId = company.getId();
    }

    @Log
    public GenericSelectModel<WaoUser> getUsersSelectModel() throws WaoException {
        if (usersSelectModel == null) {
            List<WaoUser> users = serviceUser.getUsersByCompany(getCompany());
            if (log.isDebugEnabled()) {
                log.debug("Nb users : " + users.size());
            }
            usersSelectModel = new GenericSelectModel<WaoUser>(users, WaoUser.class, "fullName", "id", propertyAccess);
        }
        return usersSelectModel;
    }

    @Log
    public WaoUser getUser() throws WaoException {
        if (user == null) {
            if (userId != null) {
                if (log.isDebugEnabled()) {
                    log.debug("User exist in selectModel : " + userId);
                }
                user = getUsersSelectModel().findObject(userId);
            } else {
                user = serviceUser.getNewUser(getCompany());
//                user = new WaoUserImpl();
//                if (log.isDebugEnabled()) {
//                    log.debug("Set company for new user : " + getCompany().getName());
//                }
//                user.setCompany(getCompany());
//                 Default role set to OBSERVER
//                user.setUserRole(UserRole.OBSERVER);
                generatePassword = true;
            }
        }
        return user;
    }

    void onSelectedFromAddNewUser() {
        addNewUserSelected = true;
    }

    void onSelectedFromDeleteUser() {
        deleteUserSelected = true;
    }

    void onSuccessFromUserActionsForm() throws WaoException {
        if (addNewUserSelected) {
            userId = null;
        } else if (deleteUserSelected && userId != null) {
            user = getUsersSelectModel().findObject(userId);
            try {
                serviceUser.deleteUser(user);
                user = null;
                userId = null;
                layout.addInfo("Utilisateur supprimé avec succès !");
            } catch (WaoBusinessException eee) {
                layout.addError(eee.getMessage());
            }
        }
    }

    @Log
    void onSuccessFromUser() throws WaoException {
        if (log.isDebugEnabled()) {
            log.debug("User : " + user);
        }

        if (!StringUtils.isEmpty(password)) {
            user.setPassword(password);
            user.setPasswordChanged(true);
        }
        try {
            serviceUser.createUpdateUser(user, generatePassword);
            userId = user.getId();
        } catch (WaoBusinessException eee) {
            if (eee.getType().equals(Type.SMTP_NOT_FOUND)) {
                layout.addInfo(eee.getMessage());
                userId = user.getId();
            } else {
                layout.addError(eee.getMessage());
            }
        }
    }
        
}
