package fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.create;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import org.nuiton.jaxx.application.swing.util.Cancelable;
import fr.ifremer.tutti.type.WeightUnit;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModel;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModelEntry;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.CaracteristicQualitativeValue;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.split.SplitBenthosBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.split.SplitBenthosBatchTableModel;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.species.SelectSpeciesUI;
import fr.ifremer.tutti.ui.swing.util.species.SelectSpeciesUIModel;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIHandler;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;

import javax.swing.JComponent;
import java.awt.Dimension;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;

/**
 * Handler of {@link CreateBenthosBatchUI}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.3
 */
public class CreateBenthosBatchUIHandler extends AbstractTuttiTableUIHandler<SplitBenthosBatchRowModel, CreateBenthosBatchUIModel, CreateBenthosBatchUI> implements Cancelable {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(CreateBenthosBatchUIHandler.class);

    /**
     * Qualitative value for the Vrac.
     *
     * @since 2.5
     */
    protected CaracteristicQualitativeValue sortedValue;

    /**
     * Sample categories model.
     *
     * @since 2.4
     */
    protected SampleCategoryModel sampleCategoryModel;

    /**
     * Weight unit.
     *
     * @since 2.6
     */
    protected WeightUnit weightUnit;

    public CreateBenthosBatchUIHandler() {
        super(SplitBenthosBatchRowModel.PROPERTY_SELECTED,
              SplitBenthosBatchRowModel.PROPERTY_CATEGORY_VALUE,
              SplitBenthosBatchRowModel.PROPERTY_WEIGHT);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public SplitBenthosBatchTableModel getTableModel() {
        return (SplitBenthosBatchTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(SplitBenthosBatchRowModel row) {
        return row.isSelected();
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<SplitBenthosBatchRowModel> rowMonitor,
                                             SplitBenthosBatchRowModel row) {
        if (rowMonitor.wasModified()) {

            if (row.isValid()) {
                if (log.isInfoEnabled()) {
                    log.info("Change row that was modified and valid");
                }
            }

            rowMonitor.clearModified();
        }
    }

    @Override
    protected void onAfterSelectedRowChanged(int oldRowIndex,
                                             SplitBenthosBatchRowModel oldRow,
                                             int newRowIndex,
                                             SplitBenthosBatchRowModel newRow) {
        super.onAfterSelectedRowChanged(oldRowIndex, oldRow, newRowIndex, newRow);
        if (newRow != null) {

            // Recompute the valid state of the row
            recomputeRowValidState(newRow);

            // Need to recompute the sample weight
            computeSampleWeight();
        }
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 SplitBenthosBatchRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        // Recompute the valid state of the row
        recomputeRowValidState(row);

        // Need to recompute the sample weight
        computeSampleWeight();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public void beforeInit(CreateBenthosBatchUI ui) {
        super.beforeInit(ui);

        this.weightUnit = getConfig().getBenthosWeightUnit();
        this.sampleCategoryModel = getDataContext().getSampleCategoryModel();

        SampleCategoryModelEntry caracteristic = sampleCategoryModel.getCategoryById(sampleCategoryModel.getFirstCategoryId());

        Integer vracId = getPersistenceService().getEnumerationFile().QUALITATIVE_VRAC_ID;

        CaracteristicQualitativeValue vracValue = null;
        for (CaracteristicQualitativeValue caracteristicQualitativeValue : caracteristic.getCaracteristic().getQualitativeValue()) {

            if (vracId.equals(caracteristicQualitativeValue.getIdAsInt())) {
                vracValue = caracteristicQualitativeValue;
                break;
            }
        }
        Preconditions.checkNotNull(vracValue, "Could not found vrac qualitative value");
        this.sortedValue = vracValue;

        CreateBenthosBatchUIModel model =
                new CreateBenthosBatchUIModel(sampleCategoryModel);
        this.ui.setContextValue(model);
        listModelIsModify(model);
    }

    @Override
    public void afterInit(CreateBenthosBatchUI ui) {

        initUI(this.ui);

        initBeanFilterableComboBox(this.ui.getSpeciesComboBox(),
                                   Lists.<Species>newArrayList(),
                                   null,
                                   DecoratorService.FROM_PROTOCOL);

        List<SampleCategoryModelEntry> categories = Lists.newArrayList();

        // add all categories
        categories.addAll(sampleCategoryModel.getCategory());

        // remove the first one (V/HV)
        categories.remove(0);

        initBeanFilterableComboBox(this.ui.getCategoryComboBox(),
                                   Lists.newArrayList(categories),
                                   null);

        Caracteristic caracteristic =
                getPersistenceService().getSortedUnsortedCaracteristic();

        initBeanFilterableComboBox(this.ui.getSampleCategoryComboBox(),
                                   Lists.newArrayList(caracteristic.getQualitativeValue()),
                                   null);

        CreateBenthosBatchUIModel model = getModel();

        model.addPropertyChangeListener(CreateBenthosBatchUIModel.PROPERTY_SPECIES, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                CreateBenthosBatchUIModel source =
                        (CreateBenthosBatchUIModel) evt.getSource();
                Species newValue = (Species) evt.getNewValue();

                if (log.isDebugEnabled()) {
                    log.debug("New Selected species " + (newValue == null ? null : decorate(newValue)));
                }

                if (newValue == null || source.getSpeciesUsed() == null) {

                    // reset V/HV category
                    source.setSampleCategory(null);

                } else {

                    // look for best value for V/HV category : if sorted is available then use it

                    List<CaracteristicQualitativeValue> qualitativeValues =
                            CreateBenthosBatchUIHandler.this.ui.getSampleCategoryComboBox().getData();

                    CaracteristicQualitativeValue newCategory = null;

                    for (CaracteristicQualitativeValue qualitativeValue : qualitativeValues) {
                        if (source.isSpeciesAndCategoryAvailable(newValue, qualitativeValue)) {
                            newCategory = qualitativeValue;

                            if (newCategory.equals(sortedValue)) {
                                break;
                            }
                        }
                    }
                    source.setSampleCategory(newCategory);

                    // reset selected category
                    if (log.isDebugEnabled()) {
                        log.debug("Remove selected category before changing the categories...");
                    }
                    source.setSelectedCategory(null);

                    // compute the selected sample category

                    SampleCategoryModelEntry selectedCategory = getDataContext().getBestFirstBenthosSampleCategory(
                            getUI().getCategoryComboBox().getData(),
                            newValue
                    );

                    if (log.isDebugEnabled()) {
                        log.debug("Selected category : " + selectedCategory);
                    }

                    // set new selected category
                    source.setSelectedCategory(selectedCategory);
                }
            }
        });

        // when selected category changed, regenerate the table model + add inside some default rows
        model.addPropertyChangeListener(CreateBenthosBatchUIModel.PROPERTY_SELECTED_CATEGORY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                // can change the selected category
                CreateBenthosBatchUIModel source =
                        (CreateBenthosBatchUIModel) evt.getSource();

                // when selected category change, sample total weight is reset
                source.setSampleWeight(null);

                SampleCategoryModelEntry newValue =
                        (SampleCategoryModelEntry) evt.getNewValue();
                generateTableModel(newValue);
            }
        });

        generateTableModel(null);

        initTable(getTable());

        listenValidatorValid(this.ui.getValidator(), model);
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getSpeciesComboBox();
    }

    @Override
    public void onCloseUI() {

        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }

        // evict model from validator
        ui.getValidator().setBean(null);

        // when canceling always invalid model
        getModel().setValid(false);

        EditCatchesUI parent = getParentContainer(EditCatchesUI.class);
        parent.getHandler().setBenthosSelectedCard(EditCatchesUIHandler.MAIN_CARD);
    }

    @Override
    public SwingValidator<CreateBenthosBatchUIModel> getValidator() {
        return ui.getValidator();
    }

    //------------------------------------------------------------------------//
    //-- Cancelable methods                                                 --//
    //------------------------------------------------------------------------//

    @Override
    public void cancel() {
        if (log.isDebugEnabled()) {
            log.debug("Cancel UI " + ui);
        }
        closeUI(ui);
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void openUI(BenthosBatchUIModel batchModel) {

        CreateBenthosBatchUIModel model = getModel();

        // connect model to validator
        ui.getValidator().setBean(model);

        model.setSpecies(null);
        model.setSampleCategory(null);
        model.setBatchWeight(null);
        model.setBatchCount(null);

        List<Species> speciesToUse = Lists.newArrayList();

        Multimap<CaracteristicQualitativeValue, Species> speciesUsed =
                model.getSpeciesUsed();
        speciesUsed.clear();

        if (batchModel != null) {

            speciesUsed.putAll(batchModel.getSpeciesUsed());

            // compute which species can still be used

            List<Species> allSpecies =
                    getDataContext().getReferentBenthosWithSurveyCode();

            speciesToUse.addAll(allSpecies);
        }

        model.setAvailableSpecies(speciesToUse);
    }

    public void saveAndContinue() {

        if (log.isDebugEnabled()) {
            log.debug("Save and Continue UI " + ui);
        }

        EditCatchesUI parent = getParentContainer(EditCatchesUI.class);
        BenthosBatchUI benthosTabContent = parent.getBenthosTabContent();
        benthosTabContent.getHandler().addBatch(getModel());

        // re-open this screen
        openUI(benthosTabContent.getModel());
    }

    public void saveAndClose() {

        if (log.isDebugEnabled()) {
            log.debug("Save and Close UI " + ui);
        }

        EditCatchesUI parent = getParentContainer(EditCatchesUI.class);
        parent.getBenthosTabContent().getHandler().addBatch(getModel());

        // close dialog
        closeUI(ui);
    }

    public Species openAddSpeciesDialog(String title, List<Species> species) {
        SelectSpeciesUI dialogContent = new SelectSpeciesUI(ui);
        SelectSpeciesUIModel model = dialogContent.getModel();
        model.setSelectedSpecies(null);
        model.setSpecies(species);

        openDialog(dialogContent, title, new Dimension(400, 130));

        Species result = model.getSelectedSpecies();
        return result;
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void computeSampleWeight() {

        Float result = null;
        List<SplitBenthosBatchRowModel> rows = getTableModel().getRows();
        for (SplitBenthosBatchRowModel row : rows) {
            if (row.isSelected()) {
                Float weight = row.getWeight();
                if (weight != null) {
                    if (result == null) {
                        result = 0f;
                    }
                    result += weight;
                }
            }
        }
        getModel().setSampleWeight(result);
    }

    protected void generateTableModel(SampleCategoryModelEntry category) {

        if (log.isDebugEnabled()) {
            log.debug("Generate table model for category " + category);
        }
        // when generate a new table model, then reset previous rows from model
        getModel().setRows(null);

        Caracteristic data = null;

        JXTable table = getTable();

        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        { // Selection

            addBooleanColumnToModel(columnModel,
                                    SplitBenthosBatchTableModel.SELECTED,
                                    table);
        }

        boolean editableCategoryValue = false;
        if (category != null) {

            if (!category.getCaracteristic().isQualitativeValueEmpty()) {

                // qualitative category
                data = category.getCaracteristic();
            } else {
                editableCategoryValue = true;
                addFloatColumnToModel(columnModel,
                                      SplitBenthosBatchTableModel.EDITABLE_CATEGORY_VALUE,
                                      TuttiUI.DECIMAL1_PATTERN,
                                      table);
            }

            if (data != null) {

                if (log.isDebugEnabled()) {
                    log.debug("Got " + data.sizeQualitativeValue() + " qualitative data to add");
                }
                addColumnToModel(columnModel,
                                 null,
                                 newTableCellRender(CaracteristicQualitativeValue.class),
                                 SplitBenthosBatchTableModel.READ_ONLY_CATEGORY_VALUE);
            }
            { // Weight

                addFloatColumnToModel(columnModel,
                                      SplitBenthosBatchTableModel.WEIGHT,
                                      weightUnit,
                                      table);
            }
        }

        // create table model
        SplitBenthosBatchTableModel tableModel =
                new SplitBenthosBatchTableModel(columnModel,
                                                getModel(),
                                                editableCategoryValue,
                                                false);

        // remove all listener on tables we could add before
        uninstallTableSaveOnRowChangedSelectionListener();
        uninstallTableKeyListener();

        if (log.isDebugEnabled()) {
            log.debug("Install new table model " + tableModel);
        }
        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        // install table listeners
        installTableSaveOnRowChangedSelectionListener();
        installTableKeyListener(columnModel, table);

        // fill datas

        List<SplitBenthosBatchRowModel> rows = Lists.newArrayList();

        if (data != null) {

            // add a row for each qualitive value
            for (CaracteristicQualitativeValue qualitativeValue : data.getQualitativeValue()) {
                if (log.isDebugEnabled()) {
                    log.debug("Add QV: " + qualitativeValue);
                }
                SplitBenthosBatchRowModel newRow = tableModel.createNewRow();
                newRow.setCategoryValue(qualitativeValue);
                rows.add(newRow);
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Will add " + rows.size() + " rows in table model " +
                      "(can add a first empty row? " + editableCategoryValue + ").");
        }

        getModel().setRows(rows);
    }
}
