package fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency;

/*
 * #%L
 * Tutti :: UI
 * $Id: BenthosFrequencyTableModel.java 1566 2014-02-04 08:31:02Z tchemit $
 * $HeadURL: https://svn.codelutin.com/tutti/tags/tutti-3.3.5/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/benthos/frequency/BenthosFrequencyTableModel.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Maps;
import org.nuiton.jaxx.application.swing.table.AbstractApplicationTableModel;
import fr.ifremer.tutti.type.WeightUnit;
import org.nuiton.jaxx.application.swing.table.ColumnIdentifier;
import org.jdesktop.swingx.table.TableColumnModelExt;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;
import java.util.Map;

import static org.nuiton.i18n.I18n.n;

/**
 * Model of the species frequency table.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class BenthosFrequencyTableModel extends AbstractApplicationTableModel<BenthosFrequencyRowModel> {

    private static final long serialVersionUID = 1L;

    public static final ColumnIdentifier<BenthosFrequencyRowModel> LENGTH_STEP = ColumnIdentifier.newId(
            BenthosFrequencyRowModel.PROPERTY_LENGTH_STEP,
            n("tutti.editSpeciesFrequencies.table.header.lengthStep"),
            n("tutti.editSpeciesFrequencies.table.header.lengthStep"));

    public static final ColumnIdentifier<BenthosFrequencyRowModel> NUMBER = ColumnIdentifier.newId(
            BenthosFrequencyRowModel.PROPERTY_NUMBER,
            n("tutti.editSpeciesFrequencies.table.header.number"),
            n("tutti.editSpeciesFrequencies.table.header.number"));

    public static final ColumnIdentifier<BenthosFrequencyRowModel> WEIGHT = ColumnIdentifier.newId(
            BenthosFrequencyRowModel.PROPERTY_WEIGHT,
            n("tutti.editSpeciesFrequencies.table.header.weight"),
            n("tutti.editSpeciesFrequencies.table.header.weight"));

    private final BenthosFrequencyUIModel uiModel;

    private final Map<Float, BenthosFrequencyRowModel> rowCache;

    /**
     * Weight unit.
     *
     * @since 2.5
     */
    protected final WeightUnit weightUnit;

    public BenthosFrequencyTableModel(WeightUnit weightUnit,
                                      TableColumnModelExt columnModel,
                                      BenthosFrequencyUIModel uiModel) {
        super(columnModel, true, true);
        this.weightUnit = weightUnit;
        this.uiModel = uiModel;
        this.rowCache = Maps.newTreeMap();
        setNoneEditableCols();
    }

    @Override
    public BenthosFrequencyRowModel createNewRow() {
        Float defaultStep = null;

        int rowCount = getRowCount();
        if (rowCount > 0) {

            BenthosFrequencyRowModel rowModel = getEntry(rowCount - 1);
            Float lengthStep = rowModel.getLengthStep();
            if (lengthStep != null) {
                defaultStep = uiModel.getLengthStep(
                        lengthStep + uiModel.getStep());
            }
        }
        BenthosFrequencyRowModel result =
                new BenthosFrequencyRowModel(weightUnit);
        result.addPropertyChangeListener(BenthosFrequencyRowModel.PROPERTY_LENGTH_STEP,
                                         new PropertyChangeListener() {
                                             @Override
                                             public void propertyChange(PropertyChangeEvent evt) {
                                                 Float oldValue = (Float) evt.getOldValue();
                                                 if (oldValue != null) {
                                                     rowCache.remove(oldValue);
                                                 }
                                                 BenthosFrequencyRowModel row = (BenthosFrequencyRowModel) evt.getSource();
                                                 Float lengthStep = row.getLengthStep();
                                                 if (lengthStep != null) {
                                                     rowCache.put(lengthStep, row);
                                                 }
                                             }
                                         });
        result.setLengthStepCaracteristic(uiModel.getLengthStepCaracteristic());
        result.setLengthStep(defaultStep);
        result.setValid(defaultStep != null);
        return result;
    }

    @Override
    public void setValueAt(Object aValue,
                           int rowIndex,
                           int columnIndex,
                           ColumnIdentifier<BenthosFrequencyRowModel> propertyName,
                           BenthosFrequencyRowModel entry) {
        super.setValueAt(aValue, rowIndex, columnIndex, propertyName, entry);
        // TODO Rebuild the computedWeight if possible...
    }

    public Map<Float, BenthosFrequencyRowModel> getRowCache() {
        return rowCache;
    }

    @Override
    protected void onRowsChanged(List<BenthosFrequencyRowModel> data) {

        // rebuild row cache
        rowCache.clear();

        for (BenthosFrequencyRowModel row : data) {
            Float lengthStep = row.getLengthStep();
            if (lengthStep != null) {
                rowCache.put(lengthStep, row);
            }
        }
    }
}