package fr.ifremer.tutti.ui.swing.action;

/*
 * #%L
 * Tutti :: UI
 * $Id: UpdateApplicationAction.java 1566 2014-02-04 08:31:02Z tchemit $
 * $HeadURL: https://forge.codelutin.com/svn/tutti/tags/tutti-3.1.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/action/UpdateApplicationAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.TuttiConfiguration;
import fr.ifremer.tutti.persistence.ProgressionModel;
import fr.ifremer.tutti.ui.swing.RunTutti;
import fr.ifremer.tutti.ui.swing.TuttiApplicationUpdaterCallBack;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.content.MainUI;
import fr.ifremer.tutti.ui.swing.content.MainUIHandler;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.updater.ApplicationUpdater;

import java.io.File;

import static org.nuiton.i18n.I18n.t;

/**
 * To update jre / i18n or tutti using the {@link ApplicationUpdater} mecanism.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0
 */
public class UpdateApplicationAction extends AbstractTuttiAction<TuttiUIContext, MainUI, MainUIHandler> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(UpdateApplicationAction.class);

    public UpdateApplicationAction(MainUIHandler handler) {
        super(handler, true);
        setActionDescription(t("tutti.main.action.updateApplication.tip"));
        types = TuttiApplicationUpdaterCallBack.UpdateType.values();
    }

    protected TuttiApplicationUpdaterCallBack.UpdateType[] types;

    protected boolean reload;

    @Override
    public boolean prepareAction() throws Exception {
        boolean doAction = super.prepareAction();

        if (doAction) {
            // check application url is reachable
            TuttiUIContext context = getContext();
            doAction = context.checkUpdateApplicationReachable(true);
        }
        return doAction;
    }

    @Override
    public void releaseAction() {
        super.releaseAction();
        types = TuttiApplicationUpdaterCallBack.UpdateType.values();
    }

    @Override
    public void doAction() throws Exception {

        reload = false;

        TuttiUIContext context = getContext();
        TuttiConfiguration config = getConfig();

        File current = config.getBasedir();
        if (current == null || !current.exists()) {

            // can not update application
            if (log.isWarnEnabled()) {
                log.warn("No application base directory defined, skip updates.");
            }
        } else {

            String url = config.getUpdateApplicationUrl();
            File dest = new File(config.getBasedir(), "NEW");

            if (log.isInfoEnabled()) {
                log.info(String.format("Try to update jre, i18N, help or tutti (current application location: %s), using update url: %s", current, url));
            }

            ProgressionModel progressionModel = new ProgressionModel();
            context.getActionUI().getModel().setProgressionModel(progressionModel);
            progressionModel.setMessage(t("tutti.updateApplication.checkUpdates"));

            TuttiApplicationUpdaterCallBack callback =
                    new TuttiApplicationUpdaterCallBack(this, progressionModel);

            callback.setTypes(types);

            ApplicationUpdater up = new ApplicationUpdater();
            up.update(url,
                      current,
                      dest,
                      false,
                      callback,
                      progressionModel);

            if (callback.isApplicationUpdated()) {

                reload = true;

            } else {

                sendMessage(t("tutti.updateApplication.noUpdate"));
            }
        }
    }

    public void setTypes(TuttiApplicationUpdaterCallBack.UpdateType... types) {
        this.types = types;
    }

    @Override
    public void postSuccessAction() {
        super.postSuccessAction();

        if (reload) {
            // wait 1 second to be sure action ui is up
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                if (log.isWarnEnabled()) {
                    log.warn("Could not wait 1 second...", e);
                }
            }

            // tell user restart will be done

            getHandler().showSuccessMessage(t("tutti.updateApplication.title.success"),
                                            t("tutti.updateApplication.message.success"));

            CloseApplicationAction action = getContext().getActionFactory().createLogicAction(
                    getHandler(), CloseApplicationAction.class);
            action.setExitCode(RunTutti.UPATE_EXIT_CODE);
            getActionEngine().runAction(action);
        }
    }

    public boolean isReload() {
        return reload;
    }
}