
package fr.ifremer.tutti.ui.swing.content.operation.catches;

/*
 * #%L
 * Tutti :: UI
 * $Id: ComputeWeightsAction.java 666 2013-03-23 18:20:58Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/ComputeWeightsAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.TuttiBusinessException;
import fr.ifremer.tutti.persistence.entities.data.SortedUnsortedEnum;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency.BenthosFrequencyRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency.SpeciesFrequencyRowModel;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.action.AbstractTuttiAction;
import fr.ifremer.tutti.ui.swing.util.table.AbstractSelectTableAction;
import org.apache.commons.collections.CollectionUtils;

import javax.swing.JOptionPane;
import java.util.List;

import static org.nuiton.i18n.I18n._;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since 1.0
 */
public class ComputeWeightsAction extends AbstractTuttiAction<EditCatchesUIModel, EditCatchesUI, EditCatchesUIHandler> {

    public ComputeWeightsAction(EditCatchesUIHandler handler) {
        super(handler, false);
    }

    @Override
    protected void doAction() throws Exception {
        EditCatchesUIModel model = getModel();

        computeSpeciesBatches();
        computeBenthosBatches();

        Float speciesTotalSortedWeight = model.getSpeciesTotalSortedWeight();
        if (speciesTotalSortedWeight == null) {
            speciesTotalSortedWeight = model.getSpeciesTotalSortedComputedWeight();
        }

        Float benthosTotalSortedWeight = model.getBenthosTotalSortedWeight();
        if (benthosTotalSortedWeight == null) {
            benthosTotalSortedWeight = model.getBenthosTotalSortedComputedWeight();
        }

        Float carrouselWeight = model.getCatchTotalSortedCarousselWeight();
        Float totalUnsortedWeight = model.getSpeciesTotalUnsortedComputedWeight() +
                                    model.getBenthosTotalUnsortedComputedWeight();
        Float totalSortedWeight;
        if (carrouselWeight != null) {
            totalSortedWeight = carrouselWeight;
        } else {
            totalSortedWeight = speciesTotalSortedWeight + benthosTotalSortedWeight;
        }
        model.setCatchTotalSortedComputedWeight(totalSortedWeight);
        model.setCatchTotalUnsortedComputedWeight(totalUnsortedWeight);

        Float totalWeight = model.getCatchTotalWeight();
        Float rejectedWeight = model.getCatchTotalRejectedWeight();

        if (rejectedWeight == null && totalWeight != null) {
            if (!totalWeight.equals(totalUnsortedWeight
                                    + totalSortedWeight)) {
                throw new TuttiBusinessException(_("tutti.editCatchBatch.action.computeWeights.error.incoherentTotal"));

            } else {
                model.setCatchTotalRejectedComputedWeight(totalWeight
                                                          - totalUnsortedWeight
                                                          - totalSortedWeight);
            }

        } else if (totalWeight == null) {
            if (rejectedWeight == null) {
                rejectedWeight = 0f;
                model.setCatchTotalRejectedComputedWeight(0f);
            }
            model.setCatchTotalComputedWeight(totalUnsortedWeight
                                              + totalSortedWeight
                                              + rejectedWeight);

        } else if (!totalWeight.equals(totalUnsortedWeight + totalSortedWeight + rejectedWeight)) {
            throw new TuttiBusinessException(_("tutti.editCatchBatch.action.computeWeights.error.incoherentTotal"));
        }
    }

    @Override
    public void postSuccessAction() {
        super.postSuccessAction();

        getUI().repaint();
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void computeSpeciesBatches() {
        EditCatchesUIModel model = getModel();

        Float totalSortedWeight = 0f;
        Float totalUnsortedWeight = 0f;

        List<SpeciesBatchRowModel> roots = getUI().getSpeciesTabContent().getModel().getRows();
        for (int i = 0; i < roots.size(); i++) {
            SpeciesBatchRowModel row = roots.get(i);
            if (row.isBatchRoot()) {
                Float weight = computeSpeciesBatch(row);
                if (weight == null) {
                    AbstractSelectTableAction.doSelectCell(getUI().getSpeciesTabContent().getTable(), i, 1);
                    throw new TuttiBusinessException(_("tutti.editCatchBatch.action.computeWeights.error.noWeight"));
                }
                if (SortedUnsortedEnum.SORTED.matchValue(row.getSortedUnsortedCategory().getCategoryValue())) {
                    totalSortedWeight += weight;
                } else {
                    totalUnsortedWeight += weight;
                }
            }
        }

        Number inertWeight = model.getSpeciesTotalInertWeight();
        if (inertWeight != null) {
            totalSortedWeight += inertWeight.floatValue();
        } else {
            model.setSpeciesTotalInertComputedWeight(0f);
        }

        Number livingNotItemizedWeight = model.getSpeciesTotalLivingNotItemizedWeight();
        if (livingNotItemizedWeight != null) {
            totalSortedWeight += livingNotItemizedWeight.floatValue();
        } else {
            model.setSpeciesTotalLivingNotItemizedComputedWeight(0f);
        }

        model.setSpeciesTotalSampleSortedComputedWeight(totalSortedWeight);

        Float speciesTotalSortedWeight = model.getSpeciesTotalSortedWeight();
        Float rate = getConfig().getDifferenceRateBetweenSortedAndTotalWeights();
        if (speciesTotalSortedWeight == null) {
            speciesTotalSortedWeight = totalSortedWeight;
            model.setSpeciesTotalSortedComputedWeight(totalSortedWeight);

        } else if (speciesTotalSortedWeight < totalSortedWeight) {
            throw new TuttiBusinessException(_("tutti.editCatchBatch.action.computeWeights.error.incoherentSpeciesTotalSorted"));

        } else if (speciesTotalSortedWeight < (1 + rate / 100) * totalSortedWeight) {
            // Si  le "Poids total VRAC" est saisi est que sa valeur
            // est supérieure de moins de x% (x en configuration)
            // du "Poids total Vrac trié", demander confirmation que
            // le "Poids total VRAC" est bien une valeur observée
            // sinon la remplacer par le "Poids total Vrac trié"
            String htmlMessage = String.format(
                    AbstractTuttiUIHandler.CONFIRMATION_FORMAT,
                    _("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.message", rate),
                    _("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.help"));

            int answer = JOptionPane.showConfirmDialog(getContext().getActionUI(),
                                                       htmlMessage,
                                                       _("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.title"),
                                                       JOptionPane.YES_NO_OPTION,
                                                       JOptionPane.QUESTION_MESSAGE);

            if (answer == JOptionPane.NO_OPTION) {
                model.setSpeciesTotalSortedWeight(null);
                speciesTotalSortedWeight = totalSortedWeight;
                model.setSpeciesTotalSortedComputedWeight(totalSortedWeight);
            }
        }

        model.setSpeciesTotalUnsortedComputedWeight(totalUnsortedWeight);

        Float totalWeight = totalUnsortedWeight + speciesTotalSortedWeight;
        model.setSpeciesTotalComputedWeight(totalWeight);
    }

    protected Float computeSpeciesBatch(SpeciesBatchRowModel row) {
        SampleCategory finestCategory = row.getFinestCategory();

        Float result = null;
        Float categoryWeight = finestCategory.getCategoryWeight();
        Float rowWeight = row.getWeight();

        List<SpeciesBatchRowModel> children = row.getChildBatch();
        // if the row is not a leaf
        if (!row.isBatchLeaf()) {
            Float sum = 0f;
            // make the sum of the children weights
            for (SpeciesBatchRowModel child : children) {
                Float weight = computeSpeciesBatch(child);
                if (weight == null) {
                    sum = null;
                    break;
                }
                sum += weight;
            }

            if (sum != null) {
                if (categoryWeight == null) {
                    finestCategory.setComputedWeight(sum);
                    for (SpeciesBatchRowModel child : children) {
                        child.getFinestCategory().setSubSample(false);
                    }

                } else if (categoryWeight < sum) {
                    throw new TuttiBusinessException(_("tutti.editCatchBatch.action.computeWeights.error.incoherentParentCategoryWeight"));

                } else {
                    boolean subSample = categoryWeight > sum;
                    for (SpeciesBatchRowModel child : children) {
                        child.getFinestCategory().setSubSample(subSample);
                    }
                    sum = categoryWeight;
                }
                result = sum;
            }

        } else {// the row is a leaf
            row.setComputedWeight(null);

            List<SpeciesFrequencyRowModel> frequency = row.getFrequency();

            if (CollectionUtils.isNotEmpty(frequency)) {
                // if there are frequencies, then compute their weight
                Float frequencyWeight = 0f;
                for (SpeciesFrequencyRowModel frequencyModel : frequency) {
                    Float w = frequencyModel.getWeight();
                    if (w == null) {

                        // can't sum when a null value appears
                        frequencyWeight = null;
                        break;

                    } else if (frequencyWeight != null) {

                        // still can sum weights
                        frequencyWeight += w;
                    }
                }

                if (categoryWeight == null && rowWeight != null) {
                    throw new TuttiBusinessException(_("tutti.editCatchBatch.action.computeWeights.error.incoherentRowWeightCategory"));

                } else if (categoryWeight == null && frequencyWeight != null) {
                    // if the category weight is null and the frequencies have a weight,
                    // then this weight is the result
                    finestCategory.setComputedWeight(frequencyWeight);
                    result = frequencyWeight;

                } else if (frequencyWeight != null
                           && !frequencyWeight.equals(categoryWeight)) {

                    // if the weight of the frequencies is different from the category
                    // weight, then set the weight of the sample
                    if (frequencyWeight > categoryWeight) {
                        throw new TuttiBusinessException(_("tutti.editCatchBatch.action.computeWeights.error.incoherentCategoryWeight"));

                    } else if (rowWeight == null) {
                        row.setComputedWeight(frequencyWeight);

                    } else if (!rowWeight.equals(frequencyWeight)) {
                        throw new TuttiBusinessException(_("tutti.editCatchBatch.action.computeWeights.error.incoherentRowWeightFrequency"));
                    }
                    result = categoryWeight;

                } else {
                    result = categoryWeight;
                }

            } else {
                result = categoryWeight;
            }
        }

        return result;
    }

    protected void computeBenthosBatches() {

        EditCatchesUIModel model = getModel();

        Float totalSortedWeight = 0f;
        Float totalUnsortedWeight = 0f;

        List<BenthosBatchRowModel> roots = getUI().getBenthosTabContent().getModel().getRows();
        for (int i = 0; i < roots.size(); i++) {
            BenthosBatchRowModel row = roots.get(i);
            if (row.isBatchRoot()) {
                Float weight = computeBenthosBatch(row);
                if (weight == null) {
                    AbstractSelectTableAction.doSelectCell(getUI().getBenthosTabContent().getTable(), i, 1);
                    throw new TuttiBusinessException(_("tutti.editCatchBatch.action.computeWeights.error.noWeight"));
                }
                if (SortedUnsortedEnum.SORTED.matchValue(row.getSortedUnsortedCategory().getCategoryValue())) {
                    totalSortedWeight += weight;
                } else {
                    totalUnsortedWeight += weight;
                }
            }
        }

        Number inertWeight = model.getBenthosTotalInertWeight();
        if (inertWeight != null) {
            totalSortedWeight += inertWeight.floatValue();
        } else {
            model.setBenthosTotalInertComputedWeight(0f);
        }

        Number livingNotItemizedWeight = model.getBenthosTotalLivingNotItemizedWeight();
        if (livingNotItemizedWeight != null) {
            totalSortedWeight += livingNotItemizedWeight.floatValue();
        } else {
            model.setBenthosTotalLivingNotItemizedComputedWeight(0f);
        }

        model.setBenthosTotalSampleSortedComputedWeight(totalSortedWeight);

        Float benthosTotalSortedWeight = model.getBenthosTotalSortedWeight();
        Float rate = getConfig().getDifferenceRateBetweenSortedAndTotalWeights();
        if (benthosTotalSortedWeight == null) {
            benthosTotalSortedWeight = totalSortedWeight;
            model.setBenthosTotalSortedComputedWeight(totalSortedWeight);

        } else if (benthosTotalSortedWeight < totalSortedWeight) {
            throw new TuttiBusinessException(_("tutti.editCatchBatch.action.computeWeights.error.incoherentSpeciesTotalSorted"));

        } else if (benthosTotalSortedWeight < (1 + rate / 100) * totalSortedWeight) {
            // Si  le "Poids total VRAC" est saisi est que sa valeur
            // est supérieure de moins de x% (x en configuration)
            // du "Poids total Vrac trié", demander confirmation que
            // le "Poids total VRAC" est bien une valeur observée
            // sinon la remplacer par le "Poids total Vrac trié"
            String htmlMessage = String.format(
                    AbstractTuttiUIHandler.CONFIRMATION_FORMAT,
                    _("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.message", rate),
                    _("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.help"));

            int answer = JOptionPane.showConfirmDialog(getContext().getActionUI(),
                                                       htmlMessage,
                                                       _("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.title"),
                                                       JOptionPane.YES_NO_OPTION,
                                                       JOptionPane.QUESTION_MESSAGE);

            if (answer == JOptionPane.NO_OPTION) {
                model.setSpeciesTotalSortedWeight(null);
                benthosTotalSortedWeight = totalSortedWeight;
                model.setBenthosTotalSortedComputedWeight(totalSortedWeight);
            }
        }

        model.setBenthosTotalUnsortedComputedWeight(totalUnsortedWeight);

        Float totalWeight = totalUnsortedWeight + benthosTotalSortedWeight;
        model.setBenthosTotalComputedWeight(totalWeight);
    }

    protected Float computeBenthosBatch(BenthosBatchRowModel row) {
        SampleCategory finestCategory = row.getFinestCategory();

        Float result = null;
        Float categoryWeight = finestCategory.getCategoryWeight();
        Float rowWeight = row.getWeight();

        List<BenthosBatchRowModel> children = row.getChildBatch();
        // if the row is not a leaf
        if (!row.isBatchLeaf()) {
            Float sum = 0f;
            // make the sum of the children weights
            for (BenthosBatchRowModel child : children) {
                Float weight = computeBenthosBatch(child);
                if (weight == null) {
                    sum = null;
                    break;
                }
                sum += weight;
            }

            if (sum != null) {
                if (categoryWeight == null) {
                    finestCategory.setComputedWeight(sum);
                    for (BenthosBatchRowModel child : children) {
                        child.getFinestCategory().setSubSample(false);
                    }

                } else if (categoryWeight < sum) {
                    throw new TuttiBusinessException(_("tutti.editCatchBatch.action.computeWeights.error.incoherentParentCategoryWeight"));

                } else {
                    boolean subSample = categoryWeight > sum;
                    for (BenthosBatchRowModel child : children) {
                        child.getFinestCategory().setSubSample(subSample);
                    }
                    sum = categoryWeight;
                }
                result = sum;
            }

        } else {// the row is a leaf
            row.setComputedWeight(null);

            List<BenthosFrequencyRowModel> frequency = row.getFrequency();

            if (CollectionUtils.isNotEmpty(frequency)) {
                // if there are frequencies, then compute their weight
                Float frequencyWeight = 0f;
                for (BenthosFrequencyRowModel frequencyModel : frequency) {
                    Float w = frequencyModel.getWeight();
                    if (w == null) {

                        // can't sum when a null value appears
                        frequencyWeight = null;
                        break;

                    } else if (frequencyWeight != null) {

                        // still can sum weights
                        frequencyWeight += w;
                    }
                }

                if (categoryWeight == null && rowWeight != null) {
                    throw new TuttiBusinessException(_("tutti.editCatchBatch.action.computeWeights.error.incoherentRowWeightCategory"));

                } else if (categoryWeight == null && frequencyWeight != null) {
                    // if the category weight is null and the frequencies have a weight,
                    // then this weight is the result
                    finestCategory.setComputedWeight(frequencyWeight);
                    result = frequencyWeight;

                } else if (frequencyWeight != null
                           && !frequencyWeight.equals(categoryWeight)) {

                    // if the weight of the frequencies is different from the category
                    // weight, then set the weight of the sample
                    if (frequencyWeight > categoryWeight) {
                        throw new TuttiBusinessException(_("tutti.editCatchBatch.action.computeWeights.error.incoherentCategoryWeight"));

                    } else if (rowWeight == null) {
                        row.setComputedWeight(frequencyWeight);

                    } else if (!rowWeight.equals(frequencyWeight)) {
                        throw new TuttiBusinessException(_("tutti.editCatchBatch.action.computeWeights.error.incoherentRowWeightFrequency"));
                    }
                    result = categoryWeight;

                } else {
                    result = categoryWeight;
                }

            } else {
                result = categoryWeight;
            }
        }

        return result;
    }
}
