package fr.ifremer.tutti.ichtyometer;

/*
 * #%L
 * Tutti :: Ichtyometer API
 * $Id: IchtyometerClient.java 1573 2014-02-04 16:41:40Z tchemit $
 * $HeadURL: https://forge.codelutin.com/svn/tutti/tags/tutti-3.1.2/tutti-ichtyometer/src/main/java/fr/ifremer/tutti/ichtyometer/IchtyometerClient.java $
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.intel.bluetooth.BlueCoveImpl;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.bluetooth.BluetoothStateException;
import javax.bluetooth.DataElement;
import javax.bluetooth.DeviceClass;
import javax.bluetooth.DiscoveryAgent;
import javax.bluetooth.DiscoveryListener;
import javax.bluetooth.LocalDevice;
import javax.bluetooth.RemoteDevice;
import javax.bluetooth.ServiceRecord;
import javax.bluetooth.UUID;
import javax.microedition.io.Connector;
import javax.microedition.io.StreamConnection;
import java.io.Closeable;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Ichtyometer bluetooth client.
 * <p/>
 * Created on 10/11/13.
 *
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 3.1
 */
public class IchtyometerClient implements Closeable {

    /** Logger. */
    private static final Log log = LogFactory.getLog(IchtyometerClient.class);

    /**
     * To keep a cache of discovered remote device. Keys are the name of device, values are the remote device.
     */
    protected static Map<String, RemoteDevice> REMOTE_DEVICE_CACHE;

    /**
     * To keep a cache of discovered connection url on remote devices. Keys are the name of device, values are
     * the connection of the remote device.
     */
    protected static Map<String, String> REMOTE_CONNECTION_URL_CACHE;

    public static void clearRemoteDeviceCaches() {
        REMOTE_DEVICE_CACHE = null;
        REMOTE_CONNECTION_URL_CACHE = null;
    }

    /**
     * Local bluetooth device.
     */
    protected LocalDevice localDevice;

    /**
     * Set of opened connections.
     */
    protected final Set<StreamConnection> connections = Sets.newHashSet();

    /**
     * Remote BigFin bluetooth device.
     */
    protected RemoteDevice remoteDevice;

    /**
     * Url connection to the remote device.
     */
    protected String connectionUrl;

    /**
     * Flag set to {@code true} if {@link #open(RemoteDeviceChooser, boolean)} method was invoked with success.
     */
    protected boolean open;

    /**
     * Friendly name of the device.
     */
    protected String name;

    public void open(RemoteDeviceChooser remoteDeviceChooser, boolean forceCompleteScan) throws IOException {

        Preconditions.checkState(!open, "Client is already opened");

        try {
            localDevice = LocalDevice.getLocalDevice();
        } catch (BluetoothStateException e) {
            throw new LocalDeviceNotFoundException();
        }

        if (forceCompleteScan || REMOTE_DEVICE_CACHE == null) {

            // build map of remote devices

            try {
                REMOTE_DEVICE_CACHE = discoverDevices();
            } catch (Exception e) {
                throw new RemoteDeviceNotFoundException("Could not detected devices", e);
            }
        }

        if (REMOTE_DEVICE_CACHE.isEmpty()) {
            throw new RemoteDeviceNotFoundException("No remote device found");
        }

        // ask user to choose the device
        name = remoteDeviceChooser.chooseRemoteDevice(REMOTE_DEVICE_CACHE.keySet());

        if (name == null) {
            throw new RemoteDeviceNotFoundException("User did not choose a remote device");
        }

        remoteDevice = REMOTE_DEVICE_CACHE.get(name);

        if (remoteDevice == null) {
            throw new RemoteDeviceNotFoundException(
                    "Could not find remote device with name '" + name + "'");
        }

        if (forceCompleteScan || REMOTE_CONNECTION_URL_CACHE == null) {
            REMOTE_CONNECTION_URL_CACHE = Maps.newTreeMap();
        }

        if (!REMOTE_CONNECTION_URL_CACHE.containsKey(name)) {

            int serviceIndex = 3;
            List<ServiceRecord> serviceRecords;
            try {
                serviceRecords = discoverServiceUrls(new UUID(serviceIndex), remoteDevice);
            } catch (Exception e) {
                throw new RemoteDeviceNotFoundException("Could not read remote device services", e);
            }

            if (serviceRecords.isEmpty()) {
                throw new RemoteDeviceServiceNotFoundException("No services detected.");
            }

            if (log.isInfoEnabled()) {
                log.info("Found some services for index: " + serviceIndex);
            }
            serviceRecords.addAll(serviceRecords);

            // take first service record
            ServiceRecord serviceRecord = serviceRecords.get(0);

            // get connection url
            String connectionUrl = serviceRecord.getConnectionURL(ServiceRecord.NOAUTHENTICATE_NOENCRYPT, true);

            if (log.isInfoEnabled()) {
                log.info("Found service(" + serviceIndex + "): " + connectionUrl + ", name: " + name);
            }
            REMOTE_CONNECTION_URL_CACHE.put(name, connectionUrl);
        }

        // get connection url
        connectionUrl = REMOTE_CONNECTION_URL_CACHE.get(name);

        open = true;
    }

    @Override
    public void close() throws IOException {

        if (!open) {
            return;
        }
        try {
            for (StreamConnection connection : Sets.newHashSet(connections)) {
                closeConnection(connection);
            }
        } finally {

            BlueCoveImpl.shutdown();
        }
    }

    public StreamConnection openConnection() throws IOException {

        checkIsOpened();

        StreamConnection connection = (StreamConnection) Connector.open(connectionUrl, 2);
        connections.add(connection);
        return connection;
    }

    public void closeConnection(StreamConnection connection) throws IOException {
        checkIsOpened();
        boolean remove = connections.remove(connection);
        if (!remove) {
            throw new IllegalArgumentException("Connection is not coming from this client, won't close it!");
        }
        connection.close();
    }

    public String getConnectionUrl() {
        checkIsOpened();
        return connectionUrl;
    }

    public boolean isOpen() {
        return open;
    }

    public String getName() {
        checkIsOpened();
        return name;
    }

    protected void checkIsOpened() {
        if (!open) {
            throw new IllegalStateException("Client is not opened!");
        }
    }

    protected Map<String, RemoteDevice> discoverDevices() throws BluetoothStateException, InterruptedException {

        Map<String, RemoteDevice> devices = Maps.newTreeMap();

        MyDiscoveryListener listener = new MyDiscoveryListener(this, devices, null);

        synchronized (this) {
            boolean started = localDevice.getDiscoveryAgent().startInquiry(DiscoveryAgent.LIAC, listener);
            if (started) {
                if (log.isInfoEnabled()) {
                    log.info("wait for device inquiry to complete...");
                }
                this.wait();
                if (log.isInfoEnabled()) {
                    log.info(devices.size() + " device(s) found");
                }
            }
        }
        return devices;
    }

    protected List<ServiceRecord> discoverServiceUrls(UUID serviceUUID, RemoteDevice device) throws BluetoothStateException, InterruptedException {

        List<ServiceRecord> serviceRecords = Lists.newArrayList();

        MyDiscoveryListener listener = new MyDiscoveryListener(this,
                                                               null,
                                                               serviceRecords);

        synchronized (this) {

            int[] attrIDs = new int[]{
                    0x0100 // Service name
            };

            localDevice.getDiscoveryAgent().searchServices(attrIDs,
                                                           new UUID[]{serviceUUID},
                                                           device,
                                                           listener);
            this.wait();
        }

        return serviceRecords;
    }

    protected static class MyDiscoveryListener implements DiscoveryListener {

        final Object lock;

        final Map<String, RemoteDevice> devices;

        final List<ServiceRecord> serviceRecords;

        MyDiscoveryListener(Object lock,
                            Map<String, RemoteDevice> devices,
                            List<ServiceRecord> serviceRecords) {
            this.lock = lock;
            this.devices = devices;
            this.serviceRecords = serviceRecords;
        }


        @Override
        public void deviceDiscovered(RemoteDevice btDevice, DeviceClass cod) {
            if (log.isInfoEnabled()) {
                log.info("Device " + btDevice.getBluetoothAddress() + " found");
            }

            try {
                String friendlyName = btDevice.getFriendlyName(false);
                if (log.isInfoEnabled()) {
                    log.info("Name: " + friendlyName);
                }
                devices.put(friendlyName, btDevice);
            } catch (IOException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't get name of remote", e);
                }
            }
        }

        @Override
        public void inquiryCompleted(int discType) {
            if (log.isDebugEnabled()) {
                log.debug("Device Inquiry completed!");
            }
            synchronized (lock) {
                lock.notifyAll();
            }
        }

        @Override
        public void serviceSearchCompleted(int transID, int respCode) {
            if (log.isDebugEnabled()) {
                log.debug("Service search completed!");
            }
            synchronized (lock) {
                lock.notifyAll();
            }
        }

        @Override
        public void servicesDiscovered(int transID, ServiceRecord[] servRecord) {
            for (ServiceRecord aServRecord : servRecord) {
                String url = aServRecord.getConnectionURL(ServiceRecord.NOAUTHENTICATE_NOENCRYPT, false);
                if (url == null) {
                    continue;
                }
                serviceRecords.add(aServRecord);
                String serviceName = getServiceName(aServRecord);
                if (log.isDebugEnabled()) {
                    log.debug("service found " + url + (serviceName == null ? "" : ", name: " + serviceName));
                }
            }
        }
    }

    public static String getServiceName(ServiceRecord serviceRecord) {
        DataElement serviceName = serviceRecord.getAttributeValue(0x0100);
        String result;

        if (serviceName == null) {
            result = null;
        } else {
            result = String.valueOf(serviceName.getValue());
        }
        return result;
    }

}
