/*
 * Decompiled with CFR 0.152.
 */
package com.opentable.db.postgres.embedded;

import com.google.common.base.Joiner;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.io.Closeables;
import com.opentable.db.postgres.embedded.BundledPostgresBinaryResolver;
import com.opentable.db.postgres.embedded.PgBinaryResolver;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.ServerSocket;
import java.net.UnknownHostException;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import javax.sql.DataSource;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.SystemUtils;
import org.apache.commons.lang3.time.StopWatch;
import org.postgresql.ds.PGSimpleDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tukaani.xz.XZInputStream;

public class EmbeddedPostgres
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(EmbeddedPostgres.class);
    private static final String JDBC_FORMAT = "jdbc:postgresql://localhost:%s/%s?user=%s";
    private static final String PG_STOP_MODE = "fast";
    private static final String PG_STOP_WAIT_S = "5";
    private static final String PG_SUPERUSER = "postgres";
    private static final int PG_STARTUP_WAIT_MS = 10000;
    private static final String LOCK_FILE_NAME = "epg-lock";
    private final File pgDir;
    private final File dataDirectory;
    private final File lockFile;
    private final UUID instanceId = UUID.randomUUID();
    private final int port;
    private final AtomicBoolean started = new AtomicBoolean();
    private final AtomicBoolean closed = new AtomicBoolean();
    private final Map<String, String> postgresConfig;
    private volatile Process postmaster;
    private volatile FileOutputStream lockStream;
    private volatile FileLock lock;
    private final boolean cleanDataDirectory;
    private ProcessBuilder.Redirect errorRedirector = ProcessBuilder.Redirect.INHERIT;
    private ProcessBuilder.Redirect outputRedirector = ProcessBuilder.Redirect.INHERIT;
    private static final AtomicReference<File> BINARY_DIR = new AtomicReference();
    private static final Lock PREPARE_BINARIES_LOCK = new ReentrantLock();

    EmbeddedPostgres(File parentDirectory, File dataDirectory, boolean cleanDataDirectory, Map<String, String> postgresConfig, int port, PgBinaryResolver pgBinaryResolver, ProcessBuilder.Redirect errorRedirector, ProcessBuilder.Redirect outputRedirector) throws IOException {
        this.cleanDataDirectory = cleanDataDirectory;
        this.postgresConfig = ImmutableMap.copyOf(postgresConfig);
        this.port = port;
        this.pgDir = EmbeddedPostgres.prepareBinaries(pgBinaryResolver);
        this.errorRedirector = errorRedirector;
        this.outputRedirector = outputRedirector;
        if (parentDirectory != null) {
            EmbeddedPostgres.mkdirs(parentDirectory);
            this.cleanOldDataDirectories(parentDirectory);
            this.dataDirectory = (File)MoreObjects.firstNonNull((Object)dataDirectory, (Object)new File(parentDirectory, this.instanceId.toString()));
        } else {
            this.dataDirectory = dataDirectory;
        }
        Preconditions.checkArgument((this.dataDirectory != null ? 1 : 0) != 0, (Object)"null data directory");
        LOG.trace("{} postgres data directory is {}", (Object)this.instanceId, (Object)this.dataDirectory);
        Verify.verify((this.dataDirectory.exists() || this.dataDirectory.mkdir() ? 1 : 0) != 0, (String)"Failed to mkdir %s", (Object[])new Object[]{this.dataDirectory});
        this.lockFile = new File(this.dataDirectory, LOCK_FILE_NAME);
        if (cleanDataDirectory || !new File(dataDirectory, "postgresql.conf").exists()) {
            this.initdb();
        }
        this.lock();
        this.startPostmaster();
    }

    public DataSource getTemplateDatabase() {
        return this.getDatabase(PG_SUPERUSER, "template1");
    }

    public DataSource getTemplateDatabase(Map<String, String> properties) {
        return this.getDatabase(PG_SUPERUSER, "template1", properties);
    }

    public DataSource getPostgresDatabase() {
        return this.getDatabase(PG_SUPERUSER, PG_SUPERUSER);
    }

    public DataSource getPostgresDatabase(Map<String, String> properties) {
        return this.getDatabase(PG_SUPERUSER, PG_SUPERUSER, properties);
    }

    public DataSource getDatabase(String userName, String dbName) {
        return this.getDatabase(userName, dbName, Collections.emptyMap());
    }

    public DataSource getDatabase(String userName, String dbName, Map<String, String> properties) {
        PGSimpleDataSource ds = new PGSimpleDataSource();
        ds.setServerName("localhost");
        ds.setPortNumber(this.port);
        ds.setDatabaseName(dbName);
        ds.setUser(userName);
        properties.forEach((propertyKey, propertyValue) -> {
            try {
                ds.setProperty(propertyKey, propertyValue);
            }
            catch (SQLException e) {
                throw new RuntimeException(e);
            }
        });
        return ds;
    }

    public String getJdbcUrl(String userName, String dbName) {
        return String.format(JDBC_FORMAT, this.port, dbName, userName);
    }

    public int getPort() {
        return this.port;
    }

    private static int detectPort() throws IOException {
        try (ServerSocket socket = new ServerSocket(0);){
            int n = socket.getLocalPort();
            return n;
        }
    }

    private void lock() throws IOException {
        this.lockStream = new FileOutputStream(this.lockFile);
        this.lock = this.lockStream.getChannel().tryLock();
        Verify.verify((this.lock != null ? 1 : 0) != 0, (String)"could not lock %s", (Object[])new Object[]{this.lockFile});
    }

    private void initdb() {
        StopWatch watch = new StopWatch();
        watch.start();
        EmbeddedPostgres.system(this.errorRedirector, this.outputRedirector, this.pgBin("initdb"), "-A", "trust", "-U", PG_SUPERUSER, "-D", this.dataDirectory.getPath(), "-E", "UTF-8");
        LOG.info("{} initdb completed in {}", (Object)this.instanceId, (Object)watch);
    }

    private void startPostmaster() throws IOException {
        StopWatch watch = new StopWatch();
        watch.start();
        Preconditions.checkState((!this.started.getAndSet(true) ? 1 : 0) != 0, (Object)"Postmaster already started");
        ArrayList args = Lists.newArrayList((Object[])new String[]{this.pgBin("pg_ctl"), "-D", this.dataDirectory.getPath(), "-o", Joiner.on((String)" ").join(this.createInitOptions()), "start"});
        ProcessBuilder builder = new ProcessBuilder(args);
        builder.redirectErrorStream(true);
        builder.redirectError(this.outputRedirector);
        builder.redirectOutput(this.outputRedirector);
        this.postmaster = builder.start();
        LOG.info("{} postmaster started as {} on port {}.  Waiting up to {}ms for server startup to finish.", new Object[]{this.instanceId, this.postmaster.toString(), this.port, 10000});
        Runtime.getRuntime().addShutdownHook(this.newCloserThread());
        this.waitForServerStartup(watch);
    }

    private List<String> createInitOptions() {
        ArrayList initOptions = Lists.newArrayList((Object[])new String[]{"-p", Integer.toString(this.port), "-F"});
        for (Map.Entry<String, String> config : this.postgresConfig.entrySet()) {
            initOptions.add("-c");
            initOptions.add(config.getKey() + "=" + config.getValue());
        }
        return initOptions;
    }

    private void waitForServerStartup(StopWatch watch) throws UnknownHostException, IOException {
        SQLException lastCause = null;
        long start = System.nanoTime();
        long maxWaitNs = TimeUnit.NANOSECONDS.convert(10000L, TimeUnit.MILLISECONDS);
        while (System.nanoTime() - start < maxWaitNs) {
            try {
                this.verifyReady();
                LOG.info("{} postmaster startup finished in {}", (Object)this.instanceId, (Object)watch);
                return;
            }
            catch (SQLException e) {
                lastCause = e;
                LOG.trace("While waiting for server startup", (Throwable)e);
                try {
                    Thread.sleep(100L);
                }
                catch (InterruptedException e2) {
                    Thread.currentThread().interrupt();
                    return;
                }
            }
        }
        throw new IOException("Gave up waiting for server to start after 10000ms", lastCause);
    }

    private void verifyReady() throws SQLException {
        try (Connection c = this.getPostgresDatabase().getConnection();
             Statement s = c.createStatement();
             ResultSet rs = s.executeQuery("SELECT 1");){
            Verify.verify((rs.next() ? 1 : 0) != 0, (String)"expecting single row", (Object[])new Object[0]);
            Verify.verify((1 == rs.getInt(1) ? 1 : 0) != 0, (String)"expecting 1", (Object[])new Object[0]);
            Verify.verify((!rs.next() ? 1 : 0) != 0, (String)"expecting single row", (Object[])new Object[0]);
        }
    }

    private Thread newCloserThread() {
        Thread closeThread = new Thread(new Runnable(){

            @Override
            public void run() {
                try {
                    Closeables.close((Closeable)EmbeddedPostgres.this, (boolean)true);
                }
                catch (IOException ex) {
                    LOG.error("Unexpected IOException from Closeables.close", (Throwable)ex);
                }
            }
        });
        closeThread.setName("postgres-" + this.instanceId + "-closer");
        return closeThread;
    }

    @Override
    public void close() throws IOException {
        if (this.closed.getAndSet(true)) {
            return;
        }
        StopWatch watch = new StopWatch();
        watch.start();
        try {
            this.pgCtl(this.dataDirectory, "stop");
            LOG.info("{} shut down postmaster in {}", (Object)this.instanceId, (Object)watch);
        }
        catch (Exception e) {
            LOG.error("Could not stop postmaster " + this.instanceId, (Throwable)e);
        }
        if (this.lock != null) {
            this.lock.release();
        }
        Closeables.close((Closeable)this.lockStream, (boolean)true);
        if (this.cleanDataDirectory && System.getProperty("ot.epg.no-cleanup") == null) {
            try {
                FileUtils.deleteDirectory((File)this.dataDirectory);
            }
            catch (IOException e) {
                LOG.error("Could not clean up directory {}", (Object)this.dataDirectory.getAbsolutePath());
            }
        } else {
            LOG.info("Did not clean up directory {}", (Object)this.dataDirectory.getAbsolutePath());
        }
    }

    private void pgCtl(File dir, String action) {
        EmbeddedPostgres.system(this.pgBin("pg_ctl"), "-D", dir.getPath(), action, "-m", PG_STOP_MODE, "-t", PG_STOP_WAIT_S, "-w");
    }

    private void cleanOldDataDirectories(File parentDirectory) {
        File[] children = parentDirectory.listFiles();
        if (children == null) {
            return;
        }
        for (File dir : children) {
            boolean isTooNew;
            if (!dir.isDirectory()) continue;
            File lockFile = new File(dir, LOCK_FILE_NAME);
            boolean bl = isTooNew = System.currentTimeMillis() - lockFile.lastModified() < 600000L;
            if (!lockFile.exists() || isTooNew) continue;
            try (FileOutputStream fos = new FileOutputStream(lockFile);
                 FileLock lock = fos.getChannel().tryLock();){
                if (lock == null) continue;
                LOG.info("Found stale data directory {}", (Object)dir);
                if (new File(dir, "postmaster.pid").exists()) {
                    try {
                        this.pgCtl(dir, "stop");
                        LOG.info("Shut down orphaned postmaster!");
                    }
                    catch (Exception e) {
                        if (LOG.isDebugEnabled()) {
                            LOG.warn("Failed to stop postmaster " + dir, (Throwable)e);
                        }
                        LOG.warn("Failed to stop postmaster " + dir + ": " + e.getMessage());
                    }
                }
                FileUtils.deleteDirectory((File)dir);
            }
            catch (OverlappingFileLockException e) {
                LOG.trace("While cleaning old data directories", (Throwable)e);
            }
            catch (Exception e) {
                LOG.warn("While cleaning old data directories", (Throwable)e);
            }
        }
    }

    private String pgBin(String binaryName) {
        String extension = SystemUtils.IS_OS_WINDOWS ? ".exe" : "";
        return new File(this.pgDir, "bin/" + binaryName + extension).getPath();
    }

    private static File getWorkingDirectory() {
        File tempWorkingDirectory = new File(System.getProperty("java.io.tmpdir"), "embedded-pg");
        return new File(System.getProperty("ot.epg.working-dir", tempWorkingDirectory.getPath()));
    }

    public static EmbeddedPostgres start() throws IOException {
        return EmbeddedPostgres.builder().start();
    }

    public static Builder builder() {
        return new Builder();
    }

    private static List<String> system(String ... command) {
        return EmbeddedPostgres.system(ProcessBuilder.Redirect.INHERIT, ProcessBuilder.Redirect.INHERIT, command);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static List<String> system(ProcessBuilder.Redirect errorRedirector, ProcessBuilder.Redirect outputRedirector, String ... command) {
        try {
            ProcessBuilder builder = new ProcessBuilder(command);
            builder.redirectError(errorRedirector);
            builder.redirectOutput(outputRedirector);
            Process process = builder.start();
            Verify.verify((0 == process.waitFor() ? 1 : 0) != 0, (String)"Process %s failed\n%s", (Object[])new Object[]{Arrays.asList(command), IOUtils.toString((InputStream)process.getErrorStream())});
            try (InputStream stream = process.getInputStream();){
                List list = IOUtils.readLines((InputStream)stream);
                return list;
            }
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    private static void mkdirs(File dir) {
        Verify.verify((dir.mkdirs() || dir.isDirectory() && dir.exists() ? 1 : 0) != 0, (String)"could not create %s", (Object[])new Object[]{dir});
    }

    private static String getOS() {
        if (SystemUtils.IS_OS_WINDOWS) {
            return "Windows";
        }
        if (SystemUtils.IS_OS_MAC_OSX) {
            return "Darwin";
        }
        if (SystemUtils.IS_OS_LINUX) {
            return "Linux";
        }
        throw new UnsupportedOperationException("Unknown OS " + SystemUtils.OS_NAME);
    }

    private static String getArchitecture() {
        return "amd64".equals(SystemUtils.OS_ARCH) ? "x86_64" : SystemUtils.OS_ARCH;
    }

    private static void extractTxz(String tbzPath, String targetDir) throws IOException {
        try (InputStream in = Files.newInputStream(Paths.get(tbzPath, new String[0]), new OpenOption[0]);
             XZInputStream xzIn = new XZInputStream(in);
             TarArchiveInputStream tarIn = new TarArchiveInputStream((InputStream)xzIn);){
            TarArchiveEntry entry;
            while ((entry = tarIn.getNextTarEntry()) != null) {
                String individualFile = entry.getName();
                File fsObject = new File(targetDir + "/" + individualFile);
                if (entry.isSymbolicLink() || entry.isLink()) {
                    Path target = FileSystems.getDefault().getPath(entry.getLinkName(), new String[0]);
                    Files.createSymbolicLink(fsObject.toPath(), target, new FileAttribute[0]);
                } else if (entry.isFile()) {
                    byte[] content = new byte[(int)entry.getSize()];
                    int read = tarIn.read(content, 0, content.length);
                    Verify.verify((read != -1 ? 1 : 0) != 0, (String)"could not read %s", (Object[])new Object[]{individualFile});
                    EmbeddedPostgres.mkdirs(fsObject.getParentFile());
                    try (FileOutputStream outputFile = new FileOutputStream(fsObject);){
                        IOUtils.write((byte[])content, (OutputStream)outputFile);
                    }
                } else if (entry.isDirectory()) {
                    EmbeddedPostgres.mkdirs(fsObject);
                } else {
                    throw new UnsupportedOperationException(String.format("Unsupported entry found: %s", individualFile));
                }
                if (!individualFile.startsWith("bin/") && !individualFile.startsWith("./bin/")) continue;
                fsObject.setExecutable(true);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Loose catch block
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static File prepareBinaries(PgBinaryResolver pgBinaryResolver) {
        PREPARE_BINARIES_LOCK.lock();
        try {
            File pgDir;
            File pgTbz;
            block68: {
                InputStream pgBinary;
                if (BINARY_DIR.get() != null) {
                    File file = BINARY_DIR.get();
                    return file;
                }
                String system = EmbeddedPostgres.getOS();
                String machineHardware = EmbeddedPostgres.getArchitecture();
                LOG.info("Detected a {} {} system", (Object)system, (Object)machineHardware);
                try {
                    pgTbz = File.createTempFile("pgpg", "pgpg");
                    pgBinary = pgBinaryResolver.getPgBinary(system, machineHardware);
                }
                catch (IOException e) {
                    throw new ExceptionInInitializerError(e);
                }
                if (pgBinary == null) {
                    throw new IllegalStateException("No Postgres binary found for " + system + " / " + machineHardware);
                }
                try (DigestInputStream pgArchiveData = new DigestInputStream(pgBinary, MessageDigest.getInstance("MD5"));
                     FileOutputStream os = new FileOutputStream(pgTbz);){
                    IOUtils.copy((InputStream)pgArchiveData, (OutputStream)os);
                    pgArchiveData.close();
                    os.close();
                    String pgDigest = Hex.encodeHexString((byte[])pgArchiveData.getMessageDigest().digest());
                    pgDir = new File(EmbeddedPostgres.getWorkingDirectory(), String.format("PG-%s", pgDigest));
                    EmbeddedPostgres.mkdirs(pgDir);
                    File unpackLockFile = new File(pgDir, LOCK_FILE_NAME);
                    File pgDirExists = new File(pgDir, ".exists");
                    if (pgDirExists.exists()) break block68;
                    try {
                        try (FileOutputStream lockStream = new FileOutputStream(unpackLockFile);
                             FileLock unpackLock = lockStream.getChannel().tryLock();){
                            if (unpackLock != null) {
                                try {
                                    Preconditions.checkState((!pgDirExists.exists() ? 1 : 0) != 0, (Object)"unpack lock acquired but .exists file is present.");
                                    LOG.info("Extracting Postgres...");
                                    EmbeddedPostgres.extractTxz(pgTbz.getPath(), pgDir.getPath());
                                    Verify.verify((boolean)pgDirExists.createNewFile(), (String)"couldn't make .exists file", (Object[])new Object[0]);
                                }
                                catch (Exception e) {
                                    LOG.error("while unpacking Postgres", (Throwable)e);
                                }
                            } else {
                                int maxAttempts = 60;
                                while (!pgDirExists.exists() && --maxAttempts > 0) {
                                    Thread.sleep(1000L);
                                }
                                Verify.verify((boolean)pgDirExists.exists(), (String)"Waited 60 seconds for postgres to be unpacked but it never finished!", (Object[])new Object[0]);
                            }
                        }
                        if (!unpackLockFile.exists()) break block68;
                    }
                    catch (Throwable throwable) {
                        if (!unpackLockFile.exists()) throw throwable;
                        Verify.verify((boolean)unpackLockFile.delete(), (String)"could not remove lock file %s", (Object[])new Object[]{unpackLockFile.getAbsolutePath()});
                        throw throwable;
                    }
                    Verify.verify((boolean)unpackLockFile.delete(), (String)"could not remove lock file %s", (Object[])new Object[]{unpackLockFile.getAbsolutePath()});
                }
                catch (IOException | NoSuchAlgorithmException e) {
                    try {
                        throw new ExceptionInInitializerError(e);
                        catch (InterruptedException ie) {
                            Thread.currentThread().interrupt();
                            throw new ExceptionInInitializerError(ie);
                        }
                    }
                    catch (Throwable throwable) {
                        Verify.verify((boolean)pgTbz.delete(), (String)"could not delete %s", (Object[])new Object[]{pgTbz});
                        throw throwable;
                    }
                }
            }
            Verify.verify((boolean)pgTbz.delete(), (String)"could not delete %s", (Object[])new Object[]{pgTbz});
            BINARY_DIR.set(pgDir);
            LOG.info("Postgres binaries at {}", (Object)pgDir);
            File file = pgDir;
            return file;
        }
        finally {
            PREPARE_BINARIES_LOCK.unlock();
        }
    }

    public String toString() {
        return "EmbeddedPG-" + this.instanceId;
    }

    static /* synthetic */ File access$100() {
        return EmbeddedPostgres.getWorkingDirectory();
    }

    public static class Builder {
        private final File parentDirectory = EmbeddedPostgres.access$100();
        private File builderDataDirectory;
        private final Map<String, String> config = Maps.newHashMap();
        private boolean builderCleanDataDirectory = true;
        private int builderPort = 0;
        private PgBinaryResolver pgBinaryResolver = new BundledPostgresBinaryResolver();
        private ProcessBuilder.Redirect errRedirector = ProcessBuilder.Redirect.INHERIT;
        private ProcessBuilder.Redirect outRedirector = ProcessBuilder.Redirect.INHERIT;

        Builder() {
            this.config.put("timezone", "UTC");
            this.config.put("synchronous_commit", "off");
            this.config.put("max_connections", "300");
        }

        public Builder setCleanDataDirectory(boolean cleanDataDirectory) {
            this.builderCleanDataDirectory = cleanDataDirectory;
            return this;
        }

        public Builder setDataDirectory(File directory) {
            this.builderDataDirectory = directory;
            return this;
        }

        public Builder setServerConfig(String key, String value) {
            this.config.put(key, value);
            return this;
        }

        public Builder setPort(int port) {
            this.builderPort = port;
            return this;
        }

        public Builder setErrorRedirector(ProcessBuilder.Redirect errRedirector) {
            this.errRedirector = errRedirector;
            return this;
        }

        public Builder setOutputRedirector(ProcessBuilder.Redirect outRedirector) {
            this.outRedirector = outRedirector;
            return this;
        }

        public Builder setPgBinaryResolver(PgBinaryResolver pgBinaryResolver) {
            this.pgBinaryResolver = pgBinaryResolver;
            return this;
        }

        public EmbeddedPostgres start() throws IOException {
            if (this.builderPort == 0) {
                this.builderPort = EmbeddedPostgres.detectPort();
            }
            return new EmbeddedPostgres(this.parentDirectory, this.builderDataDirectory, this.builderCleanDataDirectory, this.config, this.builderPort, this.pgBinaryResolver, this.errRedirector, this.outRedirector);
        }
    }
}

