/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: OfferListHandler.java 1844 2010-12-22 14:30:09Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.5.1/vradi-swing/src/main/java/com/jurismarches/vradi/ui/offer/OfferListHandler.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.ui.offer;

import static org.nuiton.i18n.I18n._;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.beans.PropertyChangeEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import com.jurismarches.vradi.ui.email.helpers.EmailDataHelper;
import com.jurismarches.vradi.ui.thesaurus.helpers.ThesaurusSelectionManager;
import com.jurismarches.vradi.ui.task.VradiTask;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JSeparator;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.TableColumnModelEvent;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;

import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.ui.VradiMainUI;
import com.jurismarches.vradi.ui.admin.AdminHandler;
import com.jurismarches.vradi.ui.email.EmailHandler;
import com.jurismarches.vradi.ui.email.QueryMakerViewUI;
import com.jurismarches.vradi.ui.helpers.VradiComparators;
import com.jurismarches.vradi.ui.offer.models.OfferListTableModel;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.ErrorDialogUI;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.decorator.HighlighterFactory;
import org.jdesktop.swingx.event.TableColumnModelExtListener;
import org.jdesktop.swingx.table.TableColumnExt;
import org.nuiton.wikitty.entities.BusinessEntity;
import org.nuiton.wikitty.WikittyException;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.beans.FormPagedResult;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.ModificationTag;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.VradiUser;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.ui.helpers.UIHelper;
import com.jurismarches.vradi.ui.search.SearchHandler;
import com.jurismarches.vradi.ui.search.SearchUI;

/**
 * @author letellier
 */
public class OfferListHandler {

    static private final Log log = LogFactory.getLog(OfferListHandler.class);

    static final public int TABPANEL_LABEL_MAX_CHAR_NB = 30;

    /**
     * Methode pour initialiser l'ui principale sans l'afficher.
     *
     * @param rootContext le context applicatif
     * @return l'ui instancie et initialisee mais non visible encore
     */
    public OfferListUI initUI(JAXXContext rootContext) {

        JAXXInitialContext context = new JAXXInitialContext().add(rootContext).add(this);

        SearchHandler searchHandler = UIHelper.getHandler(context, SearchHandler.class);

        // show main ui
        OfferListTableModel offerListTableModel = new OfferListTableModel();
        OfferListColumnFactory offerListColumnFactory = new OfferListColumnFactory();
        
        context.add(offerListTableModel);
        context.add(offerListColumnFactory);

        OfferListUI ui = new OfferListUI(context);
        
        // add listeners
        JXTable listTable = ui.getListTable();
        listTable.getColumnModel().addColumnModelListener(
                getOfferListTableColumnModelListener());
        
        MouseListener offerListTableMouseListener = getOfferListTableMouseListener();
        listTable.addMouseListener(offerListTableMouseListener);
        
        TableModelListener offerListTableModelListener = getOfferListTableModelListener(listTable);
        offerListTableModel.addTableModelListener(offerListTableModelListener);
        
        // listTable cellRenderer
        // TODO SLE 30/09/2010 : Refactor it !
        OfferListTableModel.OfferListTableCellRenderer renderer =
                new OfferListTableModel.OfferListTableCellRenderer();
        listTable.setDefaultRenderer(Boolean.class, renderer);
        listTable.setDefaultRenderer(Date.class, renderer);
        listTable.setDefaultRenderer(Double.class, renderer);
        listTable.setDefaultRenderer(String.class, renderer);
        listTable.setDefaultRenderer(XmlStream.class, renderer);
        listTable.setDefaultRenderer(Status.class, renderer);

        // listTable highlighter
        Highlighter highlighter = HighlighterFactory.createAlternateStriping(
                HighlighterFactory.NOTEPAD, HighlighterFactory.GENERIC_GRAY);
        listTable.addHighlighter(highlighter);

        // Show search ui
        SearchUI searchUI = searchHandler.initUI(rootContext, offerListTableModel);
        searchUI.setOfferListUI(true);

        ui.setSearchUI(searchUI);
        
        return ui;
    }

    protected TableModelListener getOfferListTableModelListener(final JXTable listTable) {
        return new TableModelListener() {
            @Override
            public void tableChanged(TableModelEvent e) {
                listTable.packAll();
            }
        };
    }

    protected TableColumnModelExtListener getOfferListTableColumnModelListener() {
        return new TableColumnModelExtListener() {
            @Override
            public void columnPropertyChange(
                    PropertyChangeEvent paramPropertyChangeEvent) {
                String propertyName = paramPropertyChangeEvent.getPropertyName();
                
                if ("visible".equals(propertyName)) {
                    TableColumnExt column = (TableColumnExt) paramPropertyChangeEvent.getSource();
                    String fqFieldName = (String) column.getClientProperty("fqFieldName");

                    if (log.isTraceEnabled()) {
                        log.trace("clientprop: " + fqFieldName);
                        log.trace("identifier: " + column.getIdentifier());
                    }
                    
                    Boolean visible = (Boolean) paramPropertyChangeEvent.getNewValue();
                    if (visible) {
                        VradiHelper.addVradiListColumns(fqFieldName);
                    } else {
                        VradiHelper.removeVradiListColumns(fqFieldName);
                    }
                }
            }
            
            @Override
            public void columnAdded(TableColumnModelEvent e) {
            }
            @Override
            public void columnRemoved(TableColumnModelEvent e) {
            }
            @Override
            public void columnMoved(TableColumnModelEvent e) {
            }
            @Override
            public void columnMarginChanged(ChangeEvent e) {
            }
            @Override
            public void columnSelectionChanged(ListSelectionEvent e) {
            }
        };
    }

    /**
     * Implementation du click (droit/gauche) sur la table des offres résultats/
     * 
     * Comportement :
     * <ul>
     *  <li>double clic : ouvre l'offre
     *  <li>clic droit : changer le status, ajouter a une session de mail
     * </ul>
     * 
     * @return le mouse listener
     */
    public MouseListener getOfferListTableMouseListener() {
        // double click on listTable shows selected offer
        return new MouseAdapter() {
           
            @Override
            public void mouseClicked(MouseEvent e) {
                JTable listTable = (JTable)e.getSource();
                OfferListUI offerListUI = VradiContext.OFFERT_LIST_UI_ENTRY_DEF.getContextValue(VradiContext.get());
                
                // double clic
                if (e.getButton() == MouseEvent.BUTTON1 &&
                        e.getClickCount() == 2) {
                    try {
                        offerListUI.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
                        selectOffer(offerListUI, listTable);
                    } catch (Exception ex) {
                        if (log.isErrorEnabled()) {
                            log.error("Can't open offer", ex);
                        }
                        ErrorDialogUI.showError(ex);
                    } finally {
                        offerListUI.setCursor(null);
                    }

                // clic droit
                } else if (e.getButton() == MouseEvent.BUTTON3) {
                    int[] selectedRows = offerListUI.getListTable().getSelectedRows();
                    int rowClicked = offerListUI.getListTable().rowAtPoint(e.getPoint());
                    boolean rowClickedSelected = false;
                    if (selectedRows == null || selectedRows.length == 0) {
                        offerListUI.getListTable().getSelectionModel().setSelectionInterval(rowClicked, rowClicked);
                        rowClickedSelected = true;
                    } else {
                        int i = 0;
                        while(i < selectedRows.length && !rowClickedSelected) {
                            rowClickedSelected = rowClicked == selectedRows[i++];
                        }
                    }
                    
                    if (rowClickedSelected) {
                        showOfferListContextMenu(offerListUI, e);
                    }
                }
            }
        };
    }

    /**
     * Le menu contextuel lors d'un clic droit sur la table.
     * 
     * @param offerListUI l'ui d'origine
     * @param e event du clic
     */
    protected void showOfferListContextMenu(final OfferListUI offerListUI, final MouseEvent e) {
        JPopupMenu menu = new JPopupMenu();
        JMenu changeStatus = new JMenu(_("vradi.offer.status.change"));

        List<Status> statuses = AdminHandler.getAllStatuses();
        for (Status status : statuses) {
            JMenuItem statusItem = new JMenuItem(_(status.getName()));
            statusItem.addActionListener(new StatusActionListener(offerListUI, status));
            changeStatus.add(statusItem);
        }

        menu.add(changeStatus);
        menu.add(new JSeparator());

        JMenu addSending = new JMenu(_("vradi.offer.email.add"));

        // Menu to add user or group sending
        JMenuItem addUserSendingItem = new JMenuItem(_("vradi.email.addUser"));
        addUserSendingItem.addActionListener(new UserAddSendingListener(offerListUI));

        JMenuItem addgroupSendingItem = new JMenuItem(_("vradi.email.addGroup"));
        addgroupSendingItem.addActionListener(new GroupAddSendingListener(offerListUI));

        addSending.add(addUserSendingItem);
        addSending.add(addgroupSendingItem);

        menu.add(addSending);

        // Menu to remove form from session
        // Menu not asked
//        JMenuItem removeFormSession = new JMenuItem(_("vradi.offer.email.removeForm"));
//        removeFormSession.addActionListener(new RemoveFormSessionListener(offerListUI));

//        menu.add(new JSeparator());
//        menu.add(removeFormSession);

        menu.show(e.getComponent(), e.getX(), e.getY());
    }

    /**
     * Il est possible de faire un clic droit dans la liste des résutats
     * pour ajouter un appel d'offre directement dans la derniere session
     * ouverte.
     */
    protected abstract class AddSendingListener implements ActionListener {
        protected OfferListUI offerListUI;
        protected List<String> formsIdsToUpdate = new ArrayList<String>();

        protected AddSendingListener(OfferListUI offerListUI) {
            this.offerListUI = offerListUI;

            int[] selectedRows = offerListUI.getListTable().getSelectedRows();

            for(int row : selectedRows) {
                JXTable listTable = offerListUI.getListTable();
                OfferListTableModel tableModel = offerListUI.getOfferListTableModel();
                row = listTable.convertRowIndexToModel(row);

                FormPagedResult formPage = tableModel.getFormPagedResult();
                List<String> formsIdsToShow = formPage.getFormsIdsToShow();

                String formId = formsIdsToShow.get(row);
                if (formId != null) {
                    formsIdsToUpdate.add(formId);
                }
            }
        }

        protected AdminHandler getAdminHandler() {
            return UIHelper.getHandler(offerListUI, AdminHandler.class);
        }

        protected <E extends BusinessEntity> void openUI(final Class<E> clazz, String title, List<E> all) {

            try {

                Session currentSession = VradiService.getVradiDataService().getLastOpenSession();

                if (currentSession == null) {
                    JOptionPane.showMessageDialog(offerListUI,
                            _("Can't found any opened mail session, please create a new session in mail management"),
                            _("Error"), JOptionPane.ERROR_MESSAGE);
                }
                else {
                    List<Form> forms = VradiService.getWikittyProxy().restore(Form.class, formsIdsToUpdate);
                    Status selectionne = VradiHelper.getSelectionneStatus();

                    int responce = JOptionPane.showConfirmDialog(
                            getUI(offerListUI),
                            _("vradi.email.confirmSelectionForm", EmailDataHelper.convertFormsToString(forms)),
                            _("vradi.email.confirmSelectionForm.title"),
                            JOptionPane.YES_NO_OPTION);

                    if (responce == JOptionPane.YES_OPTION) {
                        changeAllStatus(getUI(offerListUI), forms, selectionne);
                    } else {
                        return;
                    }

                    VradiMainUI mainUI = VradiContext.MAIN_UI_ENTRY_DEF.getContextValue(offerListUI);
                    final JDialog ui = new JDialog(mainUI, _(title));
                    ui.setLayout(new BorderLayout());

                    final QueryMakerViewUI<E> makerViewUI = new QueryMakerViewUI<E>(offerListUI);
                    makerViewUI.setSession(currentSession);

                    makerViewUI.setFormsIds(formsIdsToUpdate);
                    makerViewUI.setClazz(clazz);

                    makerViewUI.init(all, new ArrayList<E>());

                    ui.add(makerViewUI, BorderLayout.CENTER);
                    JButton validButton = new JButton(_("vradi.action.valid"));
                    validButton.addActionListener(new ActionListener() {
                        @Override
                        public void actionPerformed(ActionEvent e) {

                            addUserAndGroupSending(makerViewUI, clazz);

                            ui.setVisible(false);
                            ui.dispose();
                        }
                    });
                    ui.add(validButton, BorderLayout.SOUTH);
                    ui.setModal(true);
                    ui.setSize(800, 600);
                    ui.setLocationRelativeTo(mainUI);
                    ui.setVisible(true);
                }
            }
            catch (VradiException ex) {
                if (log.isErrorEnabled()) {
                    log.error("Can't get current opened session", ex);
                }
            }
        }
    }

    protected <E extends BusinessEntity> void addUserAndGroupSending(final QueryMakerViewUI<E> ui, final Class<E> clazz) {

        VradiTask<Void> addUserAndGroupSendingTask = new VradiTask<Void>(ui, false) {
            @Override
            public Void doAction() throws Exception {

                try {
                    Form form = ui.getForm();
                    Session session = ui.getSession();
                    EmailHandler handler = ui.getHandler();

                    if (form != null) {
                        if (clazz == User.class) {
                            handler.addAllUserSending(ui, session, form.getWikittyId(), ui.getQueryMakerForm().getListDatas());
                        } else {
                            handler.addAllGroupSending(ui, session, form.getWikittyId(), ui.getQueryMakerForm().getListDatas());
                        }
                    } else {
                        List<String> formsIds = ui.getFormsIds();
                        if (clazz == User.class) {
                            handler.addAllUserSending(ui, session, formsIds, ui.getQueryMakerForm().getListDatas());
                        } else {
                            handler.addAllGroupSending(ui, session, formsIds, ui.getQueryMakerForm().getListDatas());
                        }
                    }
                } catch (Exception eee) {
                    log.error("Cant add group or user to session : ", eee);
                    ErrorDialogUI.showError(eee);
                }
                return null;
            }
        };

        addUserAndGroupSendingTask.execute();
    }

    protected class GroupAddSendingListener extends AddSendingListener {

        public GroupAddSendingListener(OfferListUI offerListUI) {
            super(offerListUI);
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            openUI(Group.class, "vradi.email.addGroup", getAdminHandler().getAllGroups());
        }
    }

    protected class UserAddSendingListener extends AddSendingListener {

        public UserAddSendingListener(OfferListUI offerListUI) {
            super(offerListUI);
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            openUI(User.class, "vradi.email.addUser", getAdminHandler().getAllUsers());
        }
    }

    protected class StatusActionListener implements ActionListener {
        protected Status status;
        protected OfferListUI offerListUI;
        
        public StatusActionListener(OfferListUI offerListUI, Status status) {
            this.offerListUI = offerListUI;
            this.status = status;
        }
        
        @Override
        public void actionPerformed(ActionEvent actionEvent) {
            int confirm = JOptionPane.showConfirmDialog(offerListUI,
                    _("vradi.offerList.status.change.confirmation.message", _(status.getName())),
                    _("vradi.offerList.status.change.confirmation.title"),
                    JOptionPane.YES_NO_OPTION);
            
            if (confirm == JOptionPane.YES_OPTION) {
                offerListUI.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
                try {
                    int[] selectedRows = offerListUI.getListTable().getSelectedRows();

                    List<String> formIds = new ArrayList<String>();
                    JXTable listTable = offerListUI.getListTable();
                    OfferListTableModel tableModel = offerListUI.getOfferListTableModel();
                    FormPagedResult formPageDTO = tableModel.getFormPagedResult();
                    List<String> formsIdsToShow = formPageDTO.getFormsIdsToShow();
                    
                    for(int row : selectedRows) {
                        row = listTable.convertRowIndexToModel(row);
                        String formId = formsIdsToShow.get(row);
                        if (formId != null) {
                            formIds.add(formId);
                        }
                    }

                    List<Form> forms = VradiService.getWikittyProxy().restore(Form.class, formIds);
                    changeAllStatus(offerListUI, forms, status);

                } catch (WikittyException eee) {
                    if (log.isErrorEnabled()) {
                        log.error("Can't change offer status", eee);
                    }
                    ErrorDialogUI.showError(eee);
                } finally {
                    offerListUI.setCursor(null);
                }
            }
        }
    }

    protected void changeAllStatus(JAXXContext context, List<Form> forms, Status status) {

        try {

            List<Form> formToSave = new ArrayList<Form>();
            for (Form form : forms) {
                formToSave.add(prepareChangeStatus(context, form, status));
            }

            log.info("before save");
            VradiService.getWikittyProxy().store(formToSave);
            log.info("after save");

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't change status", eee);
            }
        }
    }

    protected void changeStatus(JAXXContext context, String formId, Status status) {

        try {
            Form form = VradiService.getWikittyProxy().restore(Form.class, formId);
            form = prepareChangeStatus(context, form, status);
            VradiService.getWikittyProxy().store(form);

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't change status", eee);
            }
        }
    }

    protected Form prepareChangeStatus(JAXXContext context, Form form, Status status) {
        String initialStatus = form.getStatus();
        String updateStatus = status.getWikittyId();

        if (initialStatus == null || !updateStatus.equals(initialStatus)) {
            VradiUser vradiUser = context.getContextValue(VradiUser.class);
            String statusModifier = (vradiUser != null ? vradiUser.getLogin() : null);

            // add modification info
            ModificationTag formTag = VradiService.getWikittyProxy().cast(form, ModificationTag.class);
            formTag.setLastModified(new Date());
            formTag.setLastModifier(statusModifier);
            formTag.setLastStatusModifier(statusModifier);

            if (log.isDebugEnabled()) {
                log.debug(String.format("Updating: statusModifier=%s on form: %s", statusModifier, form.getWikittyId()));
                log.debug(String.format("New Status: %s, Old Status: %s", initialStatus, updateStatus));
            }
        }

        form.setStatus(updateStatus);
        return form;
    }

    public void addEditPane(JAXXContext context, String formId) {
        Form form = VradiService.getWikittyProxy().restore(Form.class, formId);
        addEditPane(context, form);
    }

    public OfferEditUI addEditPane(JAXXContext context, final Form form) {
        OfferListUI ui = getUI(context);
        
        // Fill free space
        ui.setDividerLocation(0);
        ui.setResizeWeight(0);
        ui.setDividerSize(12);

        // Get tab
        JTabbedPane contentTabbedPane = ui.getEditTabs();

        String formId = form.getWikittyId();

        OfferEditUI editUi = null;
        // Exist ?
        if (contentTabbedPane.indexOfTab(formId) < 0) {

            // Open new
            final OfferEditHandler handler = UIHelper.getHandler(context, OfferEditHandler.class);
            try {
                editUi = handler.initUI(context, form);
            } catch (Exception eee) {
                // Do nothing on fail
                if (log.isErrorEnabled()) {
                    log.error("Can't init UI", eee);
                }
                ErrorDialogUI.showError(eee);
                return null;
            }

            // Do nothing on fail
            if (editUi == null) {
                return null;
            }

            final OfferEditUI editUiFinal = editUi;
            contentTabbedPane.add(formId, editUiFinal);

            final VradiTabHeader panel = new VradiTabHeader();

            if (form.getObjet() != null) {
                String panelLabel = form.getObjet().length() < TABPANEL_LABEL_MAX_CHAR_NB ?
                        form.getObjet() : form.getObjet().substring(0, TABPANEL_LABEL_MAX_CHAR_NB) + "...";
                panel.getLabel().setText(panelLabel);
            }
            panel.setId(formId);

            // Table closeable
            contentTabbedPane.setTabComponentAt(contentTabbedPane.indexOfTab(formId), panel);
            panel.getCloseTab().addMouseListener(new javax.swing.event.MouseInputAdapter() {

                @Override
                public void mouseClicked(MouseEvent e) {
                    closePane(editUiFinal, panel.getId());
                }
            });

            // Detachable
            panel.getDetachTab().addMouseListener(new javax.swing.event.MouseInputAdapter() {

                @Override
                public void mouseClicked(MouseEvent e) {
                    detachPane(e);
                }
            });

            // Get all status
            List<Status> statuses = AdminHandler.getAllStatuses();

            // New modifiable list
            List<Status> orderedStatus = new ArrayList<Status>(statuses);

            // Sort by value
            Collections.sort(orderedStatus, VradiComparators.STATUSES_COMPARATOR);

            // Add all buttons
            JPanel buttonPanel = editUiFinal.getStatusButtonPanel();
            for (final Status status : orderedStatus) {
                JButton statusButton = new JButton(_(status.getName()));
                statusButton.setToolTipText(status.getDescription());
                if(status.getWikittyId().equals(form.getStatus())) {
                    Color newColor = new Color(110,150,200,
                            statusButton.getBackground().getAlpha());
                    statusButton.setBackground(newColor);
                }
                statusButton.addActionListener(new ActionListener(){

                    @Override
                    public void actionPerformed(ActionEvent actionEvent) {
                        closePane(editUiFinal, panel.getId());
                        handler.changeStatusAndSave(editUiFinal, form, status);
                    }
                });
                buttonPanel.add(statusButton);
            }

            // Register to save disposition
            UIHelper.registerComponentToSaveDispositionConfig(editUiFinal);

            // Cancel
            editUiFinal.getAnnuler().addActionListener(new ActionListener() {

                @Override
                public void actionPerformed(ActionEvent e) {
                    closePane(editUiFinal, panel.getId());
                }
            });

            // Update
            /*editUiFinal.getValidate().addActionListener(new ActionListener() {

                @Override
                public void actionPerformed(ActionEvent actionEvent) {
                    handler.save(editUiFinal, data);
                    closePane(editUiFinal, panel.getId());

                    SearchHandler handler = UIHelper.getHandler(editUiFinal, SearchHandler.class);
                    handler.executeQueries(ui);
                }
            });*/

        }
        
        // Select
        contentTabbedPane.setSelectedIndex(contentTabbedPane.indexOfTab(formId));

        return editUi;
    }

    protected VradiTabHeader getTabHeader(MouseEvent e) {
        JButton button = (JButton) e.getSource();
        return (VradiTabHeader) button.getParent();
    }

    public void closeAllTabs() {
        OfferListUI ui = VradiContext.OFFERT_LIST_UI_ENTRY_DEF.getContextValue(VradiContext.get());
        JTabbedPane contentTabbedPane = getContentPane(ui);
        contentTabbedPane.removeAll();
        allEditPanelRemoved(contentTabbedPane, ui);
    }

    protected void closePane(JAXXContext context, String id) {
        
        // Save ui disposition before close pane to keep conf of offerEditHandler
        UIHelper.saveComponentDispositionConfig();

        // Remove listener
        ThesaurusSelectionManager selectionManager = context.getContextValue(ThesaurusSelectionManager.class);

        OfferListUI ui = VradiContext.OFFERT_LIST_UI_ENTRY_DEF.getContextValue(VradiContext.get());
        JTabbedPane contentTabbedPane = getContentPane(ui);
        int index = contentTabbedPane.indexOfTab(id);
        if (index > -1) {
            contentTabbedPane.remove(index);
        }
        allEditPanelRemoved(contentTabbedPane, ui);
    }

    // Detach editUI
    protected void detachPane(MouseEvent e) {
        // Get panel
        VradiTabHeader mytab = getTabHeader(e);
        String id = mytab.getId();
        String name = mytab.getLabel().getText();

        OfferListUI ui = mytab.getParentContainer(OfferListUI.class);
        JTabbedPane contentTabbedPane = getContentPane(ui);
        OfferEditUI content = getOfferEdit(contentTabbedPane, id);

        // Create popups
        final JFrame frame = new JFrame(name);

        // Close
        /*content.getValidate().addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                closeDetachedPane(frame, content);
            }
        });*/

        // Cancel
        content.getAnnuler().addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                closeDetachedPane(frame);
            }
        });

        // Remove default listener
        frame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);

        // Add listener
        frame.addWindowListener(new WindowAdapter() {

            @Override
            public void windowClosed(WindowEvent e) {
                closeDetachedPane(frame);
            }
        });

        frame.setLayout(new BorderLayout());
        frame.add(content, BorderLayout.CENTER);
        frame.pack();
        frame.setVisible(true);

        UIHelper.registerComponentToSaveDispositionConfig(frame);

        // Remove
        allEditPanelRemoved(contentTabbedPane, ui);
    }

    protected OfferEditUI getOfferEdit(JTabbedPane contentTabbedPane, String id) {
        return (OfferEditUI) contentTabbedPane.getComponentAt(contentTabbedPane.indexOfTab(id));
    }

    protected void closeDetachedPane(JFrame f) {
        // Store disposition for this frame
        UIHelper.saveComponentDispositionConfig();

        // Close
        f.dispose();
    }

    public void addEditPane(JAXXContext context, List<Form> datas) {
        for (Form data : datas) {
            addEditPane(context, data.getWikittyId());
        }
    }

    protected void allEditPanelRemoved(JTabbedPane contentTabbedPane, OfferListUI ui) {
        if (contentTabbedPane.getTabCount() == 0) {
            ui.setDividerLocation(Integer.MAX_VALUE);
            ui.setResizeWeight(1);
            ui.setDividerSize(0);
        }
    }

    public OfferListUI getUI(JAXXContext context) {
        if (context instanceof OfferListUI) {
            return (OfferListUI) context;
        }
        OfferListUI ui = VradiContext.OFFERT_LIST_UI_ENTRY_DEF.getContextValue(VradiContext.get());
        return ui;
    }

    protected JTabbedPane getContentPane(JAXXContext context) {
        OfferListUI ui = getUI(context);
        return ui.getEditTabs();
    }

    protected JTabbedPane getContentPane(OfferListUI ui) {
        return ui.getEditTabs();
    }

    protected void selectOffer(JAXXContext context, JTable listTable) {
        OfferListTableModel tableModel = (OfferListTableModel)listTable.getModel();
        int selectedRow = listTable.getSelectedRow();
        
        if (selectedRow >= 0 && selectedRow < tableModel.getRowCount()) {
            selectedRow = listTable.convertRowIndexToModel(selectedRow);
         
            FormPagedResult formPageDTO = tableModel.getFormPagedResult();
            List<String> formsToShow = formPageDTO.getFormsIdsToShow();
            
            String formId = formsToShow.get(selectedRow);
            if (formId != null) {
                if (log.isDebugEnabled()) {
                    Form form = VradiService.getWikittyProxy().restore(Form.class, formId);
                    log.debug("Show Form: " + form.getObjet());
                }
                addEditPane(context, formId);
            }
        }
    }

}
