/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: OfferEditHandler.java 1715 2010-10-27 19:21:28Z tchemit $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.3.2/vradi-swing/src/main/java/com/jurismarches/vradi/ui/offer/OfferEditHandler.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.ui.offer;

import static org.nuiton.i18n.I18n._;

import java.awt.Component;
import java.awt.Desktop;
import java.io.File;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import com.jurismarches.vradi.services.VradiDataService;
import javax.swing.DefaultListModel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.filechooser.FileFilter;

import com.jurismarches.vradi.entities.Group;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.ErrorDialogUI;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.FileUtil;
import org.nuiton.wikitty.BusinessEntity;
import org.nuiton.wikitty.BusinessEntityWikitty;
import org.nuiton.wikitty.FieldType;
import org.nuiton.wikitty.WikittyExtension;
import org.nuiton.wikitty.WikittyObsoleteException;
import org.nuiton.wikitty.WikittyProxy;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.beans.QueryBean;
import com.jurismarches.vradi.entities.FieldTypeEnum;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.ModificationTag;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.entities.VradiUser;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.services.VradiStorageService;
import com.jurismarches.vradi.ui.VradiMainUIHandler;
import com.jurismarches.vradi.ui.helpers.UIHelper;
import com.jurismarches.vradi.ui.offer.editors.CurrencyEditor;
import com.jurismarches.vradi.ui.offer.editors.DateEditor;
import com.jurismarches.vradi.ui.offer.editors.DateTimeEditor;
import com.jurismarches.vradi.ui.offer.editors.EmailEditor;
import com.jurismarches.vradi.ui.offer.editors.NumEditor;
import com.jurismarches.vradi.ui.offer.editors.StringEditor;
import com.jurismarches.vradi.ui.offer.editors.TextEditor;
import com.jurismarches.vradi.ui.offer.editors.UrlEditor;
import com.jurismarches.vradi.ui.offer.editors.VradiEditor;
import com.jurismarches.vradi.ui.offer.thesaurus.ThesaurusHandler;
import com.jurismarches.vradi.ui.offer.thesaurus.ThesaurusUI;
import com.jurismarches.vradi.ui.task.VradiTask;

/**
 * @author letellier
 */
public class OfferEditHandler {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(OfferEditHandler.class);
    private static final String DEFAULT_EDITOR_NAME = "$default";
    public static final String REQUIRED_FIELD_CONTEXT = "required";

    protected VradiStorageService getVradiStorageService() {
        return VradiService.getVradiStorageService();
    }

    protected VradiDataService getDataService() { 
        return VradiService.getVradiDataService();
    }
    
    /**
     * Methode pour initialiser l'ui principale sans l'afficher.
     *
     * @param rootContext le context applicatif
     * @param formId id du formulaire
     * @return l'ui instancie et initialisee mais non visible encore
     */
    public OfferEditUI initUI(JAXXContext rootContext, String formId) {
        Form data = VradiService.getWikittyProxy().restore(Form.class, formId);
        return initUI(rootContext, data);
    }

    /**
     * Methode pour initialiser l'ui principale sans l'afficher.
     *
     * @param rootContext le context applicatif
     * @param data formulaire
     * @return l'ui instancie et initialisee mais non visible encore
     */
    public OfferEditUI initUI(JAXXContext rootContext, Form data) {

        JAXXInitialContext context = new JAXXInitialContext().add(rootContext).add(this).add(data);

        if (log.isDebugEnabled()) {
            // show main ui
            log.debug("Load form '" + data.getObjet() + "' with status : " + data.getStatus());
        }

        UIHelper.getHandler(context, ThesaurusHandler.class);
        OfferEditUI ui = new OfferEditUI(context);

        fillfields(ui, data);

        //init files
        DefaultListModel fileModel = ui.getEmbeddedFileListModel();
        if(data != null && data.getFiles() != null) {
            for(String fileName : data.getFiles()) {
                fileModel.addElement(fileName);
            }
        }

        //init attachments
        DefaultListModel attachmentModel = ui.getAttachmentFileListModel();
        if(data != null && data.getAttachments() != null) {
            for(String attachmentName : data.getAttachments()) {
                attachmentModel.addElement(attachmentName);
            }
        }

        // Init propositions
        try {
            List<Thesaurus> thesaurusToPropose =
                    getDataService().proposeThesaurus(data.getWikittyId());

            if (log.isDebugEnabled()) {
                log.debug("Find " + thesaurusToPropose.size() +
                        " thesaurus to propose for form " + data.getObjet());
            }

            addPropositions(ui, thesaurusToPropose.toArray(
                    new Thesaurus[thesaurusToPropose.size()]));

        } catch(VradiException eee) {
            log.error("Cant get propositions of thesaurus ", eee);
            ErrorDialogUI.showError(eee);
        }

        // Init thesaurus
        ThesaurusHandler thesaurusHandler = UIHelper.getHandler(context, ThesaurusHandler.class);
        thesaurusHandler.initThesaurus(ui);

        return ui;
    }

    public void fillfields(OfferEditUI ui, Form form) {
        Collection<String> extensions = form.getExtensionNames();
        for (String extensionName : extensions) {
            if (!extensionName.equals(ModificationTag.EXT_MODIFICATIONTAG)
                    && !extensionName.equals(Form.EXT_INFOGENE)
                    && !extensionName.equals(Form.EXT_FORM)) {
                
                BusinessEntityWikitty entityWikitty = (BusinessEntityWikitty) form;
                WikittyExtension extension = entityWikitty.getWikitty().getExtension(extensionName);
                
                Collection<String> fieldNames = extension.getFieldNames();
                for (String fieldName : fieldNames) {
                    createField(ui, extension, fieldName, form);
                }
            }
        }
    }

    protected static Map<String, Class<? extends VradiEditor<?, ?>>> editorMapping;

    protected void createField(OfferEditUI ui, WikittyExtension extension, String fieldName, BusinessEntity data) {
        JPanel content = ui.getContent();
        FieldType fieldType = extension.getFieldType(fieldName);
        
        if (!fieldType.isCollection()) {
            VradiEditor editor = getEditor(fieldType, ui);
            
            if (editor != null) {
                String extensionName = extension.getName();
                Object value = data.getField(extensionName, fieldName);
                editor.setFieldName(fieldName);
                
                String fieldDesc = VradiHelper.getFieldTypeDescription(fieldType);
                editor.setFieldDescription(fieldDesc);
                
                editor.setExtensionName(extensionName);
                editor.setExtensionFieldName(fieldName);
                
                editor.setValue(value);
                
                if (fieldType.isNotNull()) {
                    editor.setValidatorContext(REQUIRED_FIELD_CONTEXT);
                }
                content.add(editor);
                content.setVisible(true);
            }
        }
    }

    public OfferEditUI getUI(JAXXContext context) {
        if (context instanceof OfferEditUI) {
            return (OfferEditUI) context;
        }
//        OfferEditUI ui = VradiContext.OFFERT_EDIT_UI_ENTRY_DEF.getContextValue(context);
        return null;
    }

    protected void addPropositions(JAXXContext context, Thesaurus[] props) {
        OfferEditUI ui = getUI(context);
        DefaultListModel propositionListModel = ui.getPropositionListModel();
        propositionListModel.removeAllElements();
        for (Thesaurus p : props) {
            if (log.isDebugEnabled()){
                log.debug("Add proposition element : " + p.getName());
            }
            propositionListModel.addElement(p);
        }
    }

    /**
     * @param context
     * @param form
     */
    protected void save(JAXXContext context, Form form) {
        log.info("Saving Form: " + form.getWikittyId());

        // Get proxy
        WikittyProxy proxy = VradiService.getWikittyProxy();

        OfferEditUI offerEditUI = getUI(context);

        try {

            List<Thesaurus> thesaurusToSave = new ArrayList<Thesaurus>();

            // Find all propositions selected
            for (Object s : offerEditUI.getPropositionList().getSelectedValues()){
                Thesaurus selected = (Thesaurus)s;

                // Save thesaurus
                //selected = ThesaurusDataHelper.restoreThesaurus(selected.getWikittyId());
                selected.addAttachment(form.getWikittyId());

                if (!thesaurusToSave.contains(selected)) {
                    thesaurusToSave.add(selected);
                }
            }

            // Find all thesaurus selected
            for (Component c : offerEditUI.getThesaurus().getComponents()) {
                ThesaurusUI thesaurusUI = (ThesaurusUI) c;
                DefaultListModel model = thesaurusUI.getThesaurusSelectedModel();
                Object[] selected = model.toArray();

                for (Object s : selected) {
                    Thesaurus thesaurus = (Thesaurus) s;

                    // Save thesaurus
                    //thesaurus = ThesaurusDataHelper.restoreThesaurus(thesaurus.getWikittyId());
                    thesaurus.addAttachment(form.getWikittyId());

                    if (!thesaurusToSave.contains(thesaurus)) {
                        thesaurusToSave.add(thesaurus);
                    }
                }
            }
            
            // supprime tous les thesaurus qui ne sont plus selectionné
            List<Thesaurus> actualThesauruses = getDataService().getThesaurusAttachedToForm(form);
            for (Thesaurus actualThesaurus : actualThesauruses) {
                if (!thesaurusToSave.contains(actualThesaurus)) {
                    actualThesaurus.removeAttachment(form.getWikittyId());
                    thesaurusToSave.add(actualThesaurus);
                }
            }

            proxy.store(thesaurusToSave);
        } catch (Exception eee) {
            log.error("Can't save thesaurus : ", eee);
            ErrorDialogUI.showError(eee);
        }

        // Save dynamic content
        JPanel content = offerEditUI.getContent();
        Component[] components = content.getComponents();
        for (Component c : components) {
            VradiEditor editor = (VradiEditor) c;
            updateFormField(editor, form);
        }

        // Save infogene content
        updateFormField(offerEditUI.getObjetEditor(), form);
        updateFormField(offerEditUI.getIdEditor(), form);
        updateFormField(offerEditUI.getSourceTextEditor(), form);
        updateFormField(offerEditUI.getSourceURLEditor(), form);
        updateFormField(offerEditUI.getEntityEditor(), form);
        updateFormField(offerEditUI.getCountryEditor(), form);
        updateFormField(offerEditUI.getDepartmentEditor(), form);
        updateFormField(offerEditUI.getDatePubEditor(), form);
        updateFormField(offerEditUI.getDatePeremeptionEditor(), form);
        updateFormField(offerEditUI.getDescriptionEditor(), form);

        VradiUser vradiUser = context.getContextValue(VradiUser.class);
        // add modification info
        ModificationTag formTag = VradiService.getWikittyProxy().cast(form, ModificationTag.class);
        formTag.setLastModified(new Date());
        formTag.setLastModifier(vradiUser.getLogin());

        try {
            form = proxy.store(form);
        } catch (WikittyObsoleteException eee) {
            if (log.isErrorEnabled()) {
                log.error("Your form is obselete (" + form.getObjet() + ")", eee);
            }
            JOptionPane.showMessageDialog(getUI(context),
                    _("vradi.error.formWikittyObselete", form.getObjet()),
                    _("vradi.error.formWikittyObselete.title"),
                    JOptionPane.ERROR_MESSAGE);
            return;
        }

        try {
            for (File embeddedFile : offerEditUI.getEmbeddedFilesToUpload()) {
                VradiService.getFileService().uploadFormEmbeddedFile(embeddedFile, form.getId());
            }
            for (File attachementFile : offerEditUI.getAttachmentFilesToUpload()) {
                VradiService.getFileService().uploadFormAttachment(attachementFile, form.getId());
            }

        } catch (Exception eee) {
            log.error("Can't upload files for form : " + form.getObjet(), eee);
            ErrorDialogUI.showError(eee);
        }
    }

    void updateFormField(VradiEditor editor, Form form) {
        String extensionName = editor.getExtensionName();
        String extensionFieldName = editor.getExtensionFieldName();
        Object value = editor.getValue();
        if (log.isDebugEnabled()) {
            log.debug("updating: " + extensionName + "." + extensionFieldName + "=" + value);
        }
        form.setField(extensionName, extensionFieldName, value);
    }
    
    protected void update(JAXXContext context) {
        VradiMainUIHandler rootHandler = VradiContext.get().getContextValue(VradiMainUIHandler.class);
        rootHandler.goToHome(context);
    }

    protected void findQueries(JAXXContext context) {
        OfferEditUI ui = getUI(context);
        Form form = ui.getData();
        
        try {
            Map<Group, List<QueryBean>> queries = getDataService()
                    .findQueriesReturningForm(form.getWikittyId());

            RequestFormViewHandler handler = UIHelper.getHandler(context,
                    RequestFormViewHandler.class);
            
            RequestFormViewUI requestFormViewUI = handler.initUI(context, queries);
            requestFormViewUI.setVisible(true);

        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
        }
    }
    
    public void viewInPDF(JAXXContext context) {
        final OfferEditUI ui = getUI(context);

        final Form form = ui.getData();

        VradiTask<String> generatePDF = new VradiTask<String>(context, false) {
            @Override
            public String doAction() throws Exception {
                return VradiService.getVradiStorageService().generatePDF(form.getWikittyId(), true);
            }

            @Override
            public void doWhenDone() throws Exception {
                String fileUri = get();
                if (fileUri == null){
                    JOptionPane.showMessageDialog(ui,
                            _("vradi.offerEdit.cantGeneratePDF"),
                            _("vradi.offerEdit.cantGeneratePDFTitle"),
                            JOptionPane.OK_OPTION);

                    return;
                }

                // file is temporary
                File file = VradiService.getFileService().downloadPDF(fileUri);
                Desktop.getDesktop().open(file);
            }
        };

        generatePDF.execute();
    }

    /**
     * Add new embedded file in current form.
     * 
     * @param context parent ui
     */
    public void addEmbeddedFile(JAXXContext context) {
        if(log.isDebugEnabled()) {
            log.debug("addFile(context)");
        }
        FileFilter imageFileFilter = new FileFilter() {

            private final String[] FILTER_EXTENSIONS =
                new String[] {"jpg", "jpeg", "png", "gif"};

            @Override
            public boolean accept(File file) {
                if(file.isDirectory()) {
                    return true;
                }
                for (String extension : FILTER_EXTENSIONS) {
                    if (file.getName().toLowerCase().endsWith(extension)) {
                        return true;
                    }
                }
                return false;
            }

            @Override
            public String getDescription() {
                return "Images(.jpg, .jpeg, .png, .gif)";
            }
        };
        File file = FileUtil.getFile(getUI(context),
                new FileFilter[]{imageFileFilter});
        if(file != null) {
            getUI(context).getEmbeddedFilesToUpload().add(file);
            DefaultListModel model = getUI(context).getEmbeddedFileListModel();
            model.addElement(file.getName());
            getUI(context).getData().addFiles(file.getName());
        }
    }

    /**
     * Remove one form embedded file.
     * 
     * @param context
     */
    public void removeEmbeddedFile(JAXXContext context) {
        int fileIndex = getUI(context).getEmbeddedFileList().getSelectedIndex();
        if(fileIndex >= 0) {
            DefaultListModel model = getUI(context).getEmbeddedFileListModel();
            String fileName = (String)model.remove(fileIndex);
            getUI(context).getData().removeFiles(fileName);
        }
    }

    /**
     * Add new attachement file.
     * 
     * @param context
     */
    public void addAttachmentFile(JAXXContext context) {
        if(log.isDebugEnabled()) {
            log.debug("addAttachment(context)");
        }
        File attachment = FileUtil.getFile(getUI(context), new FileFilter[0]);
        if(attachment != null) {
            getUI(context).getAttachmentFilesToUpload().add(attachment);
            DefaultListModel model = getUI(context).getAttachmentFileListModel();
            model.addElement(attachment.getName());
            getUI(context).getData().addAttachments(attachment.getName());
        }
    }

    /**
     * Remove attachement.
     * 
     * TODO EC20100609 really remove physical file too on server
     * 
     * @param context parent ui
     */
    public void removeAttachmentFile(JAXXContext context) {
        int attachmentIndex = getUI(context).getAttachmentFileList().getSelectedIndex();
        if (attachmentIndex >= 0) {
            DefaultListModel model = getUI(context).getAttachmentFileListModel();
            String attachmentName = (String)model.remove(attachmentIndex);
            getUI(context).getData().removeAttachments(attachmentName);
        }
    }

    protected synchronized VradiEditor getEditor(FieldType type, JAXXContext context) {
        if (editorMapping == null) {
            editorMapping = new TreeMap<String, Class<? extends VradiEditor<?, ?>>>();
            editorMapping.put(FieldTypeEnum.STRING.name(), StringEditor.class);
            editorMapping.put(FieldTypeEnum.DATE_TIME.name(), DateTimeEditor.class);
            editorMapping.put(FieldTypeEnum.DATE.name(), DateEditor.class);
            editorMapping.put(FieldTypeEnum.EMAIL.name(), EmailEditor.class);
            editorMapping.put(FieldTypeEnum.NUMERIC.name(), NumEditor.class);
            editorMapping.put(FieldTypeEnum.TEXT.name(), TextEditor.class);
            editorMapping.put(FieldTypeEnum.URL.name(), UrlEditor.class);
            editorMapping.put(FieldTypeEnum.CURRENCY.name(), CurrencyEditor.class);
            editorMapping.put(DEFAULT_EDITOR_NAME, StringEditor.class);
        }

        FieldTypeEnum typeEnum = FieldTypeEnum.valueOf(type);
        if (log.isDebugEnabled()) {
            log.debug("typeEnum of type " + type + " = " + typeEnum);
        }
        String editorName = typeEnum == null ? DEFAULT_EDITOR_NAME : typeEnum.name();
        if (log.isDebugEnabled()) {
            log.debug("editor name for type [" + type.getType() + "] : " + editorName);
        }

        Class<? extends VradiEditor<?, ?>> editorClass = editorMapping.get(editorName);

        if (log.isDebugEnabled()) {
            log.debug("editor class to use : " + editorClass);
        }
        try {
            Constructor[] constructors = editorClass.getConstructors();
            for (Constructor constructor : constructors) {
                if (constructor.getParameterTypes() != null
                        && constructor.getParameterTypes().length == 1
                        && constructor.getParameterTypes()[0].equals(JAXXContext.class)) {
                    VradiEditor editor = (VradiEditor) constructor.newInstance(context);
                    editor.init();
                    return editor;
                }
            }
            return null;
        } catch (Exception eee) {
            // should never happens
            throw new IllegalStateException(eee);
        }
    }

    /**
     * Changes the status of the form and save it.
     *
     * @param context the actual context of the ui
     * @param form the form to modify and save
     * @param status the new status of the form
     */
    public void changeStatusAndSave(JAXXContext context, Form form, Status status) {

        try {

            String intialStatus = form.getStatus();
            String updateStatus = status.getWikittyId();

            if (intialStatus == null || !updateStatus.equals(intialStatus)) {

                VradiUser vradiUser = context.getContextValue(VradiUser.class);
                String statusModifier = vradiUser != null ? vradiUser.getLogin() : null;

                // add modification info
                ModificationTag formTag = VradiService.getWikittyProxy().cast(form, ModificationTag.class);
                formTag.setLastModified(new Date());
                formTag.setLastModifier(statusModifier);
                formTag.setLastStatusModifier(statusModifier);
                
                if (log.isDebugEnabled()) {
                    log.debug(String.format("Updating: statusModifier=%s on form: %s", statusModifier, form.getWikittyId()));
                    log.debug(String.format("New Status: %s, Old Status: %s", intialStatus, updateStatus));
                }
            }

            form.setStatus(updateStatus);
            save(context, form);

        } catch (Exception eee) {
            log.error("Cant change status for form '" + (form == null ? "null" : form.getObjet()) + "' with status '" + (status == null ? "null" : _(status.getName())) + "'" , eee);
        }
    }

}
