/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: ThesaurusTreeTableHelper.java 1715 2010-10-27 19:21:28Z tchemit $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.3.2/vradi-swing/src/main/java/com/jurismarches/vradi/ui/admin/helpers/ThesaurusTreeTableHelper.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.ui.admin.helpers;

import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.ui.helpers.ThesaurusDataHelper;
import com.jurismarches.vradi.ui.admin.loadors.RootThesaurusTreeTableNodeLoadors;
import com.jurismarches.vradi.ui.admin.loadors.ThesaurusTreeTableNodeLoadors;
import com.jurismarches.vradi.ui.admin.models.ThesaurusTreeTableModel;
import com.jurismarches.vradi.ui.tree.VradiTreeTableNode;

import jaxx.runtime.swing.nav.treetable.NavTreeTableHelper;
import jaxx.runtime.swing.nav.treetable.NavTreeTableModel;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTreeTable;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyServiceEvent;
import org.nuiton.wikitty.WikittyServiceListener;
import org.nuiton.wikitty.WikittyService.ServiceListenerType;

import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.ui.tree.VradiDataProvider;

import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeExpansionListener;
import javax.swing.event.TreeSelectionListener;
import javax.swing.event.TreeWillExpandListener;
import java.util.*;

/**
 * Navigation tree helpers.
 *
 * @author sletellier
 * @see jaxx.runtime.swing.nav.tree.NavTreeHelper
 */
public class ThesaurusTreeTableHelper extends NavTreeTableHelper<VradiTreeTableNode> implements WikittyServiceListener {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(ThesaurusTreeTableHelper.class);

    protected RootThesaurus rootThesaurus;

    protected List<String> idsLoaded = new ArrayList<String>();

    /**
     * Create helper with first nodes are the root thesaurus
     *
     */
    public ThesaurusTreeTableHelper() {
        this(new VradiDataProvider());
    }

    /**
     * Create helper with first nodes are the root thesaurus in parm (for move)
     *
     * @param rootThesaurus root thesaurus to display
     */
    public ThesaurusTreeTableHelper(RootThesaurus rootThesaurus) {
        this(new VradiDataProvider());
        this.rootThesaurus = rootThesaurus;
    }

    /**
     * Create helper with first nodes are the root thesaurus
     *
     * @param dataProvider vradi data provider
     */
    public ThesaurusTreeTableHelper(VradiDataProvider dataProvider) {
        super();
        init(dataProvider);
    }

    protected void init(VradiDataProvider dataProvider) {
        setDataProvider(dataProvider);

        // register each tree on wikitty service
        VradiService.getWikittyService().addWikittyServiceListener(this, ServiceListenerType.ALL);
    }

    @Override
    public void setUI(JXTreeTable tree,
                      boolean addExpandTreeListener,
                      boolean addOneClickSelectionListener,
                      TreeSelectionListener listener,
                      TreeWillExpandListener willExpandListener) {
        super.setUI(tree,
                addExpandTreeListener,
                addOneClickSelectionListener,
                listener,
                willExpandListener);

        tree.addTreeExpansionListener(new TreeExpansionListener(){
            @Override
            public void treeExpanded(TreeExpansionEvent event) {
                VradiTreeTableNode node = (VradiTreeTableNode) event.getPath().getLastPathComponent();
                registerLoadedIds(node);
            }

            @Override
            public void treeCollapsed(TreeExpansionEvent event) {
                VradiTreeTableNode node = (VradiTreeTableNode) event.getPath().getLastPathComponent();
                Enumeration<VradiTreeTableNode> enumeration = node.children();
                while (enumeration.hasMoreElements()) {
                    VradiTreeTableNode child = enumeration.nextElement();
                    idsLoaded.remove(child.getId());
                }
            }
        });
    }

    public void registerLoadedIds(VradiTreeTableNode node) {
        idsLoaded.add(node.getId());
        Enumeration<VradiTreeTableNode> enumeration = node.children();
        while (enumeration.hasMoreElements()) {
            VradiTreeTableNode child = enumeration.nextElement();
            idsLoaded.add(child.getId());
        }
    }

    /**
     * Try to select thesaurus pass in param
     *
     * @param id of thesaurus to select
     */
    public void tryToSelect(String id) {
        Thesaurus thesaurus = VradiService.getWikittyProxy().restore(Thesaurus.class, id);
        tryToSelect(thesaurus);
    }

    /**
     * Try to select thesaurus pass in param
     *
     * @param thesaurus to select
     */
    public void tryToSelect(Thesaurus thesaurus) {
        // Skip if root
        if (!thesaurus.getWikittyId().equals(getRootNode().getId())) {

            VradiTreeTableNode nodeFound = findThesaurusNode(thesaurus);
            if (nodeFound != null) {
                selectNode(nodeFound);
            }

        }
    }

    /**
     * Try to select thesaurus ids pass in param
     *
     * @param thesaurusIds list of thesaurus ids to select
     */
    public void tryToSelect(List<String> thesaurusIds) {
        WikittyProxy proxy = VradiService.getWikittyProxy();
        tryToSelect(proxy.restore(Thesaurus.class, thesaurusIds));
    }

    /**
     * Try to select thesaurus pass in param
     *
     * @param thesauruses list of thesaurus to select
     */
    public void tryToSelect(Collection<Thesaurus> thesauruses) {
        if (thesauruses == null) {
            // Do nothing
            return;
        }
        List<VradiTreeTableNode> nodes = new ArrayList<VradiTreeTableNode>();

        // Find recursivly
        for (Thesaurus thesaurus : thesauruses) {

            if (thesaurus != null) {

                String id = thesaurus.getWikittyId();

                // Skip if root
                if (!id.equals(getRootNode().getId())) {

                    VradiTreeTableNode toSelect = findThesaurusNode(thesaurus);

                    if (toSelect != null) {
                        nodes.add(toSelect);
                    }
                }
            }
        }

        selectNodes(nodes);
    }

    /**
     * Try to unselect thesaurus pass in param
     *
     * @param id of thesaurus to unselect
     */
    public void tryToUnselect(String id) {
        WikittyProxy proxy = VradiService.getWikittyProxy();
        tryToUnselect(proxy.restore(Thesaurus.class, id));
    }

    /**
     * Try to unselect thesaurus pass in param
     *
     * @param thesaurus to unselect
     */
    public void tryToUnselect(Thesaurus thesaurus) {
        tryToUnselect(thesaurus.getWikittyId());

        // Find recursivly
        VradiTreeTableNode nodeFound = findThesaurusNode(thesaurus);
        if (nodeFound != null) {
            unSelectNode(nodeFound);
        }
    }

    /**
     * Try to unselect thesaurus ids pass in param
     *
     * @param thesaurusIds list of thesaurus ids to select
     */
    public void tryToUnselect(List<String> thesaurusIds) {
        WikittyProxy proxy = VradiService.getWikittyProxy();
        tryToUnselect(proxy.restore(Thesaurus.class, thesaurusIds));
    }

    /**
     * Try to unselect thesaurus pass in param
     *
     * @param thesauruses list of thesaurus to unselect
     */
    public void tryToUnselect(Collection<Thesaurus> thesauruses) {
        if (thesauruses == null) {
            // Do nothing
            return;
        }
        List<VradiTreeTableNode> nodes = new ArrayList<VradiTreeTableNode>();

        // Find recursivly
        for (Thesaurus thesaurus : thesauruses) {

            if (thesaurus != null) {

                String id = thesaurus.getWikittyId();

                // Skip if root
                if (!id.equals(getRootNode().getId())) {

                    VradiTreeTableNode toSelect = findThesaurusNode(thesaurus);

                    if (toSelect != null) {
                        nodes.add(toSelect);
                    }
                }
            }
        }

        unSelectNodes(nodes);
    }

    public VradiTreeTableNode findThesaurusNode(String thesaurusId) {
        WikittyProxy proxy = VradiService.getWikittyProxy();
        return findThesaurusNode(proxy.restore(Thesaurus.class, thesaurusId));
    }

    public VradiTreeTableNode findThesaurusNode(Thesaurus thesaurus) {

        VradiTreeTableNode parentNode = getRootNode();
        VradiTreeTableNode rootNode = getChild(parentNode, thesaurus.getRootThesaurus());

        // If rootNode is rootThesaurus
        if (rootNode != null) {
            parentNode = rootNode;
        }

        WikittyProxy proxy = VradiService.getWikittyProxy();

        // To optimize thesaurus search
        List<Thesaurus> parentList = getParentsPathThesaurus(proxy, thesaurus);
        
        // Find in depth for loading only necessary nodes
        for (Thesaurus parent : parentList) {
            VradiTreeTableNode nodeFound = getChild(parentNode, parent.getWikittyId());
            if (nodeFound != null) {
                parentNode = nodeFound;
            }
        }

        // If not found
        if (!parentNode.getId().equals(thesaurus.getWikittyId())) {
            return null;
        }
        return parentNode;
    }

    protected List<Thesaurus> getParentsPathThesaurus(WikittyProxy proxy, Thesaurus thesaurus) {

        List<Thesaurus> parents = new ArrayList<Thesaurus>();
        if (!ThesaurusDataHelper.isFirstChild(thesaurus)) {
            Thesaurus parent = proxy.restore(Thesaurus.class, thesaurus.getParent());

            if (parent != null) {
                List<Thesaurus> thesauruses = getParentsPathThesaurus(proxy, parent);
                parents.addAll(thesauruses);
            }
        }
        parents.add(thesaurus);

        return parents;
    }

    protected List<String> extractIds(List<Thesaurus> beans){
        List<String> ids = new ArrayList<String>();
        if (beans != null){
            for (Thesaurus bean : beans){
                ids.add(bean.getWikittyId());
            }
        }
        return ids;
    }

    @Override
    public VradiDataProvider getDataProvider() {
        return (VradiDataProvider)dataProvider;
    }

    public NavTreeTableModel createTreeModel() {

        NavTreeTableModel model;
        if (rootThesaurus != null) {
            VradiTreeTableNode root = new VradiTreeTableNode(
                    String.class,
                    "Root node",
                    null,
                    null
            );
            VradiTreeTableNode rootThesaurusNode = new VradiTreeTableNode(
                    RootThesaurus.class,
                    rootThesaurus.getWikittyId(),
                    null,
                    new ThesaurusTreeTableNodeLoadors()
            );

            root.add(rootThesaurusNode);

            model = createModel(root, new ThesaurusTreeTableModel());

            // Populate childs nodes
            rootThesaurusNode.populateChilds(getBridge(), getDataProvider());

            registerLoadedIds(root);
        } else {

            VradiTreeTableNode root = new VradiTreeTableNode(
                    String.class,
                    "Root node",
                    null,
                    new RootThesaurusTreeTableNodeLoadors()
            );

            model = createModel(root, new ThesaurusTreeTableModel());

            // Populate childs nodes
            root.populateChilds(getBridge(), getDataProvider());

            registerLoadedIds(root);
        }

        return model;
    }
    
    public static RootThesaurus getRootThesaurus(VradiTreeTableNode node) {
        if (!node.getInternalClass().equals(RootThesaurus.class)) {
            return null;
        }
        return ThesaurusDataHelper.restoreRootThesaurus(node.getId());
    }

    public static Thesaurus getThesaurus(VradiTreeTableNode node) {
        if (!node.getInternalClass().equals(Thesaurus.class)) {
            return null;
        }
        return ThesaurusDataHelper.restoreThesaurus(node.getId());
    }

    protected void refresh(VradiTreeTableNode node) {
        if (node == null) {
            return;
        }
        getBridge().nodeChanged(node);
    }

    public VradiTreeTableNode createRootThesaurusAndSelect(String rootThesaurusIdToAdd) {
        VradiTreeTableNode newNode = createRootThesaurus(rootThesaurusIdToAdd);
        selectNode(newNode);

        return newNode;
    }

    public VradiTreeTableNode createRootThesaurus(String rootThesaurusIdToAdd) {
        VradiTreeTableNode newNode = getChildLoador(ThesaurusTreeTableNodeLoadors.class).createNode(rootThesaurusIdToAdd, getDataProvider());
        insertNode(getRootNode(), newNode);

        return newNode;
    }

    public VradiTreeTableNode createThesaurusAndSelect(String parentId, String thesaurusIdToAdd) {
        VradiTreeTableNode newNode = createThesaurus(parentId, thesaurusIdToAdd);
        selectNode(newNode);

        return newNode;
    }

    public VradiTreeTableNode createThesaurus(String parentId, String thesaurusIdToAdd) {
        VradiTreeTableNode parentNode = findThesaurusNode(parentId);
        VradiTreeTableNode newNode = getChildLoador(ThesaurusTreeTableNodeLoadors.class).createNode(thesaurusIdToAdd, getDataProvider());
        insertNode(parentNode, newNode);

        return newNode;
    }

    public VradiTreeTableNode createThesaurusChildOfRootAndSelect(String rootThesaurusId, String thesaurusIdToAdd) {
        VradiTreeTableNode newNode = createThesaurusChildOfRoot(rootThesaurusId, thesaurusIdToAdd);
        selectNode(newNode);

        return newNode;
    }

    public VradiTreeTableNode createThesaurusChildOfRoot(String rootThesaurusId, String thesaurusIdToAdd) {
        VradiTreeTableNode rootNode = getChild(getRootNode(), rootThesaurusId);
        VradiTreeTableNode newNode = getChildLoador(ThesaurusTreeTableNodeLoadors.class).createNode(thesaurusIdToAdd, getDataProvider());
        insertNode(rootNode, newNode);

        return newNode;
    }

    @Override
    public void putWikitty(WikittyServiceEvent event) {
        
        // map between id and extensions "name" (not extension ids)
        Map<String, Set<String>> idAndExtensions = event.getIdExtensions();
        for (final String wikittyId : event.getIds()) {

            Set<String> wikittyExtensions = idAndExtensions.get(wikittyId);

            // Thesaurus
            if (wikittyExtensions.contains(Thesaurus.EXT_THESAURUS)) {

                // Restore
                Thesaurus thesaurusConcerned = ThesaurusDataHelper.restoreThesaurus(wikittyId);

                if (thesaurusConcerned != null) {

                    // Search on parent to creation case
                    if (idsLoaded.contains(thesaurusConcerned.getParent()) || idsLoaded.contains(thesaurusConcerned.getWikittyId())) {

                        // Find existing
                        VradiTreeTableNode existingNode = findThesaurusNode(wikittyId);

                        // cas modification
                        if (existingNode != null) {

                            // Parent id
                            VradiTreeTableNode parent = existingNode.getParent();

                            // Move case
                            String newParentId = thesaurusConcerned.getParent();
                            if (parent != null && !parent.getId().equals(newParentId)) {

                                if (log.isDebugEnabled()) {
                                    log.debug("Move case :  " + thesaurusConcerned.getName());
                                }

                                VradiTreeTableNode newParentNode = findThesaurusNode(newParentId);

                                // Delete old
                                removeNode(existingNode);

                                // Verify that node is not already created
                                if (getChild(newParentNode, wikittyId) == null) {

                                    createThesaurus(newParentNode.getId(), wikittyId);
                                }

                            } else {
                                if (log.isDebugEnabled()) {
                                    log.debug("Modification case :  " + thesaurusConcerned.getName());
                                }

                                // Modification case
                                refresh(existingNode);
                            }
                        }

                        // Creation case
                        else {

                            if (log.isDebugEnabled()) {
                                log.debug("Creation case :  " + thesaurusConcerned.getName());
                            }

                            // Verify that node is not already created
                            if (findThesaurusNode(wikittyId) == null) {

                                createThesaurus(thesaurusConcerned.getParent(), wikittyId);
                            }
                        }
                    }
                }
            }

            // Root thesaurus
            if (wikittyExtensions.contains(RootThesaurus.EXT_ROOTTHESAURUS)) {
                        
                // Find existing
                VradiTreeTableNode existingNode = findThesaurusNode(wikittyId);

                // Modification case
                if (existingNode != null) {
                    refresh(existingNode);
                } else {
                    if (log.isDebugEnabled()) {
                        log.debug("Refresh root thesaurus : " + wikittyId);
                    }

                    // Creation
                    VradiTreeTableNode newNode = getChildLoador(RootThesaurusTreeTableNodeLoadors.class).createNode(wikittyId, getDataProvider());
                    insertNode(getRootNode(), newNode);
                }
            }
        }
    }

    @Override
    public void removeWikitty(WikittyServiceEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("Receive wikitty service remove event : " + event);
        }

        for (final String wikittyId : event.getIds()) {

            // if null : not in tree
            if (idsLoaded.contains(wikittyId)) {
                
                if (log.isDebugEnabled()) {
                    log.debug("Removing node " + wikittyId + " from parent");
                }
                VradiTreeTableNode node = findThesaurusNode(wikittyId);

                // Remove from cache
                idsLoaded.remove(wikittyId);

                // If is already deleted
                if (node != null) {

                    removeNode(node);
                }
            }
        }
    }

    @Override
    public void clearWikitty(WikittyServiceEvent event) {
        // should not happen in vradi
    }

    @Override
    public void putExtension(WikittyServiceEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("Receive wikitty service put extension event : " + event);
        }
    }

    @Override
    public void removeExtension(WikittyServiceEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("Receive wikitty service remove extension event : " + event);
        }
    }

    @Override
    public void clearExtension(WikittyServiceEvent event) {
        // should not happen in vradi
    }
}
