package com.jurismarches.vradi.ui.offer.thesaurus.models;

import java.awt.Component;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.JComponent;
import javax.swing.JTree;
import javax.swing.tree.DefaultTreeCellRenderer;

import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.ui.helpers.ThesaurusDataHelper;
import com.jurismarches.vradi.ui.helpers.VradiComparators;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.treetable.AbstractTreeTableModel;
import org.nuiton.i18n.I18n;

import com.jurismarches.vradi.beans.ThesaurusCartography;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.ui.helpers.ToolTipHelper;

/**
 * ThesaurusCartographyModel.
 *
 * @author schorlet
 * @version $Revision: 1252 $ $Date: 2010-09-01 19:24:11 +0200 (mer., 01 sept. 2010) $
 * @since 14 avr. 2010 14:40:00
 */
public class ThesaurusCartographyModel extends AbstractTreeTableModel {
    
    private static final Log log = LogFactory.getLog(ThesaurusCartographyModel.class);
    
    /** chidren nodes indexed by parent node id */
    protected Map<String, List<Thesaurus>> nodeList = new HashMap<String, List<Thesaurus>>();
    /** chidren form indexed by parent node id */
    protected Map<String, List<Form>> formList = new HashMap<String, List<Form>>();
    
    /** top level nodes */
    protected Map<String, List<Thesaurus>> topLevels = new HashMap<String, List<Thesaurus>>();
    
    /** number of matching forms indexed by node */
    protected Map<String, Integer> cartography;
    
    public ThesaurusCartographyModel(ThesaurusCartography thesaurusCartography) {
        super(1L);
        this.cartography = thesaurusCartography.getCartography();
        List<Form> forms = thesaurusCartography.getForms();

        // creates the lists of nodes/forms indexed by parent node id
        Set<String> keySet = cartography.keySet();
        List<Thesaurus> thesauruss = ThesaurusDataHelper.restoreThesaurus(keySet);
        for (Thesaurus node : thesauruss) {
            nodeList.put(node.getWikittyId(), new ArrayList<Thesaurus>());
            formList.put(node.getWikittyId(), new ArrayList<Form>());
            topLevels.put(node.getRootThesaurus(), new ArrayList<Thesaurus>());
        }

        // fill the node lists with children nodes
        for (Thesaurus node : thesauruss) {
            String parent = node.getParent();
            if (nodeList.containsKey(parent)) {
                List<Thesaurus> list = nodeList.get(parent);
                list.add(node);
            }
            if (ThesaurusDataHelper.isFirstChild(node)) {
                // the parent is root
                String rootThesaurusId = node.getRootThesaurus();
                List<Thesaurus> thesauruses = topLevels.get(rootThesaurusId);
                thesauruses.add(node);
                topLevels.put(rootThesaurusId, thesauruses);
            }
        }

        // fill the form lists with forms
        for (Form form : forms) {
            Set<String> thesaurus = form.getThesaurus();
            if (thesaurus == null || thesaurus.isEmpty()) {
                continue;
            }

            for (String id : thesaurus) {
                if (formList.containsKey(id)) {
                    List<Form> list = formList.get(id);
                    list.add(form);
                }
            }
        }

        // sort the node lists
        for (List<Thesaurus> entry : nodeList.values()) {
            if (entry.isEmpty()) {
                continue;
            }
            Collections.sort(entry, VradiComparators.THESAURUS_COMPARATOR);
        }

        // sort the root node lists
        for (List<Thesaurus> entry : topLevels.values()) {
            Collections.sort(entry, VradiComparators.THESAURUS_COMPARATOR);
        }
    }
    
    @Override
    public int getColumnCount() {
        return 3;
    }
    
    @Override
    public Object getValueAt(Object node, int column) {
        if (node instanceof RootThesaurus) {
            RootThesaurus treeNode = (RootThesaurus) node;
            
            if (column == 0) {
                return treeNode.getName();
                
            }
        }
        if (node instanceof Thesaurus) {
            Thesaurus treeNode = (Thesaurus) node;

            if (column == 0) {
                return treeNode;

            } else if (column == 1) {
                return cartography.get(treeNode.getWikittyId());

            } else if (column == 2) {
                Set<String> tags = (Set<String>) treeNode.getField(
                        Thesaurus.EXT_THESAURUS,
                        Thesaurus.FIELD_THESAURUS_TAGS);
                return StringUtils.join(tags, ',');
            }
        }
        return null;
    }

    @Override
    public Object getChild(Object parent, int index) {
        if (parent instanceof Long) {
            List<String> keySet = new ArrayList<String>(topLevels.keySet());
            List<RootThesaurus> rootThesaurusList = ThesaurusDataHelper.restoreRootThesaurus(keySet);

            // Sort
            Collections.sort(rootThesaurusList, VradiComparators.ROOT_THESAURUS_COMPARATOR);
            return rootThesaurusList.get(index);
            
        } else if (parent instanceof RootThesaurus) {
            RootThesaurus treeNode = (RootThesaurus) parent;
            String id = treeNode.getWikittyId();
            List<Thesaurus> list = topLevels.get(id);
            return list.get(index);
        } else if (parent instanceof Thesaurus) {
            Thesaurus treeNode = (Thesaurus) parent;
            String id = treeNode.getWikittyId();
            List<Thesaurus> list = nodeList.get(id);
            return list.get(index);
        }
        return null;
    }

    @Override
    public int getChildCount(Object parent) {
        if (parent instanceof Long) {
            return topLevels.size();
            
        } else if (parent instanceof RootThesaurus) {
            RootThesaurus treeNode = (RootThesaurus) parent;
            String id = treeNode.getWikittyId();
            
            List<Thesaurus> list = topLevels.get(id);
            return list.size();

        } else if (parent instanceof Thesaurus) {
            Thesaurus treeNode = (Thesaurus) parent;
            String id = treeNode.getWikittyId();

            List<Thesaurus> list = nodeList.get(id);
            return list.size();
        }
        return 0;
    }

    @Override
    public int getIndexOfChild(Object parent, Object child) {
        if (parent instanceof Long) {
            List<String> keySet = new ArrayList<String>(topLevels.keySet());
            List<RootThesaurus> rootThesaurusList = ThesaurusDataHelper.restoreRootThesaurus(keySet);

            // Sort
            Collections.sort(rootThesaurusList, VradiComparators.ROOT_THESAURUS_COMPARATOR);
            return rootThesaurusList.indexOf(child);
            
        } else if (parent instanceof RootThesaurus) {
            RootThesaurus treeNode = (RootThesaurus) parent;
            String id = treeNode.getWikittyId();
            List<Thesaurus> list = topLevels.get(id);
            return list.indexOf(child);

        } else if (parent instanceof Thesaurus) {
            Thesaurus treeNode = (Thesaurus) parent;
            String id = treeNode.getWikittyId();
            List<Thesaurus> list = nodeList.get(id);
            return list.indexOf(child);
        }
        return -1;
    }

    @Override
    public String getColumnName(int column) {
        String columnName = null;
        if (column == 0) {
            columnName = I18n._("vradi.thesaurus.name");
        } else if (column == 1) {
            columnName = I18n._("vradi.thesaurus.nbforms");
        } else if (column == 2) {
            columnName = I18n._("vradi.thesaurus.tags");
        }
        return columnName;
    }
    
    public class TreeCellRenderer extends DefaultTreeCellRenderer {
        private static final long serialVersionUID = 1L;
        
        @Override
        public Component getTreeCellRendererComponent(JTree tree, Object value,
                boolean selected, boolean expanded, boolean leaf, int row,
                boolean hasFocus) {

            String toolTip = null;

            if (value instanceof RootThesaurus) {
                RootThesaurus node = (RootThesaurus) value;
                value = node.getName();
            }

            if (value instanceof Thesaurus) {
                Thesaurus node = (Thesaurus) value;
                value = node.getName();

                List<Form> forms = formList.get(node.getWikittyId());
                toolTip = ToolTipHelper.getToolTip(forms);
            }

            JComponent component = (JComponent) super.getTreeCellRendererComponent(
                    tree, value, selected, expanded, leaf, row, hasFocus);
            component.setToolTipText(toolTip);
            
            return component;
        }
    }
}
