/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui.models;

import java.awt.Component;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import javax.swing.JTable;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;

import jaxx.runtime.JAXXObject;
import jaxx.runtime.binding.DefaultJAXXBinding;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.WikittyExtension;

import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.FormImpl;
import com.jurismarches.vradi.entities.Infogene;
import com.jurismarches.vradi.entities.InfogeneImpl;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import com.jurismarches.vradi.ui.helpers.ToolTipHelper;

/**
 * OfferListTableModel is the data model for search results table.
 * Its column model is based on the Infogene fields.
 * 
 * @author letellier
 * @version $Revision$ $Date$
 */
public class OfferListTableModel extends AbstractTableModel {

    static private final Log log = LogFactory.getLog(OfferListTableModel.class);

    private static final long serialVersionUID = 1L;
    
    public static final String PROPERTY_PAGE_TO_SHOW = "pageToShow";
    public static final String PROPERTY_NB_FORMS_PER_PAGE = "nbFormsPerPage";
    /**
     * serves as a unique binding for either PROPERTY_PAGE_TO_SHOW or
     * PROPERTY_NB_FORMS_PER_PAGE or the two.
     */
    public static final String PROPERTY_BINDING_CHANGE = "bindings";
    
    public static final String PROPERTY_TOTAL_FORMS = "totalFoundFormNb";
    public static final String PROPERTY_NB_PAGES = "nbPagesAsText";
    public static final String PROPERTY_LAST_PAGE = "lastPage";
    
    protected final PropertyChangeSupport propertyChangeSupport =
            new PropertyChangeSupport(this);
    protected final List<Column> columns = new ArrayList<Column>();
    
    protected VradiFormPageDTO formPageDTO = new VradiFormPageDTO();

    // Show thesaurus tooltip
    protected boolean showThesaurusToolTip = false;
    
    public OfferListTableModel() {
        this(false);
    }

    public OfferListTableModel(boolean showThesaurusToolTip) {
        initColumns();
        formPageDTO.setPageToShow(1);
        formPageDTO.setNbFormsToShow(10);
        this.showThesaurusToolTip = showThesaurusToolTip;
    }
    
    public OfferListTableModel(VradiFormPageDTO formPageDTO) {
        this(formPageDTO, false);
    }

    public OfferListTableModel(VradiFormPageDTO formPageDTO, boolean showThesaurusToolTip) {
        initColumns();
        formPageDTO.setPageToShow(formPageDTO.getPageToShow());
        formPageDTO.setNbFormsToShow(formPageDTO.getNbFormsToShow());
        setFormPageDTO(formPageDTO);
        this.showThesaurusToolTip = showThesaurusToolTip;
    }

    /**
     * Raises property change event for PROPERTY_TOTAL_FORMS,
     * PROPERTY_NB_PAGES, PROPERTY_LAST_PAGE.
     *
     * Plus, the method <code>AbstractTableModel.fireTableDataChanged()</code> is invoked.
     *
     * @param formPageDTO
     */
    public void setFormPageDTO(VradiFormPageDTO formPageDTO) {
        if (formPageDTO == null) {
            throw new IllegalArgumentException("formPageDTO is null");
        }
        
        this.formPageDTO = formPageDTO;
        
        propertyChangeSupport.firePropertyChange(PROPERTY_TOTAL_FORMS, null,
                formPageDTO.getTotalFoundFormNb());
        
        propertyChangeSupport.firePropertyChange(PROPERTY_NB_PAGES, null,
                getNbPages());
        
        propertyChangeSupport.firePropertyChange(PROPERTY_LAST_PAGE, null,
                isLastPage());
        
        fireTableDataChanged();
    }
    
    public VradiFormPageDTO getFormPageDTO() {
        return formPageDTO;
    }
    
    /**
     * Column.
     */
    static class Column {
        private static final String I18N_COLUMN_PREFIX = "vradi.offer.";
        
        final String i18name;
        final Class columnClass;
        final String fqName;

        Column(String extension, String name, Class columnClass) {
            this.fqName = extension + "." + name;
            this.columnClass = columnClass;
            this.i18name = I18n._(I18N_COLUMN_PREFIX + fqName);
        }
    }
    
    protected void initColumns() {
        // FIXME: retrieve extension from service
        List<String> fieldNames = new ArrayList<String>();
        for(String fieldName : InfogeneImpl.extensionInfogene.getFieldNames()) {
            fieldNames.add(Infogene.EXT_INFOGENE + "." + fieldName);
        }
        fieldNames.add(Form.FQ_FIELD_XMLSTREAMURL);

        for (String fqFieldName : fieldNames) {
            int dot = fqFieldName.lastIndexOf(".");
            String extensionName = fqFieldName.substring(0, dot);
            String fieldName = fqFieldName.substring(dot + 1);
            WikittyExtension extension =
                    extensionName.equals(Infogene.EXT_INFOGENE) ?
                            InfogeneImpl.extensionInfogene :
                            FormImpl.extensionForm;

            FieldType fieldType = extension.getFieldType(fieldName);

            Class columnClass = null;
            if (fieldType.getType() == FieldType.TYPE.BOOLEAN) {
                columnClass = Boolean.class;

            } else if (fieldType.getType() == FieldType.TYPE.DATE) {
                columnClass = Date.class;

            } else if (fieldType.getType() == FieldType.TYPE.NUMERIC) {
                columnClass = Double.class;

            } else {
                columnClass = String.class;
            }

            Column column = new Column(extensionName, fieldName,
                    columnClass);
            columns.add(column);
        }
    }
    
    public Integer getNbFormsPerPage() {
        return formPageDTO.getNbFormsToShow();
    }

    /**
     * Raises property change event for PROPERTY_NB_FORMS_PER_PAGE and
     * PROPERTY_PAGE_TO_SHOW, PROPERTY_BINDING_CHANGE.
     * 
     * As PROPERTY_BINDING_CHANGE property is changed, a new search is executed.
     * 
     * @param nbFormsPerPage
     */
    public void setNbFormsPerPage(Integer nbFormsPerPage) {
        Integer nbFormsToShow = formPageDTO.getNbFormsToShow();
        
        int topRow = nbFormsToShow * (formPageDTO.getPageToShow() - 1);
        topRow = topRow + 1;
        
        Integer pageToShow = (int) Math.ceil(topRow / nbFormsPerPage.doubleValue());
        if (pageToShow < 1) {
            pageToShow = 1;
        }
        
        formPageDTO.setNbFormsToShow(nbFormsPerPage);
        propertyChangeSupport.firePropertyChange(PROPERTY_NB_FORMS_PER_PAGE, nbFormsToShow,
                nbFormsPerPage);
        
        setPageToShow(pageToShow);
    }

    public int getPageToShow() {
        return formPageDTO.getPageToShow();
    }
    
    /**
     * Raises property change event for PROPERTY_PAGE_TO_SHOW, PROPERTY_BINDING_CHANGE.
     * 
     * As PROPERTY_BINDING_CHANGE property is changed, a new search is executed.
     * 
     * @param pageToShow
     */
    public void setPageToShow(int pageToShow) {
        int oldValue = formPageDTO.getPageToShow();
        formPageDTO.setPageToShow(pageToShow);
        
        propertyChangeSupport.firePropertyChange(PROPERTY_PAGE_TO_SHOW, oldValue,
                pageToShow);
        
        propertyChangeSupport.firePropertyChange(PROPERTY_BINDING_CHANGE, Boolean.FALSE,
                Boolean.TRUE);
    }
    
    public boolean isLastPage() {
        int nbPages = getNbPages();
        int pageToShow = formPageDTO.getPageToShow();
        
        boolean result = (nbPages == pageToShow);
        return result;
    }

    public String getNbPagesAsText() {
        int nbPages = getNbPages();
        return " / " + String.valueOf(nbPages);
    }

    public int getNbPages() {
        int totalFoundFormNb = formPageDTO.getTotalFoundFormNb();
        int nbFormsPerPage = formPageDTO.getNbFormsToShow();
        
        int nbPages = (int) Math.ceil(totalFoundFormNb / (double) nbFormsPerPage);
        
        if (nbPages == 0) {
            nbPages = 1;
        }
        
        return nbPages;
    }
    
    public String getFieldToSort() {
        return formPageDTO.getFieldToSort();
    }

    public void setFieldToSort(String fieldToSort) {
        if (fieldToSort != null) {
            formPageDTO.setFieldToSort(fieldToSort);
        }
    }

    public boolean isAscending() {
        return formPageDTO.isAscending();
    }

    public void setAscending(boolean ascending) {
        formPageDTO.setAscending(ascending);
    }

    public int getTotalFoundFormNb() {
        return formPageDTO.getTotalFoundFormNb();
    }

    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        return false;
    }

    @Override
    public String getColumnName(int col) {
        Column column = columns.get(col);
        return column.i18name;
    }

    @Override
    public int getRowCount() {
        return formPageDTO.getFormsToShow() != null ?
                formPageDTO.getFormsToShow().size() :
                0;
    }

    @Override
    public int getColumnCount() {
        return columns.size();
    }

    @Override
    public Object getValueAt(int row, int col) {
        List<? extends Form> formsToShow = formPageDTO.getFormsToShow();
        if (row >= formsToShow.size()) {
            return null;
        }
        
        Form form = formsToShow.get(row);
        if (form == null) {
            return null;
        }

        String fqColumnName = getColumnFqName(col);
        if(fqColumnName.equals(Form.FQ_FIELD_STATUS)) {
            String statusId = form.getStatus();
            Status status = VradiHelper.getStatus(statusId);
            return status;
        }
        int indexOf = fqColumnName.indexOf('.');
        
        String extension = fqColumnName.substring(0, indexOf);
        String fieldName = fqColumnName.substring(indexOf + 1);
        
        Object field = form.getField(extension, fieldName);
        FieldType fieldType = form.getFieldType(extension, fieldName);
        Object result = fieldType.getValidValue(field);
        return result;
    }

    @Override
    public Class getColumnClass(int col) {
        Column column = columns.get(col);
        return column.columnClass;
    }

    public String getColumnFqName(int col) {
        Column column = columns.get(col);
        return column.fqName;
    }

    public String getToolTip(int row) {
        List<? extends Form> formsToShow = formPageDTO.getFormsToShow();
        if (row >= formsToShow.size()) {
            return null;
        }

        Form form = formsToShow.get(row);
        if (showThesaurusToolTip){
            return ToolTipHelper.getThesaurusToolTip(form);
        }
        return ToolTipHelper.getToolTip(form);
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        propertyChangeSupport.addPropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        propertyChangeSupport.removePropertyChangeListener(listener);
    }

    public void addPropertyChangeListener(String propertyName,
                                          PropertyChangeListener listener) {
        propertyChangeSupport.addPropertyChangeListener(propertyName, listener);
    }

    public void removePropertyChangeListener(String propertyName,
                                             PropertyChangeListener listener) {
        propertyChangeSupport.removePropertyChangeListener(propertyName, listener);
    }

    static public abstract class OfferListTableModelBinding
            extends DefaultJAXXBinding {

        protected OfferListTableModel model;

        public OfferListTableModelBinding(JAXXObject source, String id,
                                          OfferListTableModel model) {
            super(source, id, false);
            this.model = model;
        }

        @Override
        public void applyDataBinding() {
            if (model != null) {
                model.addPropertyChangeListener(PROPERTY_BINDING_CHANGE, this);
            }
        }

        @Override
        public void removeDataBinding() {
            if (model != null) {
                model.removePropertyChangeListener(PROPERTY_BINDING_CHANGE, this);
            }
        }
    }

    static public class OfferListTableCellRenderer extends
            DefaultTableCellRenderer {
        private static final long serialVersionUID = 1L;

        @Override
        public Component getTableCellRendererComponent(JTable table,
                Object value, boolean isSelected, boolean hasFocus, int row,
                int column) {
            String stringValue = null;
            if (value == null) {
                stringValue = null;
            } else if (value instanceof Date) {
                stringValue = DateFormat.getDateTimeInstance(DateFormat.SHORT,
                        DateFormat.SHORT, Locale.FRANCE).format((Date) value);
                
            } else if(value instanceof Status) {
                stringValue = ((Status)value).getName();
            } else {
                stringValue = String.valueOf(value);
            }

            OfferListTableModel model = (OfferListTableModel) table.getModel();
            setToolTipText(model.getToolTip(table.convertRowIndexToModel(row)));

            return super.getTableCellRendererComponent(table, stringValue,
                    isSelected, hasFocus, row, column);
        }
    }
}
