/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui.email;

import static com.jurismarches.vradi.VradiContext.EMAIL_UI_ENTRY_DEF;
import static com.jurismarches.vradi.VradiContext.MAIN_UI_ENTRY_DEF;
import static com.jurismarches.vradi.VradiContext.findFormsInEntryDef;
import static com.jurismarches.vradi.VradiContext.findSessionInEntryDef;
import static com.jurismarches.vradi.VradiContext.getClientsEntryDef;
import static com.jurismarches.vradi.VradiContext.getGroupsEntryDef;
import static com.jurismarches.vradi.VradiContext.getSessionsDTOInEntryDef;
import static com.jurismarches.vradi.VradiContext.getSessionsInEntryDef;
import static com.jurismarches.vradi.VradiContext.getUsersEntryDef;
import static com.jurismarches.vradi.VradiContext.setSessionDTOEntryDef;
import static com.jurismarches.vradi.VradiContext.setSessionEntryDef;
import static org.nuiton.i18n.I18n._;

import java.awt.Cursor;
import java.awt.event.MouseListener;
import java.io.File;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;

import javax.swing.JOptionPane;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.ErrorDialogUI;
import jaxx.runtime.swing.navigation.tree.NavigationTreeNode;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.decorator.HighlighterFactory;
import org.sharengo.wikitty.WikittyExtension;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.VradiService;
import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.Infogene;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.services.MailingService;
import com.jurismarches.vradi.services.ServiceFactory;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.VradiStorageService;
import com.jurismarches.vradi.services.dto.VradiClientDTO;
import com.jurismarches.vradi.services.dto.VradiFormDTO;
import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import com.jurismarches.vradi.services.dto.VradiGroupDTO;
import com.jurismarches.vradi.services.dto.VradiSendingDTO;
import com.jurismarches.vradi.services.dto.VradiSessionDTO;
import com.jurismarches.vradi.services.dto.VradiUserDTO;
import com.jurismarches.vradi.ui.AbstractProgressBarUI;
import com.jurismarches.vradi.ui.OfferListColumnFactory;
import com.jurismarches.vradi.ui.OfferListHandler;
import com.jurismarches.vradi.ui.OfferTable;
import com.jurismarches.vradi.ui.VradiComparators;
import com.jurismarches.vradi.ui.VradiMainUI;
import com.jurismarches.vradi.ui.VradiTask;
import com.jurismarches.vradi.ui.helpers.EmailNavigationTreeHelper;
import com.jurismarches.vradi.ui.helpers.UIHelper;
import com.jurismarches.vradi.ui.models.OfferListTableModel;

/**
 * @author morin
 */
public class EmailHandler {

    static private final Log log = LogFactory.getLog(EmailHandler.class);

    public EmailPopupUI init(JAXXContext rootContext, Object... datas) {
        EmailPopupUI ui = EMAIL_UI_ENTRY_DEF
                .getContextValue(rootContext);

        if (ui == null) {
            JAXXInitialContext context = new JAXXInitialContext()
                    .add(rootContext).add(this);

            VradiMainUI mainUI = MAIN_UI_ENTRY_DEF
                    .getContextValue(context);

            ui = new EmailPopupUI(context);

            EMAIL_UI_ENTRY_DEF.setContextValue(rootContext, ui);
        }
        return ui;
    }

    public EmailPopupUI getUI(JAXXContext context) {
        if (context instanceof EmailPopupUI) {
            return (EmailPopupUI) context;
        }
        EmailPopupUI ui = EMAIL_UI_ENTRY_DEF
                .getContextValue(context);
        return ui;
    }

    public SessionsListUI getSessionsListUI(JAXXContext context) {
        if (context instanceof SessionsListUI) {
            return (SessionsListUI) context;
        }
        return context.getContextValue(SessionsListUI.class);
    }

    void close(JAXXContext context) {
        EmailPopupUI ui = getUI(context);
        ui.setVisible(false);
        ui.dispose();
    }

    protected VradiStorageService getVradiStorageService() {
        return VradiService.getVradiStorageService();
    }

    protected MailingService getMailingService() {
        return VradiService.getMailingService();
    }

    public List<VradiSessionDTO> getSessions(Date date) {
        if (log.isDebugEnabled()){
            log.debug("getSessions for date : " + date);
        }
        List<Session> sessions = null;
        try {
            sessions = getVradiStorageService().getSessions(date);
        } catch (VradiException eee) {
            log.error(eee);
            ErrorDialogUI.showError(eee);
        }
        // put in cache
        VradiContext.addAllSessionEntryDef(sessions);
        
        List<VradiSessionDTO> result = fillSessionsDTO(sessions);

        return result;
    }

    protected List<VradiSessionDTO> fillSessionsDTO(List<Session> sessions) {
        List<VradiSessionDTO> sessionDTOs = new ArrayList<VradiSessionDTO>();
        for (Session session : sessions){
            sessionDTOs.add(fillSessionDTO(session));
        }
        return sessionDTOs;
    }

    protected VradiSessionDTO fillSessionDTO(Session session){

        // Load entity
        VradiSessionDTO sessionDTO = new VradiSessionDTO();
        sessionDTO.fromWikitty(session);

        // Load default paragraph
        if (sessionDTO.getParagraph() == null) {
            String paragraph = ServiceFactory.getServiceConfiguration().getSessionDefaultParagraph();
            sessionDTO.setParagraph(paragraph);
            updateSession(sessionDTO);
        }

        // Add all sendings
        Set<String> sendingList = session.getSending();
        if (sendingList != null){

            List<Sending> sendings = null;
            try {
                sendings = getVradiStorageService().getEntities(new ArrayList(sendingList), Sending.class);

                VradiContext.addAllSendingInEntryDef(sendings);
            } catch (VradiException eee) {
                log.error("Cant restore sendings" + eee);
            }

            // Create dtos
            for (Sending sending : sendings){
                VradiSendingDTO sendingDTO = new VradiSendingDTO();
                sendingDTO.fromWikitty(sending);
                sendingDTO.setSessionDTO(sessionDTO);

                // Attach sendings
                sessionDTO.addSending(sendingDTO);
            }

            // Attach to queryMakers
            for (VradiSendingDTO sendingDTO : new ArrayList<VradiSendingDTO>(sessionDTO.getSendingList())){

                fillForms(sendingDTO);

                // Client
                String clientId = sendingDTO.getClient();
                if (clientId != null){
                    Client client = VradiHelper.findEntityInRef(getClientsEntryDef(), clientId);
                    if (log.isDebugEnabled()){
                        log.debug("Found client " +
                                (client == null ? "none" : client.getName()) +
                                " in ref with id " + sendingDTO.getClient());
                    }
                    if (client == null) {
                        // Remove sending
                        sessionDTO.removeSending(sendingDTO);
                        log.warn("User with id " + clientId + " not found !");
                        break;
                    }
                    VradiClientDTO clientDTO = new VradiClientDTO();
                    clientDTO.fromWikitty(client);
                    clientDTO.setCurrentSending(sendingDTO);
                    fillClientDTO(sessionDTO, clientDTO);
                    sendingDTO.setClientDTO(clientDTO);
                }

                // User
                String userId = sendingDTO.getUser();
                if (userId != null){
                    User user = VradiHelper.findEntityInRef(getUsersEntryDef(), userId);
                    if (log.isDebugEnabled()){
                        log.debug("Found user " +
                                (user == null ? "none" : user.getName()) +
                                " in ref with id " + sendingDTO.getUser());
                    }
                    if (user == null) {
                        // Remove sending
                        sessionDTO.removeSending(sendingDTO);
                        log.warn("User with id " + userId + " not found !");
                        break;
                    }
                    VradiUserDTO userDTO = new VradiUserDTO();
                    userDTO.fromWikitty(user);
                    userDTO.setCurrentSending(sendingDTO);

                    // Set client name to display
                    Client client = VradiHelper.findEntityInRef(getClientsEntryDef(), user.getClient());
                    if (client != null){
                        userDTO.setClientName(client.getName());
                    }

                    sendingDTO.setUserDTO(userDTO);

                    // Set reception proof to true if user email is not safe
                    sendingDTO.setReceptionProof(!userDTO.getValidEmail());
                    updateSending(sendingDTO);
                }

                // Group
                String groupId = sendingDTO.getGroup();
                if (groupId != null){
                    Group group = VradiHelper.findEntityInRef(getGroupsEntryDef(), groupId);
                    if (log.isDebugEnabled()){
                        log.debug("Found group " +
                                (group == null ? "none" : group.getName()) +
                                " in ref with id " + sendingDTO.getGroup());
                    }
                    if (group == null) {
                        // Remove sending
                        sessionDTO.removeSending(sendingDTO);
                        log.warn("User with id " + groupId + " not found !");
                        break;
                    }
                    VradiGroupDTO groupDTO = new VradiGroupDTO();
                    groupDTO.fromWikitty(group);
                    groupDTO.setCurrentSending(sendingDTO);
                    fillGroupDTO(sessionDTO, groupDTO);
                    sendingDTO.setGroupDTO(groupDTO);
                }
            }
        }

        return sessionDTO;
    }

    protected void fillForms(VradiSendingDTO sending){
        List<VradiFormDTO> formsDTO = new ArrayList<VradiFormDTO>();

        for (String formId : sending.getForm()){

            // Fill forms
            Form form = null;
            try {
                form = getVradiStorageService().getEntity(formId, Form.class);
            } catch (VradiException eee) {
                log.error("Cant restore forms" + eee);
            }
            VradiFormDTO formDTO = new VradiFormDTO();
            formDTO.fromWikitty(form);

            // put in context
            VradiContext.addFormInEntryDef(form);
            VradiContext.addFormDTOInEntryDef(formDTO);
            formDTO.addSendingDTO(sending);
            formsDTO.add(formDTO);
        }
        sending.setFormDTOs(formsDTO);
    }

    protected void fillGroupDTO(VradiSessionDTO session, VradiGroupDTO group) {

        List<VradiClientDTO> clientsDTO = new ArrayList<VradiClientDTO>();
        for (String c : group.getClient()){
            for (VradiSendingDTO sending : session.getSendingList()){
                if (c.equals(sending.getClient())){
                    Client client = VradiHelper.findEntityInRef(getClientsEntryDef(), c);
                    VradiClientDTO clientDTO = new VradiClientDTO();
                    clientDTO.fromWikitty(client);
                    fillForms(sending);
                    clientDTO.setCurrentSending(sending);
                    clientsDTO.add(clientDTO);

                    fillClientDTO(session, clientDTO);
                }
            }
        }
        group.setClientsDTO(clientsDTO);

        List<VradiUserDTO> usersDTO = new ArrayList<VradiUserDTO>();
        for (String u : group.getUser()){
            for (VradiSendingDTO sending : session.getSendingList()){
                if (u.equals(sending.getUser())){
                    User user = VradiHelper.findEntityInRef(getUsersEntryDef(), u);
                    VradiUserDTO userDTO = new VradiUserDTO();
                    userDTO.fromWikitty(user);
                    fillForms(sending);
                    userDTO.setCurrentSending(sending);

                    // Set client name to display
                    Client client = VradiHelper.findEntityInRef(getClientsEntryDef(), user.getClient());
                    if (client != null){
                        userDTO.setClientName(client.getName());
                    }

                    usersDTO.add(userDTO);
                }
            }
        }
        group.setUsersDTO(usersDTO);
    }

    protected void fillClientDTO(VradiSessionDTO session, VradiClientDTO client) {
        List<VradiUserDTO> usersDTO = new ArrayList<VradiUserDTO>();
        
        List<User> users = getUsersForClient(client);
        for (User user : users){
            for (VradiSendingDTO sending : session.getSendingList()){
                if (user.getWikittyId().equals(sending.getUser())){

                    if (log.isDebugEnabled()) {
                        log.debug("Client " + client.getName() + " have user : " + user.getName());
                    }

                    VradiUserDTO userDTO = new VradiUserDTO();
                    userDTO.fromWikitty(user);
                    fillForms(sending);
                    userDTO.setCurrentSending(sending);

                    // Set client name to display
                    userDTO.setClientName(client.getName());
                    
                    usersDTO.add(userDTO);
                }
            }
        }
        client.setUsersDTO(usersDTO);
    }

    public void bindEmailsWithForms(final EmailPopupUI ui) {

        VradiTask<Void> task = new VradiTask<Void>(ui){
            @Override
            public Void doAction() throws Exception {
                getVradiStorageService().bindForms();
                return null;
            }

            @Override
            public void doWhenDone() throws Exception {
                reloadTree(ui);
            }
        };

        task.execute();
    }

    public List<Form> extractFormsFromSending(VradiSendingDTO sendingDTO){
        List<Form> forms = new ArrayList<Form>();
        for (VradiFormDTO formsDTO : sendingDTO.getFormDTOs()){
            forms.add(findFormsInEntryDef(formsDTO.getWikittyId()));
        }
        if (log.isDebugEnabled()){
            log.debug("display " + forms.size() + " forms");
        }
        return forms;
    }

    public void displaySessions(JAXXContext context, EmailNavigationTreeHelper helper, NavigationTreeNode node) {
        SessionsListUI ui = getSessionsListUI(context);
        
        // Get datas
        List<VradiSendingDTO> sendings = helper.getSendingToDisplay(node);
        if (log.isDebugEnabled()){
            log.debug(sendings.size() + " sendings to display");
        }

        // Remove all sessions ui
        ui.getContent().removeAll();
        for (VradiSendingDTO sending : sendings){

            if (log.isDebugEnabled()){
                log.debug("Sending displayed : " + sending);
            }

            // Get all forms
            List<Form> forms = extractFormsFromSending(sending);

            // Init listTable
            OfferListTableModel offerListTableModel =
                    new OfferListTableModel(true);
            OfferListColumnFactory offerListColumnFactory =
                    new OfferListColumnFactory();

            // Init session ui
            JAXXInitialContext initContext = new JAXXInitialContext()
                    .add(context).add(this).add(sending)
                    .add(offerListTableModel)
                    .add(offerListColumnFactory);

            SessionViewUI viewUI = new SessionViewUI(initContext);
            ui.getContent().add(viewUI);

            // listTable cellRenderer
            final OfferListTableModel.OfferListTableCellRenderer renderer =
                    new OfferListTableModel.OfferListTableCellRenderer();

            OfferTable listTable = viewUI.getListTable();

            OfferListTableModel listTableModel = (OfferListTableModel) listTable.getModel();

            VradiFormPageDTO formPageDTO = new VradiFormPageDTO();
            formPageDTO.setFormsToShow(forms);
            
            listTableModel.setFormPageDTO(formPageDTO);
            listTable.setModel(listTableModel);

            listTable.setDefaultRenderer(Boolean.class, renderer);
            listTable.setDefaultRenderer(Date.class, renderer);
            listTable.setDefaultRenderer(Double.class, renderer);
            listTable.setDefaultRenderer(String.class, renderer);

            // listTable highlighter
            final Highlighter highlighter =
                    HighlighterFactory.createAlternateStriping(
                            
                HighlighterFactory.NOTEPAD, HighlighterFactory.GENERIC_GRAY);
            
            // Add listener to open AO
            final MouseListener offerListTableMouseListener =
                UIHelper.getHandler(context, OfferListHandler.class).getOfferListTableMouseListener();
            listTable.addMouseListener(offerListTableMouseListener);

            listTable.addHighlighter(highlighter);
         }
    }

    public void displayForm(JAXXContext context, EmailNavigationTreeHelper helper, NavigationTreeNode node) {
        VradiSessionDTO session = (VradiSessionDTO) node.getParent().getBean(context);
        context.setContextValue(session);
        context.setContextValue(node.getBean(context));
    }

    public <E extends QueryMaker> void addQueryMaker(
            QueryMakerViewUI<E> queryMakerViewUI,
            Map<E, VradiSendingDTO> beans,
            VradiSessionDTO sessionDTO,
            VradiFormDTO form,
            E selected) {

        int yes = JOptionPane.showConfirmDialog(
                queryMakerViewUI,
                _("vradi.email.confirmAdd", form.getObjet()),
                _("vradi.email.confirmAddTitle"),
                JOptionPane.YES_NO_OPTION);

        if (yes == JOptionPane.YES_OPTION){
            // Prepare creation
            List<Form> formsToBind = new ArrayList<Form>();
            formsToBind.add(form);

            // Get bean in cache
            QueryMaker bean = getRealQueryMaker(selected);

            List<VradiSessionDTO> sessionsDTO = getSessionsDTOInEntryDef();
            List<Session> sessions = getSessionsInEntryDef();
            sessionsDTO.remove(sessionDTO);

            // Get cached session
            Session sessionConcerned = VradiContext
                    .findSessionInEntryDef(sessionDTO.getWikittyId());
            sessions.remove(sessionConcerned);
            try{

                // Create sendings
                List<Sending> sendings = getVradiStorageService()
                        .createSending(sessionConcerned, bean, formsToBind);

                // Store in context
                VradiContext.addAllSendingInEntryDef(sendings);

                // Restore session
                sessionConcerned = getVradiStorageService().getEntity(sessionConcerned.getWikittyId(), Session.class);
            } catch (VradiException eee){
                ErrorDialogUI.showError(eee);
                log.error("Cant add sending : " + eee);
            }

            // Update sessions dto cache
            sessionDTO = fillSessionDTO(sessionConcerned);
            sessionsDTO.add(sessionDTO);
            setSessionDTOEntryDef(sessionsDTO);

            // Update sessions cache
            sessions.add(sessionConcerned);
            setSessionEntryDef(sessions);

            reloadTree(queryMakerViewUI.getParentContainer(EmailPopupUI.class));
        }
    }

    public <E extends QueryMaker> void removeQueryMaker(
            QueryMakerViewUI<E> queryMakerViewUI,
            Map<E, VradiSendingDTO> beans,
            VradiSessionDTO sessionDTO,
            VradiFormDTO formDTO,
            VradiSendingDTO selected) {

        int yes = JOptionPane.showConfirmDialog(
                queryMakerViewUI,
                _("vradi.email.confirmDelete", formDTO.getObjet()),
                _("vradi.email.confirmDeleteTitle"),
                JOptionPane.YES_NO_OPTION);

        if (yes == JOptionPane.YES_OPTION){

            // Get query maker
            E queryMaker = null;
            for (E key : beans.keySet()){
                if (beans.get(key).equals(selected)){
                    queryMaker = key;
                    break;
                }
            }

            // Get in entry def
            Form form = findFormsInEntryDef(formDTO.getWikittyId());

            List<Session> sessions = getSessionsInEntryDef();
            List<VradiSessionDTO> sessionsDTO = getSessionsDTOInEntryDef();
            sessionsDTO.remove(sessionDTO);

            // Reload models was listening
            Session session = findSessionInEntryDef(sessionDTO.getWikittyId());
            sessions.remove(session);
            try{
                List<Sending> sendings = getVradiStorageService().removeAllSending(session, form, getRealQueryMaker(queryMaker));
                VradiContext.addAllSendingInEntryDef(sendings);
            } catch (VradiException eee){
                ErrorDialogUI.showError(eee);
                log.error("Cant add sending : " + eee);
            }

            // Update cache
            sessionDTO = fillSessionDTO(session);
            sessionsDTO.add(sessionDTO);
            sessions.add(session);
            setSessionEntryDef(sessions);
            setSessionDTOEntryDef(sessionsDTO);

            reloadTree(queryMakerViewUI.getParentContainer(EmailPopupUI.class));
        }
    }

    protected void reloadTree(EmailPopupUI ui) {

        // Refresh
        EmailNavigationTreeHelper helper = ui.getHelper();

        // Get selected path
        String path = helper.getSelectedPath(ui);

        if (log.isDebugEnabled()) {
            log.debug("Select path : " + path);
        }

        // Update datas
        ui.updateTree();

        // Select
        if (path != null){
            helper.selectNode(ui, path);
        }
    }

    protected <E extends QueryMaker> QueryMaker getRealQueryMaker(E toConvert){
        // FIXME : Find better method
        String wikittyId = toConvert.getWikittyId();
        QueryMaker converted = VradiHelper.findEntityInRef(getClientsEntryDef(), wikittyId);
        if (converted == null){
            converted = VradiHelper.findEntityInRef(getUsersEntryDef(), wikittyId);
        } if (converted == null){
            converted = VradiHelper.findEntityInRef(getGroupsEntryDef(), wikittyId);
        }
        return converted;

    }

    public List<VradiClientDTO> getAllClients() {
        List<VradiClientDTO> result = new ArrayList<VradiClientDTO>();
        List<Client> allClients = VradiContext.getClientsInEntryDef();
        for (Client c : allClients){
            VradiClientDTO dto = new VradiClientDTO();
            dto.fromWikitty(c);
            result.add(dto);
        }
        return result;
    }

//    public Map<VradiClientDTO, Set<String>> getAllClientsAndUsers() {
//        SortedMap<VradiClientDTO, Set<String>> clientsAndUsers = new TreeMap<VradiClientDTO, Set<String>>(VradiComparators.CLIENT_COMPARATOR);
//
//        // map client id = user ids
//        Map<String, Set<String>> clientsIdAndUsers = new TreeMap<String, Set<String>>();
//
//        List<VradiClientDTO> clients = getAllClients();
//        List<User> users = VradiContext.getUsersInEntryDef();
//
//        for (VradiClientDTO client : clients) {
//            Set<String> ids = new HashSet<String>();
//            clientsAndUsers.put(client, ids);
//            clientsIdAndUsers.put(client.getWikittyId(), ids);
//        }
//
//        for (User user : users) {
//            clientsIdAndUsers.get(user.getClient()).add(user.getWikittyId());
//        }
//
//        return clientsAndUsers;
//    }
    
    public List<User> getUsersForClient(VradiClientDTO client) {
        List<User> result = new ArrayList<User>();

        List<User> allUsers = VradiContext.getUsersInEntryDef();
        for (User user : allUsers) {
            if (user.getClient().equals(client.getWikittyId())) {
                result.add(user);
            }
        }
        return result;
    }

    public List<VradiUserDTO> getAllUsers() {
        List<VradiUserDTO> result = new ArrayList<VradiUserDTO>();
        List<User> allUsers = VradiContext.getUsersInEntryDef();
        for (User u : allUsers){
            VradiUserDTO dto = new VradiUserDTO();
            dto.fromWikitty(u);

            Client client = VradiHelper.findEntityInRef(VradiContext.getClientsEntryDef(), u.getClient());
            if (client != null){
                dto.setClientName(client.getName());
            }

            result.add(dto);
        }
        return result;
    }

    public List<VradiGroupDTO> getAllGroups() {
        List<VradiGroupDTO> result = new ArrayList<VradiGroupDTO>();
        List<Group> allGroups = VradiContext.getGroupsInEntryDef();
        for (Group g : allGroups){
            VradiGroupDTO dto = new VradiGroupDTO();
            dto.fromWikitty(g);
            result.add(dto);
        }
        return result;
    }

    /**
     * Called to send only one simple sending.
     * 
     * @param ui parent ui
     * @param sendingDTO sending to send
     */
    public void sendSending(SessionViewUI ui, final VradiSendingDTO sendingDTO){
        Object[] options = {_("vradi.email.confirmSendSending.optionYes"),
                            _("vradi.email.confirmSendSending.optionNo"),
                            _("vradi.email.confirmSendSending.optionOtherMail")};
        
        int i = JOptionPane.showOptionDialog(ui,
                _("vradi.email.confirmSendSending", getQueryMakerName(sendingDTO)),
                _("vradi.email.confirmSendSendingTitle"),
                JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE,
                null,
                options,
                options[2]);

        String email = null;
        
        // FIXME EC20100510 JOptionPane.CANCEL_OPTION has no sense !
        if (i == JOptionPane.CANCEL_OPTION) {

            // Ask email to send
            email = JOptionPane.showInputDialog(
                ui,
                _("vradi.email.confirmSendSending.askMail"),
                _("vradi.email.confirmSendSending.askMailTitle"),
                JOptionPane.PLAIN_MESSAGE);

            i = JOptionPane.OK_OPTION;
        }
        
        // FIXME EC20100510 JOptionPane.OK_OPTION has no sense !
        if (i == JOptionPane.OK_OPTION) {

            // Get session paragraph
            VradiSessionDTO session = sendingDTO.getSessionDTO();
            final String sessionParagraph = session.getParagraph();

            final String finalEmail = email;
            final EmailPopupUI finalUI = ui.getParentContainer(EmailPopupUI.class);
            VradiTask<Void> sendSendingTask = new VradiTask<Void>(ui, "vradi.email.sendSending"){
                
                @Override
                public Void doAction() throws VradiException {
                    Sending sending = VradiContext.findSendingInEntryDef(sendingDTO.getWikittyId());
                    List<Sending> sendings = VradiContext.getSendingInEntryDef();
                    sendings.remove(sending);
                    try {
                        sending = getMailingService().sendMessage(sending, sessionParagraph, finalEmail);
                    } catch (VradiException eee) {
                        JOptionPane.showMessageDialog(finalUI, _("vradi.email.sendSending.sendError",
                                eee.getMessage()), _("vradi.email.sendSession.error"),
                                JOptionPane.ERROR_MESSAGE);

                        // Set session status to error
                        sendingDTO.setStatus(VradiConstants.SendingStatus.ERROR.getValue());
                        updateSending(sendingDTO);
                        reloadTree(finalUI);
                    }
                    sendings.add(sending);
                    VradiContext.setSendingEntryDef(sendings);
                    return null;
                }

                @Override
                public void doWhenError(Exception eee){
                    sendingDTO.setStatus(VradiConstants.SendingStatus.ERROR.getValue());
                    updateSending(sendingDTO);
                    reloadTree(finalUI);
                }

                @Override
                public void doWhenDone() throws Exception {
                    reloadTree(finalUI);
                }
            };
            sendSendingTask.execute();
        }
    }

    /**
     * Called to send a full session.
     * 
     * Don't call without VradiSessionDTO selection !
     * 
     * @param ui parent ui
     * @return {@code true} if task has been 
     */
    public boolean send(final EmailPopupUI ui) {
        final VradiSessionDTO sessionDTO = ui.getHelper().getActiveSession();

        // ask user for confirmation
        int confirm = JOptionPane.showConfirmDialog(ui,
                _("vradi.email.confirmSendSession"),
                _("vradi.email.confirmSendSessionTitle"),
                JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);
        if (confirm != JOptionPane.YES_OPTION) {
            return false;
        }

        // Check if emails and templates exists
        if (!checkAllSending(ui, sessionDTO)){
            return false;
        }

        final Session session = VradiContext.findSessionInEntryDef(sessionDTO.getWikittyId());

        // Display progess bar with cancel button
        final AbstractProgressBarUI pb = new AbstractProgressBarUI(ui){
            /** serialVersionUID. */
            private static final long serialVersionUID = 5926558779822425487L;

            @Override
            public void cancel() {
                try {
                    List<Session> sessions = VradiContext.getSessionsInEntryDef();
                    sessions.remove(session);
                    sessions.add(getMailingService().stopSentMail(session));
                    VradiContext.setSessionEntryDef(sessions);
                    this.dispose();
                } catch (VradiException eee) {
                    log.info("Cant stop sending emails : ", eee);
                    ErrorDialogUI.showError(eee);
                }
                ui.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
            }
        };

        if (log.isInfoEnabled()) {
            SimpleDateFormat format = new SimpleDateFormat("dd/MM/yyyy");
            log.info("Sending session : " + sessionDTO.getNum() + " - " +
                format.format(sessionDTO.getSessionDate()) + " status : " +
                VradiConstants.SessionStatus.getStatus(sessionDTO.getStatus()).getDescription());
        }

        // Exctract sending and forms
        final List<Sending> sendings = new ArrayList<Sending>();
        final List<Form> forms = new ArrayList<Form>();
        for (VradiSendingDTO sendingDTO : sessionDTO.getSendingList()){
            Sending sending = VradiContext.findSendingInEntryDef(sendingDTO.getWikittyId());
            sendings.add(sending);
            forms.addAll(VradiContext.findAllFormsInEntryDef(sendingDTO.getFormDTOs()));
        }

        VradiTask<Boolean> pdfGenerationAndSendingTask = new VradiTask<Boolean>(ui, pb, "vradi.email.generation"){

            @Override
            public Boolean doAction() throws Exception {
                try {
                    VradiHelper.generatePDF(getVradiStorageService(), forms);
                } catch (Exception eee){
                    if (log.isErrorEnabled()) {
                        log.error(getErrorMsg(), eee);
                    }
                    doWhenError(eee);
                    ErrorDialogUI.showError(eee);
                    return false;
                }
                return true;
            }

            @Override
            public void doWhenDone() throws Exception {

                // If pdf generation is succefuly done
                Boolean result = get();
                if (result != null && result) {

                    if (log.isInfoEnabled()) {
                        log.info("Start sending mails");
                    }

                    VradiTask<Void> sentTask = new VradiTask<Void>(ui, pb, "vradi.email.sendSession"){

                        @Override
                        public Void doAction() throws Exception {
                            List<Session> sessions = VradiContext.getSessionsInEntryDef();
                            sessions.remove(session);
                            try {
                                sessions.add(getMailingService().sendMessages(session, sendings));
                            } catch (VradiException eee) {
                                JOptionPane.showMessageDialog(ui, _("vradi.email.sendSending.sendError",
                                        eee.getMessage()), _("vradi.email.sendSession.error"),
                                        JOptionPane.ERROR_MESSAGE);

                                // Set session status to error
                                updateSessionStatus(sessionDTO, VradiConstants.SessionStatus.ERROR);
                                reloadTree(ui);
                            }
                            VradiContext.setSessionEntryDef(sessions);
                            return null;
                        }

                        @Override
                        public void doWhenDone() throws Exception {
                            reloadTree(ui);
                        }

                        @Override
                        public void doWhenError(Exception eee){
                            updateSessionStatus(sessionDTO, VradiConstants.SessionStatus.ERROR);
                            reloadTree(ui);
                        }
                    };

                    sentTask.execute();
                }
            }

            @Override
            public void doWhenError(Exception eee){
                updateSessionStatus(sessionDTO, VradiConstants.SessionStatus.ERROR);
                reloadTree(ui);
            }
        };
        
        // Executes task
        pdfGenerationAndSendingTask.execute();
        
        return true;
    }

    protected boolean checkAllSending(EmailPopupUI ui, VradiSessionDTO sessionDTO){

        List<String> alreadyAskFormType = new ArrayList<String>();

        for (VradiSendingDTO sending : sessionDTO.getSendingList()){

            /*
             * do not check client email: offers are sent to client's users
             * 
            VradiClientDTO clientDTO = sending.getClientDTO();
            if (clientDTO != null){
                Client client = VradiHelper.findEntityInRef(VradiContext.getClientsEntryDef(), clientDTO.getWikittyId());
                String emailClient = client.getEmail();
                if (emailClient == null || emailClient.isEmpty()){
                    boolean abord = askAbordSending(ui, _("vradi.email.askClientEmailEmpty", client.getName()));
                    if (abord){
                        return false;
                    }

                    // Set to error status
                    sending.setStatus(VradiConstants.SendingStatus.ERROR.getValue());
                    updateSending(sending);

                    break;
                }
            }
            */
            
            // Check users email
            VradiUserDTO userDTO = sending.getUserDTO();
            if (userDTO != null){
                User user = VradiHelper.findEntityInRef(VradiContext.getUsersEntryDef(), userDTO.getWikittyId());
                String emailUser = user.getEmail();
                if (emailUser == null || emailUser.isEmpty()){
                    boolean abord = askAbordSending(ui, _("vradi.email.askUserEmailEmpty", user.getName(), userDTO.getClientName()));
                    if (abord){
                        return false;
                    }

                    // Set to error status
                    sending.setStatus(VradiConstants.SendingStatus.ERROR.getValue());
                    updateSending(sending);

                    break;
                }
            }

            // Check PDF generation
            for (VradiFormDTO formDTO : sending.getFormDTOs()){
                Form form = VradiContext.findFormsInEntryDef(formDTO.getWikittyId());

                String extensionName = null;
                for(String ext : form.getExtensionNames()) {
                    if(!ext.equals(Infogene.EXT_INFOGENE)
                            && !ext.equals(Form.EXT_FORM)) {
                        extensionName = ext;
                        break;
                    }

                }
                VradiStorageService storageService = VradiService.getVradiStorageService();
                WikittyExtension extension = null;
                try {
                    extension = storageService.getFormType(extensionName);
                } catch (VradiException eee) {
                    log.error("Cant get formType for form " + formDTO.getObjet() +
                            " with extention : " + extensionName, eee);
                    ErrorDialogUI.showError(eee);
                }

                String template = VradiHelper.getFormTypeTemplate(extension);
                File file = storageService.getTemplate(extensionName, template);

                if ((file == null || !file.exists()) && !alreadyAskFormType.contains(extension.getName())){
                    boolean abord = askAbordSending(ui, _("vradi.email.askModelEmpty", extension.getName()));
                    if (abord){
                        return false;
                    }
                    alreadyAskFormType.add(extension.getName());
                }
            }
        }
        return true;
    }

    protected boolean askAbordSending(EmailPopupUI ui, String msg) {
        int i = JOptionPane.showConfirmDialog(ui,
                msg,
                _("vradi.email.askAbordSendingTitle"),
                JOptionPane.YES_NO_OPTION);
        
        return i != JOptionPane.YES_OPTION;
    }

    public void updateSessionStatus(VradiSessionDTO sessionDTO, VradiConstants.SessionStatus status) {
        sessionDTO.setStatus(status.getValue());
        sessionDTO.setSessionDate(new Date());
        updateSession(sessionDTO);

    }

    public void updateSession(VradiSessionDTO sessionDTO) {
        Session session = VradiContext.findSessionInEntryDef(sessionDTO.getWikittyId());

        List<Session> sessions = VradiContext.getSessionsInEntryDef();
        sessions.remove(session);

        sessionDTO.toWikitty(session);
        try {
            session = getVradiStorageService().updateEntity(session);
        } catch (VradiException eee) {
            log.error("Cant save session : ", eee);
            ErrorDialogUI.showError(eee);
        }

        sessions.add(session);
        VradiContext.setSessionEntryDef(sessions);
    }

    public void updateSending(VradiSendingDTO sendingDTO) {
        Sending sending = VradiContext.findSendingInEntryDef(sendingDTO.getWikittyId());

        List<Sending> sendings = VradiContext.getSendingInEntryDef();
        sendings.remove(sending);

        sendingDTO.toWikitty(sending);
        try {
            sending = getVradiStorageService().updateEntity(sending);
        } catch (VradiException eee) {
            log.error("Cant save session : ", eee);
            ErrorDialogUI.showError(eee);
        }

        sendings.add(sending);
        VradiContext.setSendingEntryDef(sendings);
    }

    protected String getQueryMakerName(VradiSendingDTO sendingDTO){
        if (sendingDTO.getClientDTO() != null){
            return sendingDTO.getClientDTO().getName();
        } else if (sendingDTO.getUserDTO() != null){
            return sendingDTO.getUserDTO().getName();
        } else if (sendingDTO.getGroupDTO() != null){
            return sendingDTO.getGroupDTO().getName();
        }
        return "";
    }

    /**
     * Close selected session in tree.
     * 
     * Don't call it if no session is selected
     * 
     * @param ui parent ui
     */
    public void closeSession(EmailPopupUI ui) {
        VradiSessionDTO session = ui.getHelper().getActiveSession();
        if (session != null) {

            int i = JOptionPane.showConfirmDialog(
                    ui, _("vradi.email.confimCloseSession"),
                    _("vradi.email.confimCloseSession.titles"),
                    JOptionPane.YES_NO_OPTION);

            if (i == JOptionPane.YES_OPTION){
                session.setStatus(VradiConstants.SessionStatus.CLOSE.getValue());
                updateSession(session);
                reloadTree(ui);
            }
        }
    }

    // Update all sending reception proof recursivly (for clients and groups)
    public void updateReceptionProof(VradiSendingDTO sending, boolean isSelected){
        sending.setReceptionProof(isSelected);
        VradiClientDTO client = sending.getClientDTO();
        if (client != null) {
            for (VradiUserDTO user : client.getUsersDTO()) {
                updateReceptionProof(user.getCurrentSending(), isSelected);
            }
        }
        VradiGroupDTO group = sending.getGroupDTO();
        if (group != null) {
            for (VradiUserDTO user : group.getUsersDTO()) {
                updateReceptionProof(user.getCurrentSending(), isSelected);
            }
            for (VradiClientDTO groupClient : group.getClientsDTO()) {
                updateReceptionProof(groupClient.getCurrentSending(), isSelected);
            }
        }
        updateSending(sending);
    }

    // Update all sending paragraphs recursivly (for clients and groups)
    public void updateParagraph(VradiSendingDTO sending, String text){
        sending.setParagraph(text);
        VradiClientDTO client = sending.getClientDTO();
        if (client != null) {
            for (VradiUserDTO user : client.getUsersDTO()) {
                VradiSendingDTO userSending = user.getCurrentSending();
                String userParagraph = userSending.getParagraph();

                // Update only if paragraph is empty
                if (userParagraph == null || userParagraph.isEmpty()){
                    updateParagraph(userSending, text);
                }
            }
        }
        VradiGroupDTO group = sending.getGroupDTO();
        if (group != null) {
            for (VradiUserDTO user : group.getUsersDTO()) {
                VradiSendingDTO userSending = user.getCurrentSending();
                String userParagraph = userSending.getParagraph();

                // Update only if paragraph is empty
                if (userParagraph == null || userParagraph.isEmpty()){
                    updateParagraph(userSending, text);
                }
            }
            for (VradiClientDTO groupClient : group.getClientsDTO()) {
                VradiSendingDTO clientSending = groupClient.getCurrentSending();
                String clientParagraph = clientSending.getParagraph();

                // Update only if paragraph is empty
                if (clientParagraph == null || clientParagraph.isEmpty()){
                    updateParagraph(clientSending, text);
                }
            }
        }
        updateSending(sending);
    }
}

