/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui;

import static org.nuiton.i18n.I18n._;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.DefaultListModel;
import javax.swing.InputMap;
import javax.swing.JColorChooser;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTree;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreePath;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.swing.ErrorDialogUI;
import jaxx.runtime.swing.navigation.NavigationModel;
import jaxx.runtime.swing.navigation.NavigationNode;
import jaxx.runtime.swing.navigation.tree.NavigationTreeNode;
import jaxx.runtime.swing.navigation.treetable.NavigationTreeTableModel;
import jaxx.runtime.swing.navigation.treetable.NavigationTreeTableNode;

import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTreeTable;
import org.nuiton.i18n.I18n;
import org.sharengo.wikitty.TreeNodeImpl;
import org.sharengo.wikitty.BusinessEntity;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.TreeNode;
import org.sharengo.wikitty.search.Search;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.VradiService;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.VradiStorageService;
import com.jurismarches.vradi.services.dto.VradiQueryBean;
import com.jurismarches.vradi.services.dto.VradiThesaurusDTO;
import com.jurismarches.vradi.ui.admin.AdminHandler;
import com.jurismarches.vradi.ui.admin.ThesaurusEditUI;
import com.jurismarches.vradi.ui.admin.ThesaurusPathChooserUI;
import com.jurismarches.vradi.ui.admin.ThesaurusQueryChangesHandler;
import com.jurismarches.vradi.ui.admin.content.AdminThesaurusUI;
import com.jurismarches.vradi.ui.admin.content.ThesaurusQueryChangesUI;
import com.jurismarches.vradi.ui.helpers.ThesaurusTreeHelper;
import com.jurismarches.vradi.ui.helpers.UIHelper;
import com.jurismarches.vradi.ui.models.ModifThesaurusModel;
import com.jurismarches.vradi.ui.models.ModifThesaurusModel.DeleteModif;
import com.jurismarches.vradi.ui.models.ModifThesaurusModel.ModifThesaurus;
import com.jurismarches.vradi.ui.query.ThesaurusChangesTreeTableModel;
import com.jurismarches.vradi.ui.search.SearchHandler;

/**
 * @author letellier
 */
public class ThesaurusHandler {

    /**
     * to use log facility, just put in your code: log.info(\"...\");
     */
    static private Log log = LogFactory.getLog(ThesaurusHandler.class);

    static public String PREFIX_THESAURUS = "thesaurus";
    static public String PREFIX_EDIT = "edit";
    static public String PREFIX_MOVE = "move";

    private VradiStorageService vradiStorageService = null;
    
    protected VradiStorageService getVradiStorageService() {
        if (vradiStorageService == null) {
            vradiStorageService = VradiService.getVradiStorageService();
        }
        return vradiStorageService;
    }

    /**
     * Display thesaurusUI with rootThesaurus for one form and can be removed with parameter deletable
     *
     * @param context
     * @param rootThesaurus root node
     * @param bean          form associated
     * @param deletable     true if ui can be removed
     * @return thesaurusUI
     */
    public ThesaurusUI initUI(JAXXContext context, OfferEditUI parentUI, VradiThesaurusDTO rootThesaurus, Form bean, boolean deletable) {

        // Init ui
        ThesaurusUI ui = new ThesaurusUI(context, parentUI, rootThesaurus);
        ui.setDeletable(deletable);
        ui.setBean(bean);

        return ui;
    }

    public void initThesaurus(JAXXContext context, OfferEditUI ui) {
        // Remove all thsuaurus
        JPanel thesaurusPanel = ui.getThesaurus();
        thesaurusPanel.removeAll();

        List<String> thesaurusId = VradiHelper.getVradiListThesaurus();

        // Get rootThesaurus
        VradiThesaurusDTO rootThesaurus = VradiContext.getThesaurusInEntryDef();

        // Display root thesaurus UI
        ThesaurusUI rootThesaurusUI = initUI(ui, ui, rootThesaurus, ui.getData(), false);

        thesaurusPanel.add(rootThesaurusUI);

        // If no properties found and if root dont hav children display root thesaurus
        List<VradiThesaurusDTO> thesaurusSaved = new ArrayList<VradiThesaurusDTO>();

        // Find saved thesaurus
        if (thesaurusId != null) {
            for (String id : thesaurusId) {
                VradiThesaurusDTO savedThesaurus = rootThesaurus.findThesaurus(id);
                if (savedThesaurus != null) {
                    thesaurusSaved.add(savedThesaurus);
                }
            }
        }
        JList propositionList = ui.propositionList;

        // Display saved thesaurus
        for (VradiThesaurusDTO saved : thesaurusSaved) {
            ThesaurusUI thesaurusUI = initUI(context, ui, saved, ui.getData(), true);

            thesaurusPanel.add(thesaurusUI);

            // Attache selection listener
            ThesaurusListener thesaurusListener = new ThesaurusListener(ui);
            thesaurusUI.getThesaurus().addTreeSelectionListener(thesaurusListener);
        }

        // Attache selection listener
        JTree thesaurus = rootThesaurusUI.getThesaurus();
        RootThesaurusListener rootThesaurusListener = new RootThesaurusListener(ui);
        thesaurus.addTreeSelectionListener(rootThesaurusListener);

        // Attache proposition selection listener
        propositionList.getSelectionModel().addListSelectionListener(new ThesaurusListener(ui));

        // Update size thesaurus
        ui.updateThesaurusSize();
    }

    /**
     * Find thesaurusUI in context
     *
     * @param context
     * @return thesaurusUI in context
     */
    public ThesaurusUI getUI(JAXXContext context) {
        if (context instanceof ThesaurusUI) {
            return (ThesaurusUI) context;
        }
        return null;
    }

    public VradiThesaurusDTO findThesaurusInRef(String wikityId){
        VradiThesaurusDTO root = VradiContext.getThesaurusInEntryDef();
        return root.findThesaurus(wikityId);
    }

    public VradiThesaurusDTO isNameExisting(VradiThesaurusDTO root, String name){
        return root.isNameExisting(name);
    }

    /**
     * Get thesaurus panel from OfferEditUI
     *
     * @param context
     * @return thesaurus panel
     */
    protected JPanel getThesaurusContainer(JAXXContext context) {
        return getUI(context).getParentContainer(OfferEditUI.class).getThesaurus();
    }

    /**
     * Get OfferEditUI
     *
     * @param context
     * @return offer edit panel
     */
    protected OfferEditUI getEditUI(JAXXContext context) {
        return UIHelper.getHandler(context, OfferEditHandler.class).getUI(context);
    }

    /**
     * Get OfferListUI
     *
     * @param context
     * @return offer list panel
     */
    protected OfferListUI getListUI(JAXXContext context) {
        return UIHelper.getHandler(context, OfferListHandler.class).getUI(context);
    }

    /**
     * Get parent node
     *
     * @param helper
     * @param path
     * @return parent node
     */
    protected NavigationNode<?> getParentNode(ThesaurusTreeHelper helper, TreePath path) {
        if (path == null) {
            return helper.getRootNode();
        } else {
            return (NavigationTreeNode) path.getLastPathComponent();
        }
    }

    /**
     * Add criteria in search panel and save it in properties
     *
     * @param context
     * @param helper
     * @param path
     */
    public void addCriteria(JAXXContext context, ThesaurusTreeHelper helper, TreePath path) {
        OfferListUI ui = getListUI(context);

        // Get selected thesaurus
        NavigationNode<?> parentNode = getParentNode(helper, path);
        VradiThesaurusDTO thesaurus = (VradiThesaurusDTO) parentNode.getBean(context);

        // Adding to offerListUI
        UIHelper.getHandler(context, SearchHandler.class).addAndPersistCriteria(context, thesaurus);

        ui.validate();
    }

    /**
     * Refresh all thesaurus displayed
     *
     * @param context
     */
    public void refreshAllThesaurus(JAXXContext context) {
        List<OfferEditUI> editUIs = VradiContext.OFFERT_EDIT_UI_ENTRY_DEF.getContextValue(VradiContext.get());
        if (log.isDebugEnabled()) {
            log.debug("Refresh ui : " + (editUIs == null ? "is null" : editUIs.size()));
        }
        if (editUIs != null) {
            for (OfferEditUI editUI : editUIs) {
                if (log.isDebugEnabled()) {
                    log.debug("Refresh ui : " + editUI.getData().getObjet());
                }
                initThesaurus(context, editUI);
            }
        }
    }

    /**
     * Refresh all other thesaurus displayed
     *
     * @param context
     * @param ui
     */
    public void refreshOtherThesaurus(JAXXContext context, OfferEditUI ui) {
        List<OfferEditUI> editUIs = VradiContext.OFFERT_EDIT_UI_ENTRY_DEF.getContextValue(VradiContext.get());
        if (log.isDebugEnabled()) {
            log.debug("Refresh ui : " + (editUIs == null ? "is null" : editUIs.size()));
        }
        if (editUIs != null) {
            for (OfferEditUI editUI : editUIs) {
                if (!editUI.equals(ui)){
                    if (log.isDebugEnabled()) {
                        log.debug("Refresh ui : " + editUI.getData().getObjet());
                    }
                    initThesaurus(context, editUI);
                }
            }
        }
    }

    /**
     * Add thesaurus panel not removable in offerEdit and save it in properties
     *
     * @param context
     * @param path
     */
    public void addThesaurusUI(JAXXContext context, TreePath path) {
        if (path != null) {
            // Get selected thesaurus
            NavigationTreeNode parentNode = (NavigationTreeNode) path.getLastPathComponent();

            // Creating new root
            VradiThesaurusDTO thesaurusParent = (VradiThesaurusDTO) parentNode.getBean(context);

            addThesaurusUI(context, thesaurusParent, true, true);
        }
    }

    /**
     * Add thesaurus panel in offerEdit
     *
     * @param context
     * @param root          root thesaurus displayed
     * @param deletable     if ui can be removed
     * @param save          to save in properties
     */
    public void addThesaurusUI(JAXXContext context, VradiThesaurusDTO root,
            boolean deletable, boolean save) {
        
        if (root == null) {
            return;
        }
        
        OfferEditUI ui = getUI(context).getParentContainer(OfferEditUI.class);

        ThesaurusUI thesaurusChildUI = new ThesaurusUI(context, ui, root);

        thesaurusChildUI.setDeletable(deletable);

        getThesaurusContainer(context).add(thesaurusChildUI);

        // Save in properties
        if (save) {
            TreeNodeImpl treeNode = VradiContext.findTreeNodeInEntryDef(root.getWikittyId());
            VradiHelper.addVradiListThesaurus(treeNode);
        }
        ui.updateThesaurusSize();

        // Add listener
        ThesaurusListener listener = new ThesaurusListener(ui);
        ThesaurusTreeHelper helper = thesaurusChildUI.getHelper();
        helper.getTree(helper.context).addTreeSelectionListener(listener);
        helper.registerListener(listener);

        ThesaurusTreeHelper rootHelper = ui.getRootThesaurusHelper();
        List<Object> selectedBeans = rootHelper.getSelectedBeans(rootHelper.context);

        helper.tryToSelect(selectedBeans);

        refreshOtherThesaurus(context, ui);
    }

    /**
     * Add colomn in OfferList result table
     *
     * @param context
     * @param helper
     * @param path
     */
    public void addColumn(JAXXContext context, ThesaurusTreeHelper helper, TreePath path) {
        // Get selected thesaurus
        NavigationNode<?> node = getParentNode(helper, path);

        // Creating new root
        VradiThesaurusDTO thesaurus = (VradiThesaurusDTO) node.getBean(context);

        // Save in properties
        TreeNodeImpl treeNode = VradiContext.findTreeNodeInEntryDef(thesaurus.getWikittyId());
        VradiHelper.addVradiListThesaurus(treeNode);
    }

    /**
     * Remove thesaurus ui and save it in properties
     *
     * @param context to removed
     */
    public void closeThesaurus(JAXXContext context) {

        ThesaurusUI thesaurusUI = getUI(context);
        OfferEditUI ui = thesaurusUI.getParentContainer(OfferEditUI.class);
        getThesaurusContainer(context).remove(thesaurusUI);

        // Save in properties
        VradiThesaurusDTO selected = (VradiThesaurusDTO) thesaurusUI.helper.getRootNode().getBean();
        TreeNodeImpl treeNode = VradiContext.findTreeNodeInEntryDef(selected.getWikittyId());
        VradiHelper.removeVradiListThesaurus(treeNode);

        ui.updateThesaurusSize();

        refreshOtherThesaurus(thesaurusUI, ui);
    }

    /**
     * Answer to save thesaurus modification
     *
     * @param content   to be saved
     * @return if it's done
     */
    public boolean answerToSaveBeforeQuit(AdminThesaurusUI content) {
        if (content != null) {

            log.info("current content  to close "
                    + content.getClass().getSimpleName() + " : " + content
                    .isModified());

            String msgs = content.getModifModel().getAllMsgs();

            if (content.isModified()) {
                int n = JOptionPane.showConfirmDialog(content,
                        _("vradi.admin.saveAnswer", msgs),
                        _("vradi.admin.saveTitle"),
                        JOptionPane.YES_NO_CANCEL_OPTION);
                switch (n) {
                    case JOptionPane.NO_OPTION:
                        // annule les modifications
                        cancelThesaurus(content, content.getHelper(), false);
                        break;
                    case JOptionPane.OK_OPTION:
                        // sauvegarde les modifications
                        saveThesaurus(content, content.getHelper(), false);
                        break;
                    case JOptionPane.CANCEL_OPTION:
                        // annule le changement d'onglet
                        return false;

                }
            }
        }
        return true;
    }

    /**
     * Return helpers selected node
     *
     * @param context
     * @param helper
     * @return selected node
     */
    protected NavigationNode getSelectedNode(JAXXContext context,
                                                 ThesaurusTreeHelper helper) {
        // Get selected thesaurus
        NavigationNode<?> selectedNode = helper.getSelectedNode(context);
        if (selectedNode == null) {
            selectedNode = helper.getRootNode();
        }
        return selectedNode;
    }

    /**
     * Add new thesaurus to selected node
     *
     * @param context
     * @param helper
     * @param modifThesaurusModel    list of modif viewer
     * @return if it's done
     */
    public boolean addChildThesaurusToSelected(JAXXContext context,
            ThesaurusTreeHelper helper, ModifThesaurusModel modifThesaurusModel) {

        NavigationTreeTableNode parentNode = (NavigationTreeTableNode) getSelectedNode(context, helper);
        VradiThesaurusDTO created = addChildThesaurus(context, helper, parentNode, modifThesaurusModel);

        if (created == null){
            return false;
        }
        // Add created item in modif viewer
        modifThesaurusModel.addCreateModif(created);

        return true;
    }

    /**
     * Add new thesaurus to parent node in param
     *
     * @param context
     * @param helper
     * @param parentNode
     * @param modifThesaurusModel
     * @return ThesaurusCreated
     */
    private VradiThesaurusDTO addChildThesaurus(JAXXContext context,
            ThesaurusTreeHelper helper, NavigationTreeTableNode parentNode,
            ModifThesaurusModel modifThesaurusModel) {
        
        // Add new to parent
        VradiThesaurusDTO thesaurusParent = (VradiThesaurusDTO) parentNode
                .getBean(context);

        // Creating child
        TreeNodeImpl newTreeNode = new TreeNodeImpl();

        // Add child to parent thesaurus
        VradiThesaurusDTO newThesaurus = new VradiThesaurusDTO();
        newThesaurus.fromWikitty(newTreeNode);
        newThesaurus.setParentThesaurus(thesaurusParent);

        VradiThesaurusDTO cloned = newThesaurus.clone();

        ThesaurusEditUI edit = new ThesaurusEditUI(context);

        edit.init(helper,
                newThesaurus, cloned, modifThesaurusModel,
                _("vradi.adminThesaurus.creationTitle"), true);

        // If name is not empty
        if (edit.isSave()) {
            String name = newThesaurus.getName();
            if (name != null && !name.isEmpty()) {
                newThesaurus.setParentThesaurus(thesaurusParent);
                thesaurusParent.addChild(newThesaurus);

                // Mark to save
                newThesaurus.setToCreate(true);

                // Add in tree
                if (log.isDebugEnabled()) {
                    log.debug("Adding thesaurus " + newThesaurus.getName()
                            + " to parent " + parentNode);
                }
                NavigationNode<?> newNode = helper.getTreeTableBuilder()
                        .addThesaurus(context, parentNode, newThesaurus);

                // Select new node
                helper.selectNode(context, newNode);

                return newThesaurus;
            }
        }
        return null;
    }

    /**
     * Edit selected thesaurus
     *
     * @param context
     * @param helper
     * @param modifThesaurusModel    list of modif viewer
     * @return if it's done
     */
    public boolean editThesaurusSelected(JAXXContext context,
                                         ThesaurusTreeHelper helper,
                                         ModifThesaurusModel modifThesaurusModel) {
        NavigationNode selectedNode = getSelectedNode(context, helper);

        VradiThesaurusDTO selected = ((VradiThesaurusDTO) selectedNode
                .getBean(context));

        VradiThesaurusDTO cloned = selected.clone();
        if (log.isDebugEnabled()) {
            log.debug("Editing thesaurus " + (selected == null ? " null "
                    : selected.getName()));
        }

        ThesaurusEditUI edit = new ThesaurusEditUI(context);
        edit.init(helper, selected, cloned, modifThesaurusModel, _("vradi.adminThesaurus.editTitle", selected.getName()), false);

        // If name is not empty
        if (edit.isSave()) {

            // Tag to delete
            selected.setToSave(true);

            // Get new seleted (when moved)
            selectedNode = getSelectedNode(edit, helper);

            // Repaint an expend
            helper.repaintNode(context, selectedNode);
            helper.sort(context, selectedNode);
            helper.expendNode(selectedNode);

            // Update list of change
            modifThesaurusModel.confirmAttributeModif();

            return true;
        }
        // Cancel
        modifThesaurusModel.cancelAttributeModif();
        return false;
    }

    /**
     * Delete selected thesaurus
     *
     * @param context
     * @param helper
     * @param modifThesaurusModel    list of modif viewer
     * @return if it's done
     */
    public boolean deleteThesaurusSelected(JAXXContext context,
                                           ThesaurusTreeHelper helper,
                                           ModifThesaurusModel modifThesaurusModel) {

        // Get selected node
        NavigationNode<?> selectedNode = getSelectedNode(context, helper);

        // Get selected thesaurus
        VradiThesaurusDTO toDelete = (VradiThesaurusDTO) selectedNode.getBean(context);

        // Add delete item in modif viewer
        modifThesaurusModel.addDeleteModif(toDelete);

        // Delete
        return deleteThesaurus(context, helper, selectedNode, false);
    }

    /**
     * Delete thesaurus in param
     *
     * @param context
     * @param helper
     * @param nodeToDelete  node to delete
     * @param ask           if confirm dialog box must be displayed
     * @return if it's done
     */
    public boolean deleteThesaurus(JAXXContext context,
                                   ThesaurusTreeHelper helper,
                                   NavigationNode<?> nodeToDelete,
                                   boolean ask) {

        VradiThesaurusDTO selected = (VradiThesaurusDTO) nodeToDelete
                .getBean(context);

        int result;
        if (ask) {
            result = JOptionPane.showConfirmDialog(getUI(context),
                    _("vradi.thesaurus.askdeleteThesaurus") + selected
                            .getRecursifName());
        } else {
            result = JOptionPane.OK_OPTION;
        }

        if (result == JOptionPane.OK_OPTION) {

            selected.setToDelete(true);

            NavigationModel navigationModel = helper.getModel(context);
            navigationModel.removeNodeFromParent(nodeToDelete);

            // Selection root node
            NavigationNode<?> rootNode = helper.getRootNode();
            helper.setSelectedNode(context, rootNode);

            return true;
        }
        return false;
    }

    /**
     * Delete thesaurus in base
     *
     * @param toDelete  thesaurus to delete
     */
    private void deleteTreeNode(VradiThesaurusDTO toDelete) {
        try {
            TreeNodeImpl treeNode = VradiContext.findTreeNodeInEntryDef(toDelete.getWikittyId());
            VradiService.getVradiStorageService().deleteEntity(treeNode);

        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);

        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    /**
     * Cancel editing thesaurus
     *
     * @param context
     * @param helper
     * @param trigerredByCancelButton   source of cancel
     */
    public void cancelThesaurus(JAXXContext context, ThesaurusTreeHelper helper,
                                boolean trigerredByCancelButton) {
        int confirm = -1;
        if(trigerredByCancelButton) {
            confirm = JOptionPane.showConfirmDialog((JComponent)context,
                    _("vradi.admin.cancel.message"),
                    _("vradi.admin.cancel.title"),
                    JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);
        }

        if(!trigerredByCancelButton || confirm == JOptionPane.YES_OPTION) {
            // Reinit thesaurus ui
            helper = new ThesaurusTreeHelper(context, PREFIX_EDIT, ListSelectionModel.SINGLE_SELECTION);
            AdminThesaurusUI ui = (AdminThesaurusUI) context;
            ui.reinit(helper);
        }
    }

    public List<TreeNodeImpl> getAllThesaurus(VradiThesaurusDTO node) {
        List<TreeNodeImpl> result = new ArrayList<TreeNodeImpl>();
        for (VradiThesaurusDTO child : node.getChildren()) {
            TreeNodeImpl treeNode = VradiContext.findTreeNodeInEntryDef(child.getWikittyId());
            result.add(treeNode);
            result.addAll(getAllThesaurus(child));
        }
        return result;
    }

    /**
     * Save edited thesaurus
     *
     * @param helper
     * @param answer
     */
    public void saveThesaurus(AdminThesaurusUI adminThesaurusUI,
            ThesaurusTreeHelper helper, boolean answer) {
        ModifThesaurusModel modifModel = adminThesaurusUI.getModifModel();
        
        if (answer) {
            String msgs = modifModel.getAllMsgs();

            JLabel labelModifications = new JLabel(_("vradi.admin.saveThesaurusAnswer"));
            JTextArea areaModifications = new JTextArea(msgs);
            areaModifications.setEditable(false);
            areaModifications.setAutoscrolls(true);
            JScrollPane sp = new JScrollPane(areaModifications);
            sp.setPreferredSize(new Dimension(600, 100)); // don't remove popup is huge
            
            int n = JOptionPane.showConfirmDialog(adminThesaurusUI,
                    new Object[]{labelModifications, sp},
                    _("vradi.admin.saveTitle"), JOptionPane.YES_NO_OPTION);

            if (n == JOptionPane.NO_OPTION) {
                return;
            }
        }
        
        // prepare data for clients queries modifications
        // key:before, value:after
        Map<String, String> modifications = new HashMap<String, String>();
        List<ModifThesaurus> modifThesaurusList = modifModel.getValues();

        for (ModifThesaurus modifThesaurus : modifThesaurusList) {
            if (modifThesaurus.canImpactRequest()) {
                
                VradiThesaurusDTO after = modifThesaurus.getConcernedThesaurus();
                VradiThesaurusDTO before = findThesaurusInRef(after.getWikittyId());

                if (before != null) {
                    if (modifThesaurus instanceof DeleteModif) {
                        // before has been deleted, so the replacement would be the parent node
                        while (after.getDepth() > 1 && after.isToDelete()) {
                            after = after.getParentThesaurus();
                        }
                     
                        // if no convenient node found then re-select the original one
                        if (after.isRoot() || after.isToDelete()) {
                            after = modifThesaurus.getConcernedThesaurus();
                        }
                    }
                
                    modifications.put(before.getName(), after.getName());
                }
            }
        }
        
        try {
            // save thesaurus
            VradiThesaurusDTO rootThesaurus = helper.getRootThesaurus();
            saveThesaurus(rootThesaurus);
        
            // initThesaurusDef
            VradiContext.initThesaurusDef();
        
            adminThesaurusUI.getModifModel().clear();
            adminThesaurusUI.setModified(false);
            
            // refresh criteria
            UIHelper.getHandler(adminThesaurusUI, SearchHandler.class).initCriteria(adminThesaurusUI);

            // Clear cached models
            clearThesaurusModel();

            // refresh opened thesaurusUI
            refreshAllThesaurus(adminThesaurusUI);
            
            // modify clients queries
            for (Map.Entry<String, String> entry : modifications.entrySet()) {
                String before = entry.getKey();
                String after = entry.getValue();
                modifyQueries(adminThesaurusUI, before, after);
            }
            
        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    protected void clearThesaurusModel(){
        Map<String, NavigationModel> models = VradiContext.THESAURUS_NAVIGATION_MODEL.getContextValue(VradiContext.get());
        if (models != null) {
            models.clear();
            VradiContext.THESAURUS_NAVIGATION_MODEL.setContextValue(VradiContext.get(), models);
        }
    }

    protected void modifyQueries(JAXXContext context, String before, String after) {
        Map<QueryMaker, List<VradiQueryBean>> queryMakers = VradiService
                .getVradiStorageService()
                .getQueriesToModifyAfterThesaurusModification(before);

        if (queryMakers.isEmpty()) {
            // the modified thesaurus is not part of a client query
            return;
        }
        
        // show window with modified queries according thesaurus nodes changes
        ThesaurusQueryChangesHandler handler = new ThesaurusQueryChangesHandler();
        ThesaurusQueryChangesUI thesaurusChangesUI = handler.initUI(
                context, queryMakers, before, after);
        thesaurusChangesUI.setModifiedQueries(Boolean.FALSE);
        thesaurusChangesUI.setVisible(true);

        if (!thesaurusChangesUI.isModifiedQueries()) {
            return;
        }
        
        try {
            // retrieve updated queryMakers
            ThesaurusChangesTreeTableModel model = thesaurusChangesUI
                    .getContextValue(ThesaurusChangesTreeTableModel.class);
            List<QueryMaker> updatedQueryMakers = model.getUpdatedQueryMakers();
            
            QueryMaker[] queryMakers2 = new QueryMaker[updatedQueryMakers.size()];
            updatedQueryMakers.toArray(queryMakers2);
            
            // save updated queryMakers
            BusinessEntity[] updateEntities = VradiService
                    .getVradiStorageService().updateEntities(queryMakers2);
            
            // update queryMakers into cache
            for (BusinessEntity businessEntity : updateEntities) {
                VradiContext.replace(businessEntity);
            }
            
        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }
    
    /**
     * Save thesaurus in param
     *
     * @param toSave
     */
    protected void saveThesaurus(VradiThesaurusDTO toSave) {
        TreeNodeImpl bean = VradiContext.findTreeNodeInEntryDef(toSave.getWikittyId());
        if (bean == null) {
            bean = new TreeNodeImpl();
            toSave.toWikitty(bean);
        }
        // To delete
        if (toSave.isToDelete()) {
            // Delete
            deleteTreeNode(toSave);
            if (log.isDebugEnabled()) {
                log.debug("Deleting thesaurus : " + bean.getName());
            }
        }
        // To save
        if (toSave.isToSave() || toSave.isToCreate()) {
            // Save
            try {
                // Set parent to child
                VradiThesaurusDTO parentThesaurus = toSave.getParentThesaurus();

                // If it's not root
                if (parentThesaurus != null) {
                    bean.setParent(parentThesaurus.getWikittyId());
                }

                // Update
                toSave.toWikitty(bean);
                VradiService.getVradiStorageService().updateEntity(bean);
                
                if (log.isDebugEnabled()) {
                    log.debug("Saving thesaurus : " + bean.getName());
                }                                                                                                                        

            } catch (VradiException ex) {
                log.error(ex.getMessage(), ex);
                //ErrorDialogUI.showError(ex);
            }
            // Its save
            toSave.setToSave(false);
            toSave.setToCreate(false);
        }

        Collection<VradiThesaurusDTO> children = toSave.getChildren();
        if (children != null) {
            for (VradiThesaurusDTO child : children) {
                if (child != null) {
                    saveThesaurus(child);
                }
            }
        }

    }

    /**
     * Change color
     *
     * @param context
     * @param helper    helpers of adminThesaurus
     */
    public void changeThesaurusColor(JAXXContext context,
                              ThesaurusTreeHelper helper){
        VradiThesaurusDTO selected = (VradiThesaurusDTO)getSelectedNode(context, helper).getBean();

        int depth = selected.getDepth();

        Color initialColor = VradiHelper.getColorsThesaurusFromProperties().get(depth);
        Color colorChoosed = JColorChooser.showDialog((AdminThesaurusUI) context, _("vradi.thesaurus.colorTitle"), initialColor);

        // Save in properties
        if (log.isDebugEnabled()){
            log.debug("Store color " + colorChoosed + " for depth " + depth);
        }
        VradiHelper.storeThesaurusColorToProperties(depth, colorChoosed);

        helper.reinitColors();
        reinitColors();
    }

    public void reinitColors(){
        List<OfferEditUI> editUIs = VradiContext.OFFERT_EDIT_UI_ENTRY_DEF.getContextValue(VradiContext.get());
        if (editUIs != null){
            for (OfferEditUI ui : editUIs) {
                List<ThesaurusTreeHelper> treeHelperList = ui.getListThesaurusHelpers();

                if (treeHelperList != null){
                    for (ThesaurusTreeHelper helper : treeHelperList){
                        helper.reinitColors();
                    }
                }
            }
        }
    }

    /**
     * Move selected thesaurus
     *
     * @param context
     * @param helperFrom    helpers of adminThesaurus
     * @param modifThesaurusModel modif model
     * @return is moved
     */
    public boolean moveThesaurus(JAXXContext context,
                              ThesaurusTreeHelper helperFrom,
                              ModifThesaurusModel modifThesaurusModel) {

        // Get thesaurus to move
        VradiThesaurusDTO toMove = (VradiThesaurusDTO) helperFrom.getSelectedBean(context);

        // Move
        NavigationTreeTableNode navigationTreeNode = moveThesaurus(context, helperFrom, toMove);

        if (navigationTreeNode == null){
            return false;
        }

        applyMoveThesaurus(context, helperFrom, toMove, navigationTreeNode, modifThesaurusModel);

        return true;
    }

    /**
     * Move selected thesaurus
     *
     * @param context
     * @param helperFrom    helpers of adminThesaurus
     * @param beanToMove    bean to move
     * @return parent node selected
     */
    public NavigationTreeTableNode moveThesaurus(JAXXContext context,
                              ThesaurusTreeHelper helperFrom,
                              VradiThesaurusDTO beanToMove) {


        // Get root thesaurus
        VradiThesaurusDTO rootThesaurus = helperFrom.getRootThesaurus().clone();

        // Remove own branch
        boolean isDeleted = rootThesaurus.removeThesaurusRecursivly(beanToMove);
        if (log.isDebugEnabled()) {
            if (isDeleted) {
                log.debug("Thesaurus " + beanToMove.getName() + " is deleted");
            } else {
                log.debug("Thesaurus " + beanToMove.getName()
                        + " is not deleted (not found)");
            }
        }

        // Create new helpers
        ThesaurusTreeHelper selectHelper = new ThesaurusTreeHelper(context,
                rootThesaurus, PREFIX_MOVE, ListSelectionModel.SINGLE_SELECTION);

        // Create select ui
        ThesaurusPathChooserUI selectUI = new ThesaurusPathChooserUI(context,
                selectHelper);

        selectUI.setVisible(true);

        // If select
        if (selectUI.isSaved()) {
            // Get parent selected
            NavigationTreeTableNode selectedNode =
                    (NavigationTreeTableNode) getSelectedNode(selectUI,
                    selectHelper);

            return selectedNode;
        }
        return null;
    }

    /**
     * Move selected thesaurus
     *
     * @param context
     * @param helperFrom    helpers of adminThesaurus
     * @param beanToMove    bean to move
     * @param nodeParentSelected
     * @param modifThesaurusModel modif model
     */
    public void applyMoveThesaurus(JAXXContext context,
                              ThesaurusTreeHelper helperFrom,
                              VradiThesaurusDTO beanToMove,
                              NavigationTreeTableNode nodeParentSelected,
                              ModifThesaurusModel modifThesaurusModel) {

        // Get node to move
        NavigationNode<?> nodeToMove = getSelectedNode(context, helperFrom);

        // Get corresponding parent selected node
        VradiThesaurusDTO beanParentSelected = (VradiThesaurusDTO)nodeParentSelected.getBean();
        String parentSelectedPath = beanParentSelected.getPath(PREFIX_EDIT, helperFrom.getPathSeparator());
        if (log.isDebugEnabled()){
            log.debug("Searching node whith path : " + parentSelectedPath);
        }

        // Add modif item in modif viewer
        // Get old path
        String parentPath = beanToMove.getParentThesaurus()
                .getPath(PREFIX_EDIT, helperFrom.getPathSeparator());

        // Get old name path
        String parentNamePath = beanToMove.getParentThesaurus()
                .getNamePath(helperFrom.getPathSeparator());
        
        modifThesaurusModel.addMoveModif(beanToMove,
                parentNamePath,
                parentPath,
                parentSelectedPath);

        NavigationTreeTableNode nodeParentSelectedFrom = (NavigationTreeTableNode) helperFrom
                .findNode(context, parentSelectedPath);
        
        if (log.isDebugEnabled()) {
            log.debug("Selected node : " + nodeParentSelected.getFullPath()
                    + " finded node : " + nodeParentSelectedFrom
                    .getFullPath());
        }

        // Delete in tree
        deleteThesaurus(context, helperFrom, nodeToMove, false);
        beanToMove.setToDelete(false);

        // Remove from parent
        VradiThesaurusDTO oldParent = beanToMove.getParentThesaurus();
        oldParent.removeChild(beanToMove);

        // Get corresponding parent selected bean
        VradiThesaurusDTO thesaurusParentFrom
                = (VradiThesaurusDTO) nodeParentSelectedFrom
                .getBean(context);

        // Attach child
        thesaurusParentFrom.addChild(beanToMove);
        beanToMove.setParentThesaurus(thesaurusParentFrom);

        // Mark to save
        beanToMove.setToSave(true);

        // Add in tree
        NavigationTreeTableNode newNode = helperFrom.getTreeTableBuilder()
                .addThesaurusAndChildrenRecursivly(nodeParentSelectedFrom, beanToMove);

        // Expend
        helperFrom.selectNode(context, nodeParentSelectedFrom);

        // FIXME : Binding dont work
        if (context instanceof ThesaurusEditUI){
            ((ThesaurusEditUI) context).getPathField()
                .setText(beanToMove.getParentPath(helperFrom.getPathSeparator()));
        }
    }

    public void viewApply(JAXXContext context, List<VradiThesaurusDTO> concerned){
        UIHelper.getHandler(context, AdminHandler.class).openRequest(context, concerned);
    }

    public class RootThesaurusListener implements TreeSelectionListener {
        protected JList propositionList;
        protected OfferEditUI ui;

        public RootThesaurusListener(OfferEditUI ui){
            propositionList = ui.propositionList;
            this.ui = ui;
        }

        @Override
        public void valueChanged(TreeSelectionEvent e) {
            TreePath[] paths = e.getPaths();

            List<Object> toAdd = new ArrayList<Object>();
            List<Object> toRemove = new ArrayList<Object>();
            for(TreePath path : paths){
                NavigationTreeNode node = (NavigationTreeNode) path.getLastPathComponent();
                if (e.isAddedPath(path)){
                    toAdd.add(node.getBean());
                } else {
                    toRemove.add(node.getBean());
                }
            }

            List<ThesaurusTreeHelper> helpers = ui.getListThesaurusHelpers();
            for (ThesaurusTreeHelper helper : helpers){
                helper.desactiveListener();
                helper.tryToSelect(toAdd);
                helper.tryToUnselect(toRemove);
                helper.activeListener();
            }

            // Try to select propositions
            ThesaurusTreeHelper rootHelper = ui.getRootThesaurusHelper();
            List<Object> beans = rootHelper.getSelectedBeans(rootHelper.context);

            DefaultListModel propositionModel = (DefaultListModel) propositionList.getModel();
            if (beans != null){
                for (Object bean : new ArrayList(beans)){
                    int i = propositionModel.indexOf(bean);
                    if (i != -1){
                        int[] indices = propositionList.getSelectedIndices();

                        indices = Arrays.copyOf(indices, indices.length + 1);
                        indices[indices.length - 1] = i;
                        propositionList.setSelectedIndices(indices);
                    }
                }
            }
        }
    }

    public class ThesaurusListener implements TreeSelectionListener, ListSelectionListener {
        protected JList propositionList;
        protected List<Object> oldsPropositionsSelected;
        protected OfferEditUI ui;

        public ThesaurusListener(OfferEditUI ui){
            propositionList = ui.propositionList;
            this.ui = ui;
            propositionList.getSelectionModel().addListSelectionListener(this);
        }

        @Override
        public void valueChanged(TreeSelectionEvent e) {
            TreePath[] paths = e.getPaths();

            List<Object> toAdd = new ArrayList<Object>();
            List<Object> toRemove = new ArrayList<Object>();
            for(TreePath path : paths){
                NavigationTreeNode node = (NavigationTreeNode) path.getLastPathComponent();
                if (e.isAddedPath(path)){
                    toAdd.add(node.getBean());
                } else {
                    toRemove.add(node.getBean());
                }
            }
            ThesaurusTreeHelper rootHelper = ui.getRootThesaurusHelper();
            rootHelper.desactiveListener();
            rootHelper.tryToSelect(toAdd);
            rootHelper.tryToUnselect(toRemove);
            rootHelper.activeListener();
        }

        @Override
        public void valueChanged(ListSelectionEvent e) {

            List<Object> newPropositionsSelected = Arrays.asList(propositionList.getSelectedValues());
            List<String> idsToAdd = new ArrayList<String>();
            List<String> idsToRemove = new ArrayList<String>();
            for (Object o : newPropositionsSelected){
                if (oldsPropositionsSelected == null || !oldsPropositionsSelected.contains(o)){
                    VradiThesaurusDTO proposition = (VradiThesaurusDTO)o;
                    idsToAdd.add(proposition.getWikittyId());
                }
            }
            if (oldsPropositionsSelected != null){
                for (Object o : oldsPropositionsSelected){
                    if (!newPropositionsSelected.contains(o)){
                        VradiThesaurusDTO proposition = (VradiThesaurusDTO)o;
                        idsToRemove.add(proposition.getWikittyId());
                    }
                }
            }

            ThesaurusTreeHelper rootHelper = ui.getRootThesaurusHelper();
            rootHelper.desactiveListener();
            rootHelper.tryToSelect(idsToAdd);
            rootHelper.tryToUnselect(idsToRemove);
            rootHelper.activeListener();
            // Cache
            oldsPropositionsSelected = newPropositionsSelected;
        }
    }
    
    public static class CopyToClipboard extends AbstractAction {
        private static final long serialVersionUID = 1L;
        private final AdminThesaurusUI thesaurusUI;
        
        /**
         * Register the keystroke Control+C (copy) on the treetable of the
         * specified AdminThesaurusUI.
         * 
         * @param thesaurusUI
         */
        public CopyToClipboard(AdminThesaurusUI thesaurusUI) {
            super(I18n._("vradi.common.copy"));
            this.thesaurusUI = thesaurusUI;
            
            JXTreeTable treeTable = thesaurusUI.getThesaurus();
            InputMap inputMap = treeTable.getInputMap();
            inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_C, InputEvent.CTRL_MASK),
                    "ThesaurusHandler.CopyToClipboard");
            
            ActionMap actionMap = treeTable.getActionMap();
            actionMap.put("ThesaurusHandler.CopyToClipboard", this);
        }
        
        @Override
        public void actionPerformed(ActionEvent e) {
            ThesaurusTreeHelper helper = thesaurusUI.getHelper();
            
            NavigationNode selectedNode = helper.getSelectedNode(thesaurusUI);
            if (selectedNode == null || selectedNode.isRoot()) {
                return;
            }
            
            VradiThesaurusDTO bean = (VradiThesaurusDTO) selectedNode.getBean();
            if (bean == null) {
                return;
            }
            
            String name = "\"" + bean.getName() + "\"";
            StringSelection selection = new StringSelection("thesaurus:" + name);
            
            Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
            clipboard.setContents(selection, null);
        }
    }
    
    /**
     * Export all clients as CSV.
     * 
     * Used in {@link com.jurismarches.vradi.ui.admin.nav.AdminClientNavUI}.
     * 
     * @param adminThesaurusUI parent ui
     */
    public void exportThesaurusCSV(AdminThesaurusUI adminThesaurusUI) {

        try {
            JFileChooser chooser = new JFileChooser();
            chooser.setDialogTitle(_("vradi.adminThesaurus.exportThesaurusTitle"));
            chooser.setApproveButtonText(_("vradi.adminThesaurus.exportButtonText"));
            int returnVal = chooser.showOpenDialog(adminThesaurusUI);

            if (returnVal == JFileChooser.APPROVE_OPTION) {
                File selectedFile = chooser.getSelectedFile();

                if (log.isDebugEnabled()) {
                    log.debug("Exporting thesaurus to : " + selectedFile.getAbsolutePath());
                }

                // creation du criteria wikitty
                // (export de tous les wikitty avec extension "thesaurus"
                
                // TODO EC-20100416 check to not export "all" tree nodes
                // just "thesaurus" tree nodes
                Search search = Search.query().eq(org.sharengo.wikitty.search.Element.ELT_EXTENSION,
                        TreeNode.EXT_TREENODE);
                Criteria criteria = search.criteria();

                // export
                VradiStorageService vradiStorageService = getVradiStorageService();
                String vscContent = vradiStorageService.exportAsCSV(criteria);
                
                // write content into file
                FileUtils.writeStringToFile(selectedFile, vscContent);
            }
        } catch (VradiException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't export thesaurus", eee);
            }
            ErrorDialogUI.showError(eee);
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't export thesaurus", eee);
            }
        }
    }

    /**
     * Import CSV file.
     * 
     * Can be used to import clients, groups or thesaurus.
     * 
     * @param adminThesaurusUI parent ui
     */
    public void importCSV(AdminThesaurusUI adminThesaurusUI) {
        try {
            JFileChooser chooser = new JFileChooser();
            chooser.setDialogTitle(_("vradi.adminThesaurus.importTitle"));
            chooser.setApproveButtonText(_("vradi.adminThesaurus.importButtonText"));
            int returnVal = chooser.showOpenDialog(adminThesaurusUI);
            if (returnVal == JFileChooser.APPROVE_OPTION) {
                File selectedFile = chooser.getSelectedFile();
    
                if (log.isDebugEnabled()) {
                    log.debug("Importing file : " + selectedFile.getAbsolutePath());
                }
                
                // export
                VradiStorageService vradiStorageService = getVradiStorageService();
                // FIXME EC-20100415 will not work on remote mode until uri
                // means somethings for server too
                String uri = selectedFile.toURI().toURL().toExternalForm();
                vradiStorageService.importAsCSV(uri);
                
                VradiContext.initEntryDef();
                ThesaurusTreeHelper helper = adminThesaurusUI.getHelper();
                adminThesaurusUI.getThesaurus().setTreeTableModel((NavigationTreeTableModel)helper.getModel(adminThesaurusUI));
            }
        } catch (VradiException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import file", eee);
            }
            ErrorDialogUI.showError(eee);
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import file", eee);
            }
        }
    }
}

