/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi;

import static org.apache.commons.beanutils.PropertyUtils.getPropertyDescriptor;

import java.awt.Color;
import java.beans.PropertyDescriptor;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import jaxx.runtime.context.JAXXContextEntryDef;
import jaxx.runtime.swing.ErrorDialogUI;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.BusinessEntity;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.TreeNodeImpl;
import org.sharengo.wikitty.WikittyExtension;

import com.jurismarches.vradi.entities.FieldTypeEnum;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Infogene;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.VradiStorageService;
import com.jurismarches.vradi.services.dto.VradiDTO;
import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;

/**
 * @author letellier
 */
public class VradiHelper {

    private static final Log log = LogFactory.getLog(VradiHelper.class);

    public final static String QUERIES = "vradi.queries";
    public final static String XML_STREAMS = "vradi.xmlStreams";
    public final static String CRITERIAS = "vradi.criterias";
    public final static String THESAURUS = "vradi.thesaurus";
    public final static String COLUMNS = "vradi.columns";
    public final static String LAST_VERSION = "vradi.lastVersion";
    public final static String THESAURUS_COLORS = "vradi.thesaurus.color";
    public final static String STATUS_COLOR = "vradi.status.color";
    public final static String COLOR_REGEX = "\\.\\d+";
    public final static String TYPE_DESCRIPTION = "description";
    public final static String TYPE_RANK = "rank";
    public final static String TYPE_LAST_MODIFIED_DATE = "lastModifiedDate";
    public final static String TYPE_LAST_MODIFIED_TIME = "lastModifiedTime";
    public final static String TYPE_LAST_MODIFIED_BY = "lastModifiedBy";
    public final static String TYPE_TEMPLATE = "template";
    public final static String EXPEND_SELECTED = "vradi.expend.selected";

    // Singleton sur les proprietes
    public static Properties userProperties = null;

    public static VradiConfig getVradiConfig() {
        return VradiContext.get().getVradiConfig();
    }

    protected static Properties getUserProperties() {
        if (userProperties == null) {
            userProperties = getProperties(getVradiConfig().getUserFile());
        }
        return userProperties;
    }

    protected static Properties getProperties(File propertyFile) {
        Properties properties;
        try {
            properties = new Properties();
            if (getVradiConfig().getVersion().getVersion() != null
                    && !getVradiConfig().getVersion().getVersion().isEmpty()) {
                if (!propertyFile.exists()) {
                    if (!propertyFile.createNewFile()) {
                        log.error("Failled to create new property file : " + propertyFile.getName());
                    }
                }
                InputStream input = new FileInputStream(propertyFile);
                properties.load(input);
            }
        } catch (IOException ioe) {

            throw new IllegalStateException("could not get "
                    + propertyFile.getName() + " caused by : "
                    + ioe.getMessage(), ioe);
        }
        return properties;
    }

    protected static Properties store(Properties property, File file) {
        try {
            Writer fileOutputStream = new OutputStreamWriter(
                    new FileOutputStream(file), "ISO-8859-1");
            property.store(fileOutputStream, "");
        } catch (IOException e) {
            log.error("Cant save request property caused by : ", e);
            ErrorDialogUI.showError(e);
        }

        return property;
    }

    protected static void storeUserProperties(Properties properties) {

        File userFile = getVradiConfig().getUserFile();
        if (userFile != null) {
            store(properties, userFile);
        }
    }

    protected static List<String> getUserListOfStringFromProperties(
            String propertyName) {
        Properties properties = getUserProperties();
        List<String> result = new ArrayList<String>();
        // Recuperation des propriete
        if (properties != null) {
            String resultNames = properties.getProperty(propertyName);
            if (resultNames != null) {
                for (String resultName : resultNames.split(",")) {
                    String utf8ResultName;
                    try {
                        utf8ResultName = new String(
                                resultName.getBytes("ISO-8859-1"), "UTF-8");
                    } catch (UnsupportedEncodingException eee) {
                        utf8ResultName = resultName;
                    }
                    result.add(utf8ResultName);
                }
            } else {
                log.warn("Cant find property : " + propertyName);
            }
        }
        return result;
    }

    protected static void storeUserListProperties(String propertyName,
                                                  List<String> values) {
        Properties properties = getUserProperties();
        if (properties != null) {
            StringBuffer store = new StringBuffer();
            for (String value : values) {
                store.append(value).append(",");
            }

            // Removing last ","
            int length = store.length() - 1;
            if (length > 0) {
                store.deleteCharAt(length);
            }
            properties.setProperty(propertyName, store.toString());
            storeUserProperties(properties);
        }
    }

    protected static void addToUserListProperties(String propertyName,
                                                  String value) {
        addToUserListProperties(propertyName, value, false);
    }

    protected static void addToUserListProperties(String propertyName,
                                                  String value,
                                                  boolean replace) {
        addToUserListProperties(propertyName, value, replace, false, false);
    }

    protected static void addToUserListProperties(String propertyName,
                                                  String value, boolean replace,
                                                  boolean inverse,
                                                  boolean uniqueValues) {
        List<String> result = replace ? new ArrayList<String>() :
                getUserListOfStringFromProperties(propertyName);
        if (value != null) {
            String isoValue;
            try {
                isoValue = new String(value.getBytes("UTF-8"), "ISO-8859-1");
            } catch (UnsupportedEncodingException eee) {
                isoValue = value;
            }
            if (uniqueValues && result.contains(isoValue)) {
                result.remove(isoValue);
                if (inverse) {
                    result.add(0, isoValue);
                } else {
                    result.add(isoValue);
                }
            } else {
                if (inverse) {
                    result.add(0, isoValue);
                } else {
                    result.add(isoValue);
                }
            }
        }
        storeUserListProperties(propertyName, result);
    }

    protected static List<String> removeToUserListProperties(
            String propertyName, String value) {
        List<String> result = getUserListOfStringFromProperties(propertyName);
        result.remove(value);
        storeUserListProperties(propertyName, result);

        return result;
    }

    public static void addRequestToProperties(String request) {
        addToUserListProperties(QUERIES, request, false, true, true);
    }

    public static java.util.List<String> loadRequests() {
        return getUserListOfStringFromProperties(QUERIES);
    }

    public static void setLastItemOfXmlStream(String xmlStreamName,
                                              String value) {
        addToUserListProperties(XML_STREAMS + "." + xmlStreamName, value, true);
    }

    public static String getLastItemOfXmlStream(String xmlStreamName) {
        List<String> prop = getUserListOfStringFromProperties(XML_STREAMS + "."
                + xmlStreamName);
        if (!prop.isEmpty()) {
            return prop.get(0);
        }
        return null;
    }

    public static List<String> getVradiListCriteria() {
        return getUserListOfStringFromProperties(CRITERIAS);
    }

    public static void removeVradiListCriteria(TreeNodeImpl value) {
        removeToUserListProperties(CRITERIAS, value.getWikittyId());
    }

    public static void addVradiListCriteria(TreeNodeImpl value) {
        addToUserListProperties(CRITERIAS, value.getWikittyId());
    }

    public static List<String> getVradiListThesaurus() {
        return getUserListOfStringFromProperties(THESAURUS);
    }

    public static void removeVradiListThesaurus(TreeNodeImpl value) {
        removeToUserListProperties(THESAURUS, value.getWikittyId());
    }

    public static void addVradiListThesaurus(TreeNodeImpl value) {
        addToUserListProperties(THESAURUS, value.getWikittyId());
    }

    public static List<String> getVradiListColumns() {
        return getUserListOfStringFromProperties(COLUMNS);
    }

    public static void removeVradiListColumns(String value) {
        removeToUserListProperties(COLUMNS, value);
    }

    public static void addVradiListColumns(String value) {
        addToUserListProperties(COLUMNS, value);
    }

    // Status colors
    public static Map<Integer, Color> getColorsStatusFromProperties(){
        return getColorsFromProperties(STATUS_COLOR);
    }

    public static void storeStatusColorToProperties(int i, Color toStore){
        storeColorToProperties(STATUS_COLOR, i, toStore);
    }

    // Thesaurus colors
    public static Map<Integer, Color> getColorsThesaurusFromProperties(){
        return getColorsFromProperties(THESAURUS_COLORS);
    }

    public static void storeThesaurusColorToProperties(int i, Color toStore){
        storeColorToProperties(THESAURUS_COLORS, i, toStore);
    }

    // Colors
    protected static String getColorRegex(String key){
        return key + COLOR_REGEX;
    }

    public static Map<Integer, Color> getColorsFromProperties(String key){
        Properties properties = getUserProperties();
        if (properties == null){
            return null;
        }
        Map<Integer, Color> result = new HashMap<Integer ,Color>();
        for (Object property : properties.keySet()){
            String propertyString = (String) property;
            if (propertyString.matches(getColorRegex(key))){
                if (log.isDebugEnabled()){
                    log.debug("Color match in propertie " + propertyString);
                }
                String colorValue = properties.getProperty(propertyString);

                Integer depth = Integer.parseInt(propertyString.split("\\.")[3]);
                if (log.isDebugEnabled()){
                    log.debug("Found for depth " + depth + " colorValue " + colorValue);
                }
                String[] rgb = colorValue.split(",");
                result.put(depth,
                        new Color(Integer.parseInt(rgb[0]),
                                Integer.parseInt(rgb[1]),
                                Integer.parseInt(rgb[2])));
            }
        }
        if (result.isEmpty()){
            result = loadDefaultColors(key);
        }
        return result;
    }

    protected static Map<Integer, Color> loadDefaultColors(String key) {
        Color[] colors = new Color[]{Color.BLACK, Color.BLUE,
                Color.CYAN, Color.DARK_GRAY, Color.GRAY, Color.GREEN,
                Color.LIGHT_GRAY, Color.MAGENTA, Color.ORANGE,
                Color.PINK, Color.RED, Color.YELLOW};
        
        for (int i = 0; i < 100; i++){
            storeColorToProperties(key, i, colors[i%colors.length]);
        }
        return getColorsThesaurusFromProperties();
    }

    protected static void storeColorToProperties(String key, int i, Color toStore){
        Properties properties = getUserProperties();
        properties.setProperty(key + "." + i, toStore.getRed()
                + "," + toStore.getGreen() + "," + toStore.getBlue());
        storeUserProperties(properties);
    }

    // Hiddable panels
    public static boolean isExpandSelected(){
        Properties properties = getUserProperties();
        String expendSelected = properties.getProperty(EXPEND_SELECTED);
        return Boolean.valueOf(expendSelected );
    }

    public static void storeExpandSelected(boolean expendSelected){
        Properties properties = getUserProperties();
        properties.setProperty(EXPEND_SELECTED, String.valueOf(expendSelected));
        storeUserProperties(properties);
    }

    public static String getEntityName(Object o) {
        if (o == null) {
            return "";
        }
        PropertyDescriptor descriptor;
        try {
            descriptor = getPropertyDescriptor(o, "name");

            if (descriptor == null) {
                return "";
            }
            Method getter = descriptor.getReadMethod();

            Object invoke = getter.invoke(o);
            if (invoke == null) {
                return "";
            }
            String result = (String) invoke;
            return result;
        } catch (IllegalAccessException e) {
            log.error(e);
            ErrorDialogUI.showError(e);
        } catch (InvocationTargetException e) {
            log.error(e);
            ErrorDialogUI.showError(e);
        } catch (NoSuchMethodException e) {
            log.error(e);
            ErrorDialogUI.showError(e);
        }
        return "";
    }

    // Models
    public static FieldTypeEnum getFieldTypeEnum(FieldType type) {
        for (FieldTypeEnum typeEnum : FieldTypeEnum.values()) {
            if (typeEnum.isType(type)) {
                return typeEnum;
            }
        }
        return null;
    }

    public static String getFormTypeTemplate(WikittyExtension extension) {
        String result = extension == null ?
                null : extension.getTagValue(TYPE_TEMPLATE);
        return result;
    }

    private static Map<String, Status> statusCache = new HashMap<String, Status>();

    public static Status getStatus(String statusId) {
        if(statusId == null) {
            return null;
        }
        
        Status status = statusCache.get(statusId);
        
        if (status == null) {
            VradiStorageService vradiStorageService =
                    VradiService.getVradiStorageService();
            try {
                status = vradiStorageService.getStatus(statusId);
                statusCache.put(statusId, status);
                
            } catch (VradiException eee) {
                log.error("Cant get status : " + eee);
            }
        }
        
        return status;
    }

    public static VradiFormPageDTO executeQuery(String query, WikittyExtension extension,
                                    String dateType, Date beginDate,
                                    Date endDate, List<String>[] thesaurus,
                                    String[] statusIds,
                                    VradiFormPageDTO formPageDTO) {
        try {
            if (log.isDebugEnabled()) {
                log.debug(query);
            }
            
            formPageDTO.setFormsToShow(null);
            formPageDTO.setTotalFoundFormNb(0);
            
            if (formPageDTO.getNbFormsToShow() == 0) {
                formPageDTO.setNbFormsToShow(10);
            }
            
            if (formPageDTO.getPageToShow() == 0) {
                formPageDTO.setPageToShow(1);
            }
            
            formPageDTO = VradiService.getVradiStorageService().findForms2(query, extension,
                    dateType, beginDate, endDate, thesaurus, statusIds, formPageDTO);
            
            if (log.isDebugEnabled()) {
                log.debug(formPageDTO.getTotalFoundFormNb() + " forms found");
                for (Form form : formPageDTO.getFormsToShow()) {
                    log.debug(form.getWikittyId() + " : " + form.getObjet());
                }
            }
            
        } catch (VradiException eee) {
            log.error("Cant execute query : ", eee);
            ErrorDialogUI.showError(eee);
            formPageDTO = new VradiFormPageDTO();
            
        } catch (UnsupportedQueryException e) {
            log.error(e.getMessage());
            ErrorDialogUI.showError(e);
            formPageDTO = new VradiFormPageDTO();
        }

        return formPageDTO;
    }


    public static String getFieldTypeDescription(FieldType fieldType) {
        return fieldType == null ?
                null : fieldType.getTagValue(TYPE_DESCRIPTION);
    }

    public static int getFieldTypeRank(FieldType fieldType) {
        int result;
        try {
            result = Integer.valueOf(fieldType.getTagValue(TYPE_RANK));
        } catch(NumberFormatException eee) {
            if(log.isDebugEnabled()) {
                log.debug("no tag value for rank");
            }
            result = -1;
        }
        return result;
    }

    /**
     * Get the most recent version of Vradi already run by the user
     *
     * @return the most recent version that the user has ever launched
     */
    public static String getLastVersion() {
        Properties userProperties = getUserProperties();
        return userProperties.getProperty(LAST_VERSION);
    }

    public static void setLastVersion(String lastVersion) {
        Properties userProperties = getUserProperties();
        userProperties.setProperty(LAST_VERSION, lastVersion);
        storeUserProperties(userProperties);
    }

    public static boolean isVersionNewer(String version) {
        if (getLastVersion() == null) {
            return true;
        }
        String[] versionToken = version.split("\\.");
        String[] lastVersionToken = getLastVersion().split("\\.");
        int i = 0;
        while (i < versionToken.length && i < lastVersionToken.length) {
            if (Integer.valueOf(versionToken[i])
                    > Integer.valueOf(lastVersionToken[i])) {
                return true;
            } else if (Integer.valueOf(versionToken[i])
                    < Integer.valueOf(lastVersionToken[i])) {
                return false;
            }
            i++;
        }
        return false;
    }

    // to find wikitty in ref
    public static <O extends BusinessEntity> List<O> findAllEntityInRef(JAXXContextEntryDef<List<O>> ref, List<String> wikittyIds) {
        List<O> result = new ArrayList<O>();
        for (String id : wikittyIds){
            result.add(findEntityInRef(ref, id));
        }
        return result;
    }

    // to find wikitty in ref
    public static <O extends BusinessEntity> O findEntityInRef(JAXXContextEntryDef<List<O>> ref, String wikittyId) {
        for (O entity : ref.getContextValue(VradiContext.get())){
            if (entity.getWikittyId().equals(wikittyId)){
                return entity;
            }
        }
        return null;
    }

    public static File generatePDF(VradiStorageService storageService, Form form) throws Exception {
        List<File> files = generatePDF(storageService, Collections.singletonList(form));

        if (files != null && !files.isEmpty()){
            return files.get(0);
        }
        return null;
    }

    public static List<File> generatePDF(VradiStorageService storageService, List<Form> forms) throws VradiException {
        List<File> files = new ArrayList<File>();
        for (Form form : forms) {
            
            String extensionName = null;
            for(String ext : form.getExtensionNames()) {
                if(!ext.equals(Infogene.EXT_INFOGENE)
                        && !ext.equals(Form.EXT_FORM)) {
                    extensionName = ext;
                    break;
                }

            }
            WikittyExtension extension = storageService.getFormType(extensionName);
            String template = VradiHelper.getFormTypeTemplate(extension);
            File templateFile = storageService.getTemplate(extensionName, template);
            Map<String, String> fieldMap = storageService
                .getAssociatedFields(extension.getName(), template);

            if (log.isDebugEnabled()) {
                log.debug("Generating PDF file for form " + form.getWikittyId() +
                        " (template = " + template + ")");
            }

            try {
                // TODO EC-20100510, call multiples generation
                // but only those associated with current templates !
                List<Form> currentForms = Collections.singletonList(form);
                List<File> filesToAdd = VradiService.getMailingService()
                        .generateFilledDocumentInPDF(
                                templateFile, currentForms, fieldMap, false);
                if (filesToAdd != null && !filesToAdd.isEmpty()){
                    files.addAll(filesToAdd);
                }
            } catch (Exception eee) {
                throw new VradiException("Cant generate pdf : ", eee);
            }
        }

        return files;
    }

    public static List<String> extractDTOIds(List<? extends VradiDTO<?>> dtos) {
        List<String> ids = new ArrayList<String>();
        for (VradiDTO dto : dtos){
            ids.add(dto.getWikittyId());
        }
        return ids;
    }

    public static List<String> extractIds(List<? extends BusinessEntity> beans) {
        List<String> ids = new ArrayList<String>();
        for (BusinessEntity bean : beans){
            ids.add(bean.getWikittyId());
        }
        return ids;
    }
}
