package com.jurismarches.vradi.ui.search;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.services.ServiceHelper;
import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import com.jurismarches.vradi.services.dto.VradiThesaurusDTO;
import com.jurismarches.vradi.ui.OfferListTableModel;

import javax.swing.*;

import com.jurismarches.vradi.ui.OfferListUI;
import com.jurismarches.vradi.ui.UIHelper;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.ErrorDialogUI;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import static org.nuiton.i18n.I18n._;
import org.sharengo.exceptions.TechnicalException;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.TreeNode;
import org.sharengo.wikitty.WikittyExtension;

import java.awt.*;
import java.io.File;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.List;

/**
 * @author letellier
 */
public class SearchHandler {

    /**
     * to use log facility, just put in your code: log.info(\"...\");
     */
    static private final Log log = LogFactory.getLog(SearchHandler.class);

    static public final String CSV_FILE_EXTENSION = ".csv";

    /**
     * Methode pour initialiser l'ui principale sans l'afficher.
     *
     * @param rootContext le context applicatif
     * @return l'ui instancie et initialisee mais non visible encore
     */
    public SearchUI initUI(JAXXContext rootContext, OfferListTableModel resultTableModel) {
        SearchUI ui = getUI(rootContext);
        if (ui == null) {

            JAXXInitialContext context = new JAXXInitialContext().add(rootContext);

            // show main ui
            context.add(this);
            context.add(resultTableModel);

            ui = new SearchUI(context);

            initCriteria(ui);

            VradiContext.SEARCH_UI_ENTRY_DEF.setContextValue(rootContext, ui);
        }

        return ui;
    }

    public SearchUI initCloneUI(JAXXContext rootContext, OfferListTableModel resultTableModel) {

        JAXXInitialContext context = new JAXXInitialContext().add(rootContext);

        // show main ui
        context.add(this);
        context.add(resultTableModel);

        SearchUI ui = new SearchUI(context);

        initCriteria(ui);

        return ui;
    }

    SearchUI getUI(JAXXContext context) {
        if (context instanceof SearchUI) {
            return (SearchUI) context;
        }
        SearchUI ui = VradiContext.SEARCH_UI_ENTRY_DEF.getContextValue(context);
        return ui;
    }

    public ComboBoxModel getRequestModel() {
        return new DefaultComboBoxModel();
    }

    // From Admin Handler
    public void executeQuery(JAXXContext context) {
        SearchUI ui = getUI(context);
        if (ui != null) {
            String query = ui.getRequest().trim();
            OfferListTableModel listTableModel = ui.getResultTableModel();
            log.debug(listTableModel.getNbFormsPerPage() + " - " + listTableModel.getPageToShow());
            JComboBox combo = ui.isOfferListUI() ? ui.getRequests() : null;
            CriteriaUI criterias = ui.getCriterias().isVisible() ? ui.getCriterias() : null;
            log.debug(criterias == null);
            executeQuery(ui, criterias, listTableModel, combo, query);
        }
    }

    public java.util.List<String> loadRequests() {
        return VradiHelper.loadRequests();
    }

    public JPanel getCriteriaPanel(JAXXContext context) {
        return getUI(context).getCriterias().getCriteriaPanel();
    }

    public CriteriaUI getCriterias(JAXXContext context) {
        return getUI(context).getCriterias();
    }

    public void initCriteria(JAXXContext context) {
        // Remove existing
        getCriteriaPanel(context).removeAll();

        // Adding root criteria
        TreeNode root = null;
        try {
            root = ServiceHelper.getVradiStorageService().getRootThesaurus();
        } catch (TechnicalException e) {
            log.error("Cant get rootThesaurus : ", e);
            ErrorDialogUI.showError(e);
        }
        addCriteria(context, root, false);

        // Adding saved criteria
        for (String value : VradiHelper.getVradiListCriteria()) {
            addCriteria(context, value, true);
        }
    }

    public void removeCriteria(JAXXContext context, CriteriaField criteriaField) {
        getCriteriaPanel(context).remove(criteriaField);
        getUI(context).validate();
    }

    public void removeCriteriaAndPersist(JAXXContext context, TreeNode bean) {
        CriteriaField criteriaField = ((CriteriaField) context);
        VradiHelper.removeVradiListCriteria(bean);
        removeCriteria(context, criteriaField);
        getUI(context).validate();
    }

    public void addAndPersistCriteria(JAXXContext context, TreeNode node) {
        VradiHelper.addVradiListCriteria(node);
        addCriteria(context, node, true);
    }

    protected void addCriteria(JAXXContext context, String wikittyId, boolean deletable) {
        TreeNode criteria = null;
        try {
            criteria = ServiceHelper.getVradiStorageService().getThesaurus(wikittyId);
        } catch (TechnicalException e) {
            log.error("Cant get thesaurus : " + wikittyId, e);
            ErrorDialogUI.showError(e);
        }

        if (criteria != null) {
            addCriteria(context, criteria, deletable);
        }
    }

    protected void addCriteria(JAXXContext context, TreeNode criteria, boolean deletable) {
        VradiThesaurusDTO thesCriteria = new VradiThesaurusDTO();

        thesCriteria.fromWikitty(criteria);

        // New criteria field
        CriteriaField criteriaField = new CriteriaField(context);

        // Set Name
        criteriaField.setLabelName(criteria.getName());

        // Set bean
        criteriaField.setBean(criteria);

        // Deletable
        criteriaField.setDeletable(deletable);

        // Apply model
        JComboBox combo = criteriaField.getCombo();
        VradiThesaurusDTO[] thesArray = new VradiThesaurusDTO[thesCriteria.getChildren().size() + 1];
        for (int i = 1; i < thesArray.length; i++) {
            thesArray[i] = (VradiThesaurusDTO) thesCriteria.getChildren().toArray()[i - 1];
        }
        combo.setModel(new DefaultComboBoxModel(thesArray));
        combo.setRenderer(VradiContext.get().getDecoratorProviderListCellRenderer());

        // Adding to ui
        criteriaField.init();
        getCriteriaPanel(context).add(criteriaField);
        getUI(context).validate();
    }

    protected List<String> getDateFormType(WikittyExtension formType) {
        List<String> result = new ArrayList<String>();
        result.add(Form.FQ_FIELD_DATEPUB);
        result.add(Form.FQ_FIELD_DATEPEREMPTION);
        if (formType != null) {
            for (String name : formType.getFieldNames()) {
                FieldType type = formType.getFieldType(name);
                if (type.getType() == FieldType.TYPE.DATE) {
                    result.add(formType.getName() + "." + name);
                }
            }
        }
        return result;
    }


    public DefaultComboBoxModel updateDateSelectionModel(WikittyExtension selected) {
        List<String> datas = getDateFormType(selected);
        return new DefaultComboBoxModel(datas.toArray(new String[datas.size()]));
    }


    public void executeQuery(JAXXContext context, CriteriaUI criterias, OfferListTableModel tableModel, JComboBox combo, String request) {
        String query = request.trim();
        if (criterias == null) {
            VradiHelper.executeQuery(query, null, null, null, null, null, tableModel.getDatas());
        } else {
            Component[] thesaurusCriteria = getCriteriaPanel(context).getComponents();
            List<String>[] thesaurus = new ArrayList[thesaurusCriteria.length];
            for (int i = 0; i < thesaurus.length; i++) {
                VradiThesaurusDTO th = (VradiThesaurusDTO) ((CriteriaField) thesaurusCriteria[i]).getCombo().getSelectedItem();
                if (th != null) {
                    thesaurus[i] = new ArrayList<String>();
                    thesaurus[i].add(th.getWikittyId());
                    VradiHelper.getThesaurusChildren(th, thesaurus[i]);
                }
            }
            Calendar beginDate = new GregorianCalendar();
            beginDate.setTime(criterias.getFromDate().getDate());
            beginDate.set(Calendar.HOUR_OF_DAY, (Integer) criterias.getFromHour().getSelectedItem());
            beginDate.set(Calendar.MINUTE, (Integer) criterias.getFromMinute().getSelectedItem());

            Calendar endDate = new GregorianCalendar();
            endDate.setTime(criterias.getToDate().getDate());
            endDate.set(Calendar.HOUR_OF_DAY, (Integer) criterias.getToHour().getSelectedItem());
            endDate.set(Calendar.MINUTE, (Integer) criterias.getToMinute().getSelectedItem());

            VradiHelper.executeQuery(query, (WikittyExtension) criterias.getFormTypeSelection().getSelectedItem(),
                    (String) criterias.getDateTypeSelection().getSelectedItem(),
                    beginDate.getTime(), endDate.getTime(), thesaurus, tableModel.getDatas());
        }
        if (log.isDebugEnabled()) {
            log.debug("Form found : " + tableModel.getDatas().getTotalFoundFormNb());
        }

        if (context instanceof OfferListUI) {
            VradiHelper.addRequestToProperties(query);
        }

        // Update models
        if (combo != null) {
            ((DefaultComboBoxModel) combo.getModel()).removeElement(request);
            combo.insertItemAt(request, 0);
            combo.setSelectedIndex(0);
        }
    }

    public ListCellRenderer getDateTypeComboBoxRenderer() {
        return new DefaultListCellRenderer() {
            @Override
            public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
                String stringValue = null;
                if (value != null) {
                    stringValue = value.toString();
                    stringValue = stringValue.substring(stringValue.indexOf('.') + 1);
                }
                return super.getListCellRendererComponent(list, stringValue, index, isSelected, cellHasFocus);
            }
        };
    }

    public void exportToCSV(JAXXContext context) {
        SearchUI ui = getUI(context);
        File file = UIHelper.openFileChooser(CSV_FILE_EXTENSION, "vradi.export.filter.csv", ui, "vradi.export.validate.label");
        if (file != null) {
            OfferListTableModel completeModel = new OfferListTableModel(new VradiFormPageDTO());
            executeQuery(context, ui != null ? ui.getCriterias() : null, completeModel, null,
                    ui != null ? ui.getRequest() : null);

            UIHelper.exportToCSV(file, completeModel);
        }
    }

}

