/* *##%
 * Copyright (c) 2009 Sharengo, Guillaume Dufrene, Benjamin POUSSIN.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import org.sharengo.wikitty.search.Element;
import org.sharengo.wikitty.search.Search;

/**
 *
 * @author poussin
 * @version $Revision$
 *
 * Last update: $Date$
 * by : $Author$
 */
public class WikittyProxy {

    static final public int ALL_ELEMENTS = WikittyService.ALL_ELEMENTS;

    protected WikittyService wikittyService;

	public WikittyService getWikittyService() {
		return wikittyService;
	}

	public void setWikittyService(WikittyService wikittyService) {
		this.wikittyService = wikittyService;
	}

    public <E extends BusinessEntity> E store(E e) {
        Wikitty w = WikittyUtil.beanToWikitty(e);
        UpdateResponse resp = wikittyService.store(w);
        resp.update(e);
        return e;
    }

    /**
     * Store to WikittyService objects.
     * @param objets list of BeanDto
     * @return
     */
    public <E extends BusinessEntity> List<E> store(List<E> objets) {
        // prepare data to send to service
        List<Wikitty> wikitties = new ArrayList<Wikitty>(objets.size());
        for (E e : objets) {
            Wikitty w = WikittyUtil.beanToWikitty(e);
            wikitties.add(w);
        }

        // call the service with Wikitty
        UpdateResponse resp = wikittyService.store(wikitties);

        // update object
        for (E e : objets) {
            resp.update(e);
        }
        return objets;
    }

    /**
     *
     * @param clazz
     * @param id
     * @param checkExtension if true check that Wikitty result has all extension
     * declared in clazz
     * @return
     */
	public <E extends BusinessEntity> E restore(Class<E> clazz, String id, boolean checkExtension) {
        E result = null;
        HashSet extNames = null;
        try {
            Wikitty wikitty = wikittyService.restore(id);
            if (checkExtension) {
                extNames = new HashSet<String>(wikitty.getExtensionNames());
            }
            result = WikittyUtil.newInstance(clazz, wikitty);
		} catch (Exception eee) {
			throw new WikittyException(eee);
		}
        if (checkExtension) {
            // WikittyUtil.newInstance instanciate only BusinessEntityWikittyImpl
            BusinessEntityWikitty b = (BusinessEntityWikitty)result;
            for (WikittyExtension ext : b.getStaticExtensions()) {
                if (!extNames.contains(ext.getName())) {
                    // extension wanted by BusinessEntity (clazz)
                    // is not in wikitty, then wikitty is not good type
                    // for business
                    throw new WikittyException(String.format(
                            "Id '%s' don't have extension '%s' but only %s",
                            id, ext.getName(), extNames));
                }
            }
        }
        return result;
	}
    /**
     *
     * @param clazz
     * @param id
     * @return
     */
    public <E extends BusinessEntity> E restore(Class<E> clazz, String id) {
		E result = restore(clazz, id, false);
        return result;
    }

    /**
     *
     * @param clazz
     * @param id
     * @param checkExtension if true check that Wikitty result has all extension
     * @return
     */
    public <E extends BusinessEntity> List<E> restore(
            Class<E> clazz, List<String> id, boolean checkExtension) {

        List<Wikitty> wikitties = wikittyService.restore(id);
        List<E> result = new ArrayList<E>();
        for (Wikitty w : wikitties) {
            HashSet extNames = null;
            if (checkExtension) {
                extNames = new HashSet<String>(w.getExtensionNames());
            }

            E dto = WikittyUtil.newInstance(clazz, w);
            if (checkExtension) {
                // WikittyUtil.newInstance instanciate only BusinessEntityWikittyImpl
                BusinessEntityWikitty b = (BusinessEntityWikitty) dto;
                for (WikittyExtension ext : b.getStaticExtensions()) {
                    if (!extNames.contains(ext.getName())) {
                        // extension wanted by BusinessEntity (clazz)
                        // is not in wikitty, then wikitty is not good type
                        // for business
                        throw new WikittyException(String.format(
                                "Id '%s' don't have extension '%s' but only %s",
                                w.getId(), ext.getName(), extNames));
                    }
                }
            }
            result.add(dto);
        }
        return result;
    }

    public <E extends BusinessEntity> List<E> restore(Class<E> clazz, List<String> id) {
        List<E> result = restore(clazz, id, false);
        return result;
    }

    public void delete(String id) {
        wikittyService.delete(id);
    }

    public void delete(Collection<String> ids) {
        wikittyService.delete(ids);
    }

    public <E extends BusinessEntityWikitty> PagedResult<E> findAllByExample(E e,
            int firstIndex, int endIndex, String ... fieldFacet ) {

        Criteria criteria = Search.query(e.getWikitty()).criteria()
                .setFirstIndex(firstIndex).setEndIndex(endIndex)
                .setFacetField(fieldFacet);

        PagedResult<Wikitty> pagedResult = wikittyService.findAllByCriteria(criteria);
        // we can use autoconvert = true because search by example add automaticaly
        // restriction on extension
        PagedResult<E> result = (PagedResult<E>)pagedResult.cast(e.getClass(), true);
        return result;
    }

    public <E extends BusinessEntityWikitty> E findByExample(E e) {
        Criteria criteria = Search.query(e.getWikitty()).criteria();

        Wikitty w = wikittyService.findByCriteria(criteria);
        E result = (E)WikittyUtil.newInstance(e.getClass(), w);
        return result;
    }

    /**
     * Search object that correspond to criteria and that have all extension
     * needed by BusinessEntity (clazz).
     * @param clazz
     * @param criteria
     * @return
     */
    public <E extends BusinessEntity> PagedResult<E> findAllByCriteria(
            Class<E> clazz, Criteria criteria) {

        // newInstance only return BusinessEntityWikittyImpl
        BusinessEntityWikitty sample =
                (BusinessEntityWikitty)WikittyUtil.newInstance(clazz);

        Wikitty wikitty = sample.getWikitty();
        Collection<String> extensions = wikitty.getExtensionNames();
        Search search = Search.query(criteria);
        search = search.contains(Element.ELT_EXTENSION, extensions);

        Criteria serviceCriteria = search.criteria(criteria.getName());
        serviceCriteria.setFirstIndex(criteria.getFirstIndex());
        serviceCriteria.setEndIndex(criteria.getEndIndex());
        PagedResult<Wikitty> pagedResult = wikittyService.findAllByCriteria(serviceCriteria);

        // we can cast with autoconvert true because we have add restriction
        // on extension
        PagedResult<E> result = (PagedResult<E>)pagedResult.cast(sample.getClass(), true);
        return result;
    }

    public <E extends BusinessEntity> E findByCriteria(Class<E> clazz, Criteria criteria) {
        BusinessEntityWikitty sample =
                (BusinessEntityWikitty)WikittyUtil.newInstance(clazz);

        Wikitty wikitty = sample.getWikitty();
        Collection<String> extensions = wikitty.getExtensionNames();
        Search search = Search.query(criteria);
        search = search.contains(Element.ELT_EXTENSION, extensions);

        criteria = search.criteria(criteria.getName());

        Wikitty w = wikittyService.findByCriteria(criteria);
        E result = WikittyUtil.newInstance(clazz, w);
        return result;
    }

    public void addLabel(String wikittyId, String label) {
        wikittyService.addLabel(wikittyId, label);
    }

    public PagedResult<Wikitty> findAllByLabel(String label, int firstIndex, int endIndex) {
        PagedResult<Wikitty> result = wikittyService.findAllByLabel(
                label, firstIndex, endIndex);
        return result;
    }

    public Wikitty findByLabel(String label) {
        Wikitty result = wikittyService.findByLabel(label);
        return result;
    }

    public Set<String> findAllAppliedLabels(String wikittyId) {
        Set<String> result = wikittyService.findAllAppliedLabels(wikittyId);
        return result;
    }

    public Tree restoreTree(String wikittyId) {
        Tree result = wikittyService.restoreTree(wikittyId);
        return result;
    }

    public <E extends BusinessEntity> Map<E, Integer> restoreChildren(Class<E> clazz, String wikittyId) {
        Map<TreeNode, Integer> result = wikittyService.restoreChildren(wikittyId);
        if(result == null) {
            return null;
        }

        Map<E, Integer> convertedResult = new LinkedHashMap<E, Integer>();
        Set<Entry<TreeNode, Integer>> entries = result.entrySet();
        for (Entry<TreeNode, Integer> entry : entries) {

            TreeNode node = entry.getKey();
            String nodeId = node.getWikittyId();

            E businessEntity = null;
            HashSet extNames = null;
            try {
                Wikitty wikitty = wikittyService.restore(nodeId);
                extNames = new HashSet<String>(wikitty.getExtensionNames());
                businessEntity = WikittyUtil.newInstance(clazz, wikitty);
            } catch (Exception eee) {
                throw new WikittyException(eee);
            }

            // WikittyUtil.newInstance instanciate only BusinessEntityWikittyImpl
            BusinessEntityWikitty businessEntityImpl = (BusinessEntityWikitty) businessEntity;
            Collection<WikittyExtension> staticExtensions = businessEntityImpl.getStaticExtensions();

            boolean goodType = true;
            for (Iterator<WikittyExtension> iterator = staticExtensions.iterator(); iterator.hasNext() && goodType;) {
                WikittyExtension ext = iterator.next();
                if (!extNames.contains(ext.getName())) {
                    // extension wanted by BusinessEntity (clazz)
                    // is not in wikitty, then wikitty is not good type
                    // for business
                    goodType = false;
                }
            }

            if(goodType) {
                Integer nodeCount = entry.getValue();
                convertedResult.put(businessEntity, nodeCount);
            }
        }
        return convertedResult;
    }

    public Wikitty restoreVersion(String wikittyId, String version) {
        Wikitty result = wikittyService.restoreVersion(wikittyId, version);
        return result;
    }

    public void syncImportFromXml(String xml) {
        wikittyService.syncImportFromXml(xml);
    }

    public void syncImportFromUri(String uri) {
        wikittyService.syncImportFromUri(uri);
    }

    /**
     *
     * @param uri
     * @return job id
     */
    public String asyncImportFromUri(String uri) {
        String result = wikittyService.asyncImportFromUri(uri);
        return result;
    }

    /**
     *
     * @param e
     * @return job id
     */
    public <E extends BusinessEntityWikitty> String asyncExportAllByExample(E e) {
        Criteria criteria = Search.query(e.getWikitty()).criteria();
        String result = wikittyService.asyncExportAllByCriteria(criteria);
        return result;
    }

    /**
     *
     * @param e
     * @return xml string
     */
    public <E extends BusinessEntityWikitty> String syncExportAllByExample(E e) {
        Criteria criteria = Search.query(e.getWikitty()).criteria();
        String result = wikittyService.syncExportAllByCriteria(criteria);
        return result;
    }

    /**
     *
     * @param criteria
     * @return job id
     */
    public String asyncExportAllByCriteria(Criteria criteria) {
        String result = wikittyService.asyncExportAllByCriteria(criteria);
        return result;
    }

    /**
     *
     * @param criteria
     * @return xml string
     */
    public String syncExportAllByCriteria(Criteria criteria) {
        String result = wikittyService.syncExportAllByCriteria(criteria);
        return result;
    }

    /**
     * give the job state
     * @param jobId
     * @return
     */
    public JobState infoJob(String jobId) {
        JobState result = wikittyService.infoJob(jobId);
        return result;
    }

    /**
     * cancel job you must call freeJobResource
     * @param jobId
     */
    public void cancelJob(String jobId) {
        wikittyService.cancelJob(jobId);
    }

    /**
     * This method must be call when your job is done and you have result, or
     * when you call cancelJob
     * @param jobId
     */
    public void freeJobResource(String jobId) {
        wikittyService.freeJobResource(jobId);
    }

    public void changeDataDir(String newDataDir, String oldDataDir) {
        wikittyService.changeDataDir(newDataDir, oldDataDir);
    }
}
