package com.jurismarches.vradi.ui.admin;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.entities.*;
import com.jurismarches.vradi.services.ServiceHelper;
import com.jurismarches.vradi.services.VradiStorageService;
import com.jurismarches.vradi.services.XmlStreamService;
import com.jurismarches.vradi.services.dto.VradiXmlStreamDTO;
import com.jurismarches.vradi.ui.DateFormTypeFilterHandler;
import com.jurismarches.vradi.ui.OfferListTableModel;
import com.jurismarches.vradi.ui.OfferListUI;
import com.jurismarches.vradi.ui.Thesaurus;
import com.jurismarches.vradi.ui.VradiCriteriaField;
import com.jurismarches.vradi.ui.admin.content.AdminXmlStreamUI;
import com.jurismarches.vradi.ui.admin.content.FieldSelectUI;
import com.jurismarches.vradi.ui.admin.content.RequestSelectUI;
import com.jurismarches.vradi.ui.admin.content.XmlStreamFieldSelectUI;

import java.awt.BorderLayout;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.swing.navigation.NavigationTreeNode;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import static org.nuiton.i18n.I18n._;
import org.sharengo.exceptions.TechnicalException;
import org.sharengo.wikitty.BusinessEntity;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyUtil;

import javax.swing.*;
import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.*;
import java.util.List;

/**
 * @author letellier
 */
public class AdminHandler {

    /**
     * to use log facility, just put in your code: log.info(\"...\");
     */
    static private final Log log = LogFactory.getLog(AdminHandler.class);

    AdminPopupUI getUI(JAXXContext context) {
        if (context instanceof AdminPopupUI) {
            return (AdminPopupUI) context;
        }
        AdminPopupUI ui = VradiContext.ADMIN_UI_ENTRY_DEF.getContextValue(context);
        return ui;
    }

    OfferListUI getOfferListUI(JAXXContext context) {
        return VradiContext.OFFERT_LIST_UI_ENTRY_DEF.getContextValue(context);
    }

    void close(JAXXContext context) {
        AdminPopupUI ui = getUI(context);
        ui.setVisible(false);
        ui.dispose();
    }

    protected VradiStorageService getVradiStorageService() {
        return ServiceHelper.getVradiStorageService();
    }

    protected XmlStreamService getXmlStreamService() {
        return ServiceHelper.getXmlStreamService();
    }

    // COMMON
    protected boolean answerToDelete(JAXXContext context, String name) {
        AdminPopupUI ui = getUI(context);

        int n = JOptionPane.showConfirmDialog(ui, _("vradi.admin.deleteAnswer") + name + " ?", _("vradi.admin.deleteTitle"), JOptionPane.YES_NO_OPTION);

        return n == JOptionPane.OK_OPTION;
    }

    protected void repaintSelectedNode(JAXXContext context, VradiNavigationTreeHelper helper) {
        helper.repaintNode(context, getSelectedNode(context, helper));
    }

    protected NavigationTreeNode getSelectedNode(JAXXContext context, VradiNavigationTreeHelper helper) {
        return helper.getSelectedNode(context);
    }

    public void save(Object bean) {
        BusinessEntity entity = (BusinessEntity) bean;
        try {
            // up
            getVradiStorageService().updateEntity(entity);
        } catch (TechnicalException ex) {
            log.error("Cant saving bean : " + entity + " id : " + entity.getWikittyId() + " caused by : " + ex);
        }
    }

    public void save(JAXXContext context, VradiNavigationTreeHelper helper, Object... beans) {
        try {
            for (Object bean : beans) {
                log.info("will save bean " + bean);
                save(bean);
            }

        } finally {

            // Refresh nav
            repaintSelectedNode(context, helper);
        }
    }


    public boolean answerToSave(AdminContentUI<?> content) {
        if (content != null) {

            log.info("current content  to close " + content.getClass().getSimpleName() + " : " + content.isModified());
            if (content.isModified()) {
                int n = JOptionPane.showConfirmDialog(content, _("vradi.admin.saveAnswer") + VradiHelper.getEntityName(content.getBean()) + " ?", _("vradi.admin.saveTitle"), JOptionPane.YES_NO_CANCEL_OPTION);
                switch (n) {
                    case JOptionPane.NO_OPTION:
                        // annule les modifications
                        cancel(content, content.getHelper());
                        break;
                    case JOptionPane.OK_OPTION:
                        // sauvegarde les modifications
                        content.postSave();
                        break;
                    case JOptionPane.CANCEL_OPTION:
                        // annule le changement d'onglet
                        return false;

                }
            }
        }
        return true;
    }

    public void delete(JAXXContext context, VradiNavigationTreeHelper helper, Object bean) {
        BusinessEntity entity = (BusinessEntity) bean;
        try {
            // up
            getVradiStorageService().deleteEntity(entity.getWikittyId());

        } catch (TechnicalException ex) {
            log.error("Cant delete bean : " + entity + " id : " + entity.getWikittyId() + " caused by : " + ex);
        } finally {

            // remove node
            NavigationTreeNode parent = helper.removeChildNode(helper.getSelectedNode(context));

            // select parent node
            helper.selectNode(context, parent);
        }
    }

    protected <B> void cancel(AdminContentUI<B> content, VradiNavigationTreeHelper helper) {
        B selectedBean = content.getSelectedEntity();
        try {
            content.openUI(helper.getSelectedNode(content));
        } catch (Exception e) {
            log.error("Cant cancel bean : " + selectedBean + " caused by : " + e, e);
        }
    }

    // MODELS
    public OfferListTableModel getResultTableModel(JTable table, List<Form> forms) {
        OfferListTableModel model = new OfferListTableModel(forms, false);
        return model;
    }

    public ComboBoxModel getTypesModel() {
        ComboBoxModel model = new FieldTypeModel();
        return model;
    }

    public ComboBoxModel getEntitiesComboBoxModel(List entities) {
        return new DefaultComboBoxModel(entities.toArray());
    }

    // REQUEST
    public void openRequest(JAXXContext context, String request) {
        AdminRequestUI requestUI = new AdminRequestUI(context);
        DateFormTypeFilterHandler criteriaHandler = new DateFormTypeFilterHandler();

        requestUI.getCriteria().add(criteriaHandler.initUI(context), BorderLayout.CENTER);
        requestUI.show(request);

        RequestSelectUI ui = (RequestSelectUI)context;
        ui.setBean(requestUI.getBean());
    }

    // CLIENT

    public List<Client> getAllClients(JAXXContext context) {
        List<Client> values = VradiNavigationTreeHelper.CLIENTS.getContextValue(context);
        // always return a safe copy
        return new ArrayList<Client>(values);
    }

    public List<User> getAllUsers(JAXXContext context) {
        List<User> values = VradiNavigationTreeHelper.USERS.getContextValue(context);
        // always return a safe copy
        return new ArrayList<User>(values);
    }

    public void createClient(JAXXContext context, VradiNavigationTreeHelper helper) {
        NavigationTreeNode addClient = helper.addClient(context, getSelectedNode(context, helper), null);
        helper.selectNode(context, addClient);
    }

    // USER

    public void createUser(JAXXContext context, VradiNavigationTreeHelper helper) {
        NavigationTreeNode addUser = helper.addUser(context, getSelectedNode(context, helper), null);
        helper.selectNode(context, addUser);
    }

    // GROUP

    public List<Group> getAllGroups(JAXXContext context) {
        List<Group> values = VradiNavigationTreeHelper.GROUPS.getContextValue(context);
        // always return a safe copy
        return new ArrayList<Group>(values);
    }

    public void createGroup(JAXXContext context, VradiNavigationTreeHelper helper) {
        NavigationTreeNode addGroup = helper.addGroup(context, getSelectedNode(context, helper), null);
        helper.selectNode(context, addGroup);
    }

    // FORM

    public List<WikittyExtension> getAllForms(JAXXContext context) {
        List<WikittyExtension> values = VradiNavigationTreeHelper.FORMS.getContextValue(context);
        return values;
    }

    public void createFormType(JAXXContext context, VradiNavigationTreeHelper helper) {
        String name = JOptionPane.showInputDialog(getUI(context), _("vradi.adminForm.askName"));

        if (name != null && !name.isEmpty()) {
            WikittyExtension formType = new WikittyExtension(name, WikittyUtil.DEFAULT_VERSION,
                    Form.EXT_FORM, new LinkedHashMap<String, FieldType>());
            NavigationTreeNode addFormType = helper.addFormType(context, getSelectedNode(context, helper), formType);
            helper.selectNode(context, addFormType);
        }
    }

    public void createForms(XmlStream xmlStream) {
        if (xmlStream != null) {
            try {
                String lastItem = getVradiStorageService().getFormsFromXmlStream(xmlStream,
                        VradiHelper.getLastItemOfXmlStream(xmlStream.getName()));
                log.info(lastItem);
                VradiHelper.setLastItemOfXmlStream(xmlStream.getName(), lastItem);
            } catch (TechnicalException e) {
                e.printStackTrace();
            }
        }
    }

    public void saveFormType(JAXXContext context, VradiNavigationTreeHelper helper, java.util.List<FieldSelectUI> fields, WikittyExtension formType) {
        WikittyExtension newFormType = new WikittyExtension(formType.getName(), formType.getVersion(), formType.getRequires(), null);
        for (FieldSelectUI field : fields) {
            String name = field.getNameValue();
            log.debug(name);
            if (name != null && field.getType() != null) {
                FieldType fieldType = field.getType().getFieldType();
                newFormType.addField(name, fieldType);
            }
        }
        try {
            // up
            getVradiStorageService().updateFormType(newFormType);

            // Refresh nav
            repaintSelectedNode(context, helper);

            // Refresh criteria
            VradiContext.CRITERIA_UI_ENTRY_DEF.getContextValue(context).getFormTypeSelection().setModel(new DefaultComboBoxModel(VradiHelper.getFormTypes()));

        } catch (TechnicalException ex) {
            log.error("Cant adding formType : " + formType.getName() + " id : " + formType.getId() + " caused by : " + ex);
        }
    }

    public List<Form> query(String query) {
        try {
            List<Form> result = getVradiStorageService().findForms(query);
            return result;
        } catch (TechnicalException eee) {
            eee.printStackTrace();
            return null;
        }
    }

//    public ComboBoxModel getFormTypesModel(JAXXContext context) {
//        List<WikittyExtension> values = VradiNavigationTreeHelper.FORMS.getContextValue(context);
//        return getEntitiesComboBoxModel(values);
//    }

    public WikittyExtension getExtension(String extensionName) {
        try {
            return getVradiStorageService().getFormType(extensionName);
        } catch (TechnicalException e) {
            if (log.isErrorEnabled()) {
                log.error(e);
            }
            return null;
        }
    }
//    public void deleteForm(JAXXContext context, WikittyExtension form){
//        // del
//        if (answerToDelete(context, form.getName())){
//            try {
//                getVradiStorageService(context).deleteFormType(form);
//
//                // Refresh nav
//                AdminNavUI adminFormUI = getUI(context).getAdminFormUI();
//                delItemToNav(adminFormUI, form);
//            } catch (TechnicalException ex) {
//                log.error("Cant deleting user : " + form.getName() + " id : " + form.getId() + " caused by : " + ex);
//            }
//        }
//    }

    // RSS
    public void createXmlStream(JAXXContext context, VradiNavigationTreeHelper helper) {
        NavigationTreeNode addXmlStream = helper.addXmlStream(context, getSelectedNode(context, helper), null);
        helper.selectNode(context, addXmlStream);
    }

//    public void deleteXmlStream(JAXXContext context, XmlStream xmlStream) {
//        // del
//        if (answerToDelete(context, xmlStream.getUrl())) {
//            try {
//                getVradiStorageService().deleteXmlStream(xmlStream.getWikittyId());
//            } catch (TechnicalException ex) {
//                log.error("Cant deleting xmlStream : " + xmlStream.getUrl() + " id : " + xmlStream.getWikittyId() + " caused by : " + ex);
//            }
//        }
//    }

    public VradiXmlStreamDTO saveXmlStream(JAXXContext context, VradiNavigationTreeHelper helper,
                                           XmlStream xmlStream, List<XmlFieldBinding> bindings) {
        VradiXmlStreamDTO result = new VradiXmlStreamDTO();
        result.fromWikitty(xmlStream);
        try {
            getVradiStorageService().updateXmlFieldBindings(bindings);
            xmlStream.clearXmlFieldBinding();
            for (XmlFieldBinding binding : bindings) {
                xmlStream.addXmlFieldBinding(binding.getWikittyId());
            }
            xmlStream = getVradiStorageService().updateXmlStream(xmlStream);
            result.fromWikitty(xmlStream);

            // Refresh nav
            repaintSelectedNode(context, helper);
        } catch (TechnicalException ex) {
            log.error("Cant adding client : " + xmlStream.getUrl() + " id : " + xmlStream.getWikittyId() + " caused by : " + ex);
        }
        return result;
    }

    public List<String> getXmlStreamAttributes(String url) {
        List<String> xmlFields = new ArrayList<String>();
        try {
            xmlFields = getXmlStreamService().getRSSFields(url);
        } catch (TechnicalException eee) {
            eee.printStackTrace();
        }

        return xmlFields;
    }

    public List<XmlFieldBinding> getXmlStreamFields(XmlStream xmlStream) {
        List<XmlFieldBinding> result = new ArrayList<XmlFieldBinding>();
        try {
            result = getVradiStorageService().getXmlFieldBindings(xmlStream);
        } catch (TechnicalException eee) {
            log.error(eee.getMessage(), eee);
        }
        return result;
    }

    public WikittyExtension getFormType(List<XmlFieldBinding> bindings) {
        WikittyExtension result = null;
        for (XmlFieldBinding binding : bindings) {
            String formField = binding.getFormField();
            if (formField != null) {
                WikittyExtension formType = getExtension(formField.substring(0,
                        formField.lastIndexOf('.')));
                if (formType != null && !formType.getName().equals(Form.EXT_FORM)) {
                    result = formType;
                    break;
                }
            }
        }
        return result;
    }

    public void fillXmlStreamFields(final AdminXmlStreamUI ui) {
        List<XmlFieldBinding> bindings = getXmlStreamFields(ui.getBean());
        ui.getContent().removeAll();
        Map<String, XmlFieldBinding> bindingMap = new java.util.HashMap<String, XmlFieldBinding>();

        if (bindings != null) {
            for (XmlFieldBinding binding : bindings) {
                String formField = binding.getFormField();
                bindingMap.put(formField, binding);
//                if (ui.getFormType() == null && formField != null) {
//                    WikittyExtension formType = getExtension(formField.substring(0,
//                            formField.lastIndexOf('.')));
//                    if (!formType.getName().equals(Form.EXT_FORM)) {
//                        ui.setFormType(formType);
//                    }
//                }
            }
        }
        List<String> fieldNames = new ArrayList<String>();

        WikittyExtension formType = ui.getBean().getFormType();

        if (formType != null) {
            fieldNames.add(Form.FQ_FIELD_ID);
            fieldNames.add(Form.FQ_FIELD_NAME);
            fieldNames.add(Form.FQ_FIELD_DATEPUB);
            fieldNames.add(Form.FQ_FIELD_DATEPEREMPTION);
            for (String fieldName : formType.getFieldNames()) {
                fieldNames.add(formType.getName() + "." + fieldName);
            }
        }
        List<String> xmlFieldNames = ui.getBean() == null || ui.getBean().getUrl() == null
                || ui.getBean().getUrl().isEmpty() ? new ArrayList<String>() :
                getXmlStreamAttributes(ui.getBean().getUrl());

        for (String fieldName : fieldNames) {
            XmlStreamFieldSelectUI fieldSelectUI = new XmlStreamFieldSelectUI(ui);
            XmlFieldBinding binding = bindingMap.get(fieldName);
            if (binding == null) {
                binding = new XmlFieldBindingImpl();
                binding.setFormField(fieldName);
            }
            fieldSelectUI.setBean(binding);
            fieldSelectUI.setXmlFields(ui.getBean().getUrl(), xmlFieldNames);
            ui.getContent().add(fieldSelectUI);
            fieldSelectUI.addPropertyChangeListener(new PropertyChangeListener() {
                public void propertyChange(PropertyChangeEvent evt) {
                    ui.setModified(true);
                }
            });
        }
    }

    /**
     * Create the model and store it in the given context.
     * <p/>
     *
     * @param context the context where to hold the model
     */
    public void createModel(JAXXContext context) {
        if (log.isDebugEnabled()) {
            log.debug("createModel");
        }
        try {
            List<WikittyExtension> allForms = getVradiStorageService().getAllFormTypes();
            Collections.sort(allForms, FORM_COMPARATOR);
            VradiNavigationTreeHelper.FORMS.setContextValue(context, allForms);

            List<Group> allGroups = getVradiStorageService().getAllGroups();
            Collections.sort(allGroups, GROUP_COMPARATOR);
            VradiNavigationTreeHelper.GROUPS.setContextValue(context, allGroups);

            List<Client> clients = getVradiStorageService().getAllClients();
            Collections.sort(clients, CLIENT_COMPARATOR);
            VradiNavigationTreeHelper.CLIENTS.setContextValue(context, clients);

            List<XmlStream> xmlStreams = getVradiStorageService().getAllXmlStreams();
            Collections.sort(xmlStreams, XML_STREAM_COMPARATOR);
            VradiNavigationTreeHelper.XMLSTREAMS.setContextValue(context, xmlStreams);

            List<User> users = new ArrayList<User>();
            for (Client c : clients) {
                Set<String> usersId = c.getUser();
                if (usersId != null) {
                    for (String userId : usersId) {
                        User user = getVradiStorageService().getUser(userId);
                        users.add(user);
                    }
                }
            }
            Collections.sort(users, USER_COMPARATOR);
            VradiNavigationTreeHelper.USERS.setContextValue(context, users);
        } catch (TechnicalException ex) {
            log.error(ex);
            //FIXME should we do something ??? (return an empty model for example)
        }
    }

    public void executeQuery(AdminRequestUI ui) {
        String query = ui.getRequest().getText().trim();
        List<Form> forms;
        if (ui.isAdvencedSearchCollapsed()) {
            forms = VradiHelper.executeQuery(query, null, null, null, null, null);
        } else {
            Component[] thesaurusCriteria = ui.getCriteria().getCriterias().getComponents();
            List<String>[] thesaurus = new ArrayList[thesaurusCriteria.length];
            for (int i = 0; i < thesaurus.length; i++) {
                Thesaurus th = (Thesaurus) ((VradiCriteriaField) thesaurusCriteria[i]).getCombo().getSelectedItem();
                if (th != null) {
                    thesaurus[i] = new ArrayList<String>();
                    thesaurus[i].add(th.getWikittyId());
                    VradiHelper.getThesaurusChildren(th, thesaurus[i]);
                }
            }
            forms = VradiHelper.executeQuery(query, (WikittyExtension) ui.getCriteria().getFormTypeSelection().getSelectedItem(),
                    (String) ui.getCriteria().getDateTypeSelection().getSelectedItem(),
                    ui.getCriteria().getFromDate().getDate(), ui.getCriteria().getToDate().getDate(), thesaurus);
        }
        log.debug(forms.size());
        ui.getResultTableModel().setDatas(forms);
    }

    protected static final Comparator<Client> CLIENT_COMPARATOR = new Comparator<Client>() {

        @Override
        public int compare(Client o1, Client o2) {
            return o1.getName().compareTo(o2.getName());
        }
    };
    protected static final Comparator<User> USER_COMPARATOR = new Comparator<User>() {

        @Override
        public int compare(User o1, User o2) {
            return o1.getName().compareTo(o2.getName());
        }
    };
    protected static final Comparator<Group> GROUP_COMPARATOR = new Comparator<Group>() {

        @Override
        public int compare(Group o1, Group o2) {
            return o1.getName().compareTo(o2.getName());
        }
    };
    protected static final Comparator<WikittyExtension> FORM_COMPARATOR = new Comparator<WikittyExtension>() {

        @Override
        public int compare(WikittyExtension o1, WikittyExtension o2) {
            return o1.getName().compareTo(o2.getName());
        }
    };
    protected static final Comparator<XmlStream> XML_STREAM_COMPARATOR = new Comparator<XmlStream>() {

        @Override
        public int compare(XmlStream o1, XmlStream o2) {
            return o1.getName().compareTo(o2.getName());
        }
    };
}
