/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: ImportXmlStreamTasks.java 1650 2010-10-20 09:19:05Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.4.0/vradi-services/src/main/java/com/jurismarches/vradi/services/tasks/ImportXmlStreamTasks.java $
 * %%
 * Copyright (C) 2010 JurisMarches, Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package com.jurismarches.vradi.services.tasks;

import java.io.UnsupportedEncodingException;
import java.util.List;
import java.util.Properties;
import java.util.TimerTask;

import com.jurismarches.vradi.VradiServiceConfiguration;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.PasswordAuthentication;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ExceptionUtil;
import org.nuiton.wikitty.WikittyProxy;

import com.jurismarches.vradi.beans.XmlStreamImportResult;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.managers.BindingManager;

import static org.nuiton.i18n.I18n._;

/**
 * This task run at specified stream import time to import new stream.
 * 
 * @author chatellier
 * @version $Revision: 1650 $
 * 
 * Last update : $Date: 2010-10-20 11:19:05 +0200 (mer., 20 oct. 2010) $
 * By : $Author: sletellier $
 */
public class ImportXmlStreamTasks extends TimerTask {

    private static final Log log = LogFactory.getLog(ImportXmlStreamTasks.class);

    protected WikittyProxy wikittyProxy;

    protected BindingManager bindingManager;

    /** Il faut l'id du flux et non le flux (wikitty obselete). */
    protected String xmlStreamId;

    public ImportXmlStreamTasks(WikittyProxy wikittyProxy, BindingManager bindingManager, String xmlStreamId) {
        this.wikittyProxy = wikittyProxy;
        this.bindingManager = bindingManager;
        this.xmlStreamId = xmlStreamId;
    }

    /*
     * @see java.util.TimerTask#run()
     */
    @Override
    public void run() {

        try {
            XmlStream xmlStream = wikittyProxy.restore(XmlStream.class, xmlStreamId);
            if (log.isInfoEnabled()) {
                log.info("Starting xml stream import on : " + xmlStream.getName() + "...");
            }
            XmlStreamImportResult importResult = bindingManager.importFormsFromXmlStream(xmlStream);
            if (log.isInfoEnabled()) {
                log.info("Xml stream " + xmlStream.getName() + " imported : " + importResult.getCreatedFormCount() + " new forms.");
            }
        } catch (VradiException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import forms from stream " + xmlStreamId, eee);
            }
            sendEmailError(xmlStreamId, eee);
        }
    }

    protected void sendEmailError(String xmlStreamId, VradiException error) {

        log.info("Error is occured, sending error email");

        // Restore XmlStream
        XmlStream xmlStream = wikittyProxy.restore(XmlStream.class, xmlStreamId);

        // Get admin emails
        final VradiServiceConfiguration config = VradiServiceConfiguration.getInstance();
        List<String> mailAdmin = config.getMailAdmin();
        for (String email : mailAdmin) {

            if (log.isDebugEnabled()) {
                log.debug("Sending error email to " + email);
            }
            if (StringUtils.isBlank(email)) {
                if (log.isWarnEnabled()) {
                    log.warn("Email is empty : " + email);
                }
                return;
            }

            // TODO EC-20100505 use Session.getDefaultInstance()
            // and common Properties form receiving/sending mails !
            Properties props = new Properties();
            props.put("mail.smtp.host", config.getSmtpHost());
            props.put("mail.smtp.port", config.getSmtpPort());
            //props.put("mail.smtp.auth", "true");
            //props.put("mail.smtp.socketFactory.port", config.getSmtpPort());
            //props.put("mail.smtp.socketFactory.class", SSL_FACTORY);
            //props.put("mail.smtp.socketFactory.fallback", "false");
            //props.put("mail.debug", config.isMailDebug());

            Session session = Session.getInstance(props,
                new javax.mail.Authenticator() {
                    @Override
                    protected PasswordAuthentication getPasswordAuthentication() {
                        return new PasswordAuthentication(
                                config.getMailUser(), config.getMailPassword());
                    }
                });

            // EC-20100427 : attention, ca affiche le contenu des pdf joint
            // c'est absolument illisible
            //session.setDebug(log.isDebugEnabled());

            // create a message
            MimeMessage msg = new MimeMessage(session);

            try {
                // set the from and to address
                InternetAddress addressFrom = new InternetAddress(config.getMailFrom(), config.getMailFromName());

                msg.setFrom(addressFrom);

                InternetAddress addressTo = new InternetAddress(email);
                msg.addRecipient(Message.RecipientType.TO, addressTo);

                Multipart multiparts = new MimeMultipart();

                // create the message part
                MimeBodyPart msgBodyPart = new MimeBodyPart();
                msgBodyPart.setText(_("vradi.error.errorImportXmlStream", xmlStream.getName(),
                        ExceptionUtil.stackTrace(error)));
                
                multiparts.addBodyPart(msgBodyPart);

                // Put parts in message
                msg.setContent(multiparts);

                // Setting the Subject and Content Type
                msg.setSubject(_("vradi.error.errorImportXmlStream.subject", xmlStream.getName()));

                try {
                    Transport.send(msg);
                } catch (MessagingException eee) {
                    if (log.isErrorEnabled()) {
                        log.error("Cant send mail", eee);
                    }
                }


            } catch (MessagingException eee) {
                if (log.isErrorEnabled()) {
                    log.error("Can't generate message", eee);
                }
            } catch (UnsupportedEncodingException eee) {
                if (log.isErrorEnabled()) {
                    log.error("Can't generate message", eee);
                }
            }
        }
    }
}
