/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package com.jurismarches.vradi;

import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.services.ServiceFactory;
import com.jurismarches.vradi.services.VradiException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.StringUtil;
import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Search;

import java.util.ArrayList;
import java.util.List;

/**
 * This class reference all action run by Vradi Service :
 *
 * -- reIndexSolr : reindex solr index
 *
 * @author sletellier <letellier@codelutin.com>
 */
public class VradiServiceAction {

    /** log. */
    private static final Log log = LogFactory.getLog(VradiServiceAction.class);

    /**
     * Actions can be called in command line
     *
     * @author sletellier <letellier@codelutin.com>
     */
    public enum ActionDefinition {

        REINDEX_SOLR(VradiServiceAction.class.getName() + "#reIndexSolr", "--reIndexSolr"),
        DELETE_OBSELETE_THESAURUS(VradiServiceAction.class.getName() + "#deleteObseleteThesaurus", "--deleteObseleteThesaurus"),
        REINIT_STATUS(VradiServiceAction.class.getName() + "#reinitStatus", "--reinitStatus"),
        FORCE_ALL_OLD_USER_ENABLE(VradiServiceAction.class.getName() + "#forceAllOldUserEnable", "--forceAllOldUserEnable");

        public String action;

        public String[] aliases;

        ActionDefinition(String action, String... aliases) {
            this.action = action;
            this.aliases = aliases;
        }

        public String getAction() {
            return action;
        }

        public String[] getAliases() {
            return aliases;
        }
    }

    /**
     *
     * Reload solr index
     *
     */
    public static void reIndexSolr() {
        log.info("Starting reIndex solr");

        final long startingTime = System.nanoTime();

        WikittyService wikittyService = ServiceFactory.getWikittyService();
        String securityToken = ServiceFactory.getWikittyProxy().getSecurityToken();
        wikittyService.syncEngin(securityToken);

        log.info("ReIndex solr is finished in " + StringUtil.convertTime(startingTime, System.nanoTime()));

        System.exit(1);
    }

    /**
     *
     * Delete obselete thesaurus
     *
     */
    public static void deleteObseleteThesaurus() {
        log.info("Starting deleting obselete thesaurus");

        final long startingTime = System.nanoTime();

        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, Thesaurus.EXT_THESAURUS).criteria();

        WikittyProxy proxy = ServiceFactory.getWikittyProxy();
        List<Thesaurus> thesauruses = proxy.findAllByCriteria(Thesaurus.class, criteria).getAll();

        for (Thesaurus thesaurus : thesauruses) {
            if (thesaurus == null) {
                break;
            }
            String parentId = thesaurus.getParent();
            if (parentId == null) {
                proxy.delete(thesaurus.getWikittyId());

                log.info("Found thesaurus '" + thesaurus.getName() + "' has no parentId, deleting this");
            } else {
                Thesaurus parent = proxy.restore(Thesaurus.class, parentId);
                if (parent == null) {
                    proxy.delete(thesaurus.getWikittyId());
                    log.info("Found thesaurus '" + thesaurus.getName() + "' has no parent with id '" + parentId + ", deleting this");
                }
            }
        }

        log.info("Deleting obselete thesaurus is finished in " + StringUtil.convertTime(startingTime, System.nanoTime()));

        System.exit(1);
    }

    /**
     *
     * Delete obselete status.
     *
     */
    public static void reinitStatus() {
        log.info("Starting reinit status");

        final long startingTime = System.nanoTime();

        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, Status.EXT_STATUS).criteria();

        WikittyProxy proxy = ServiceFactory.getWikittyProxy();
        List<Status> statuses = proxy.findAllByCriteria(Status.class, criteria).getAll();
        List<String> statusesIds = new ArrayList<String>();

        for (Status status : statuses) {
            statusesIds.add(status.getWikittyId());
            log.info("Deleting status " + status.getName());
        }

        proxy.delete(statusesIds);

        try {
            ServiceFactory.getVradiStorageService().createDefaultStatuses();
        } catch (VradiException eee) {
            log.error(eee);
        }

        log.info("reinit status is finished in " + StringUtil.convertTime(startingTime, System.nanoTime()));

        System.exit(1);
    }

    /**
     *
     * Force all old user to enable
     *
     */
    public static void forceAllOldUserEnable() {
        log.info("Starting forcing all user enabled");

        final long startingTime = System.nanoTime();

        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, User.EXT_USER).criteria();

        WikittyProxy proxy = ServiceFactory.getWikittyProxy();
        List<User> users = proxy.findAllByCriteria(User.class, criteria).getAll();

        for (User user : users) {
            if (user == null) {
                break;
            }
            Object field = user.getField(User.EXT_USER, User.FIELD_USER_ENABLE);

            if (field == null) {
                log.info("Found user '" + user.getName() + "' with no enable");
                
                // FIXME EC20100910 remove direct access to wikitty
                user.setField(User.EXT_USER, User.FIELD_USER_ENABLE, true);
                proxy.store(user);
            }
        }

        log.info("Forcing all user enabled is finished in " + StringUtil.convertTime(startingTime, System.nanoTime()));

        System.exit(1);
    }
}
