/* *##%
 * Vradi :: Services
 * Copyright (C) 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package com.jurismarches.vradi.services.managers;

import static org.nuiton.i18n.I18n._;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.TimerTask;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.activation.FileDataSource;
import javax.mail.BodyPart;
import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.PasswordAuthentication;
import javax.mail.Session;
import javax.mail.Store;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.search.FlagTerm;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.PagedResult;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.search.Element;
import org.sharengo.wikitty.search.Search;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.services.Configuration;
import com.jurismarches.vradi.services.VradiException;
import com.sun.mail.smtp.SMTPAddressFailedException;
import com.sun.mail.smtp.SMTPSendFailedException;

/**
 * Mailing manager.
 * 
 * Handle:
 * <ul>
 *  <li>mail imap sending/receiving</li>
 *  <li>timer task to check for new mails each 10 minutes</li>
 * </ul>
 * 
 * @author Yolpo, kmorin, chatellier
 */
public class MailingManager extends TimerTask {

    private static final Log log = LogFactory.getLog(MailingManager.class);

    protected static final String SSL_FACTORY = "javax.net.ssl.SSLSocketFactory";

    protected WikittyProxy proxy;
    
    public MailingManager(WikittyProxy proxy) {
        this.proxy = proxy;
    }

    /**
     * Sends an email to the specified recipient with the specified subject
     * and the specified message and the specified forms PDF.
     * @param recipient the email address of the recipient
     * @param subject mail subject
     * @param message mail content
     * @param forms the list of forms to add to the mail
     * @param receptionProof if true, ask for a reception proof
     * @return the message id of the email sent
     * @throws VradiException
     */
    public String postMail(String recipient, String subject,
            String message, List<Form> forms, boolean receptionProof)
            throws VradiException {

        final Configuration config = Configuration.getInstance();

        if (recipient == null
                || (message == null && (forms == null || forms.isEmpty()))) {
            if (log.isWarnEnabled()) {
                log.warn("nothing to send");
            }
            return null;
        }
        String result = null;

        // TODO EC-20100505 use Session.getDefaultInstance()
        // and common Properties form receiving/sending mails !
        Properties props = new Properties();
        props.put("mail.smtp.host", config.getSmtpHost());
        props.put("mail.smtp.port", config.getSmtpPort());
        //props.put("mail.smtp.auth", "true");
        //props.put("mail.smtp.socketFactory.port", config.getSmtpPort());
        //props.put("mail.smtp.socketFactory.class", SSL_FACTORY);
        //props.put("mail.smtp.socketFactory.fallback", "false");
        //props.put("mail.debug", config.isMailDebug());

        Session session = Session.getInstance(props,
            new javax.mail.Authenticator() {
                @Override
                protected PasswordAuthentication getPasswordAuthentication() {
                    return new PasswordAuthentication(
                            config.getMailUser(), config.getMailPassword());
                }
            });

        // EC-20100427 : attention, ca affiche le contenu des pdf joint
        // c'est absolument illisible
        //session.setDebug(log.isDebugEnabled());

        // create a message
        MimeMessage msg = new MimeMessage(session);

        try {
            // set the from and to address
            InternetAddress addressFrom = new InternetAddress(config.getMailFrom(), config.getMailFromName());

            msg.setFrom(addressFrom);

            InternetAddress addressTo = new InternetAddress(recipient);
            msg.addRecipient(Message.RecipientType.TO, addressTo);

            if (receptionProof) {
                // Optional : You can also set your custom headers in the Email if you Want
                msg.addHeader("Return-Receipt-To", config.getMailUser());
                msg.addHeader("Disposition-Notification-To", config.getMailUser());
            }

            Multipart multiparts = new MimeMultipart();

            // create the message part
            MimeBodyPart msgBodyPart = new MimeBodyPart();
            msgBodyPart.setText(message);
            multiparts.addBodyPart(msgBodyPart);

            // manage pdf attachments
            if (forms != null) {
                File pdfDir = Configuration.getInstance().getPdfDir();

                for (Form form : forms) {
                    File formFile = new File(pdfDir, form.getWikittyId()
                            + ".pdf");

                    if (!formFile.exists()){
                        if (log.isWarnEnabled()){
                            log.warn("No pdf attachment found for form : " + form.getObjet());
                            continue;

                            // EC 20100510 no big deal if only one pdf is missing !
                            //return null;
                        }
                    }
                    
                    // Part two is attachment
                    MimeBodyPart msgCurrentPdfPart = new MimeBodyPart();
                    // default is set to application/octet-stream
                    msgCurrentPdfPart.setHeader("Content-Type", "application/pdf");
                    DataSource source = new FileDataSource(formFile);
                    msgCurrentPdfPart.setDataHandler(new DataHandler(source));
                    msgCurrentPdfPart.setFileName(formFile.getName());
                    multiparts.addBodyPart(msgCurrentPdfPart);
                }
            }
            // Put parts in message
            msg.setContent(multiparts);

            // Setting the Subject and Content Type
            msg.setSubject(subject);

            try {
                Transport.send(msg);
            } catch (MessagingException eee) {
                if (log.isErrorEnabled()) {
                    log.error("Cant send mail", eee);
                }
                
                if (eee.getCause() instanceof SMTPAddressFailedException) {
                    SMTPAddressFailedException exception = (SMTPAddressFailedException)eee.getCause();
                    throw new VradiException(_("SMTP fail to send mail, code returned : %d\n(%s)", exception.getReturnCode(), exception.getLocalizedMessage()), eee);
                }
                
                throw new VradiException("Can't send message", eee);
            }

            if (log.isDebugEnabled()) {
                log.debug("MessageID: " + msg.getMessageID());
            }
            result = msg.getMessageID();

        } catch (MessagingException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't generate message", eee);
            }
            throw new VradiException("Can't generate message", eee);
        } catch (UnsupportedEncodingException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't generate message", eee);
            }
            throw new VradiException("Can't generate message", eee);
        }
        return result;
    }

    /**
     * Receives the emails and check if some are error emails or reception proofs
     * and treat them in consequence.
     * 
     * @throws VradiException
     */
    public void receiveMails() throws VradiException {
        try {
            final Configuration config = Configuration.getInstance();

            // mail from mailer daemon are supposed to be
            // delivery error
            //String mailerDeamonAddress = config.getMailerDaemonEmail();

            // TODO EC-20100505 use Session.getDefaultInstance()
            // and common Properties form receiving/sending mails !
            Properties props = new Properties();
            props.put("mail.imap.host", config.getImapHost());
            props.put("mail.imap.auth", "true");
            props.put("mail.imap.user", config.getMailUser());
            props.put("mail.imap.port", config.getImapPort());

            //props.put("mail.imap.starttls.enable", "true");
            //props.put("mail.imap.socketFactory.port", config.getImapPort());
            //props.put("mail.imap.socketFactory.class", DummySSLSocketFactory.class.getName());
            //props.put("mail.imap.socketFactory.fallback", "false");

            //props.put("mail.imap.ssl.enable", "true");
            //props.put("mail.imap.ssl.port", config.getImapPort());
            //props.put("mail.imap.ssl.trust", "true");
            //props.put("mail.imap.ssl.socketFactory.class", DummySSLSocketFactory.class.getName());
            //props.put("mail.imap.ssl.socketFactory.fallback", "false");

            props.put("mail.store.protocol", "imap");

            Session session = Session.getInstance(props,
                new javax.mail.Authenticator() {
                    @Override
                    protected PasswordAuthentication getPasswordAuthentication() {
                        return new PasswordAuthentication(
                            config.getMailUser(), config.getMailPassword());
                    }
                });

            // Récupère la "messagerie" et se connecte
            Store store = session.getStore("imap");
            store.connect(config.getImapHost(), config.getMailUser(),
                    config.getMailPassword());

            // Récupère le fichier "Boite de réception" et l'ouvre
            Folder folder = store.getFolder("INBOX");
            folder.open(Folder.READ_WRITE);

            FlagTerm flagTerm = new FlagTerm(new Flags(Flags.Flag.SEEN), false);
            Message messages[] = folder.search(flagTerm);

            for (Message message : messages) {
                // pour chaque message, on regarde les messages de type
                // multipart, et plus particulierement leur pieces jointe.
                // different cas apres le parcours de toutes les "part"
                if (message.getContentType().startsWith("multipart/")) {
                    
                    // FIXME EC-20100506 , normalement tous les mails
                    // doivent être marqué comme lu
                    // mais dans le cas ou plusieurs vradi test les mails
                    // qu'il n'ont potentiellement pas envoyé
                    // on ne marque que ceux qu'il ont pu traiter
                    boolean markAsRead = false;
                    
                    if (log.isDebugEnabled()) {
                        log.debug("Testing message " + Arrays.toString(message.getHeader("Message-ID")) + " as multipart");
                    }
                    
                    Multipart multiPart = (Multipart)message.getContent();

                    String errorMessage = null;
                    String errorMessageId = null;
                    String originalMessageID = null;
                    for (int index = 0 ; index < multiPart.getCount() ; ++index) {
                        BodyPart bodyPart = multiPart.getBodyPart(index);
                        if (log.isDebugEnabled()) {
                            log.debug("Part " + index + " : "  + bodyPart.getContentType());
                        }
                        
                        // first delivery failure part
                        if ("message/delivery-status".equals(bodyPart.getContentType())) {
                            // Diagnostic-Code is part of the content
                            String content = getStreamContent(bodyPart.getInputStream());
                            errorMessage = getHeaderValueInContent("Diagnostic-Code", content);
                        }
                        
                        // second delivery failure part
                        if ("text/rfc822-headers".equals(bodyPart.getContentType())) {
                            String content = getStreamContent(bodyPart.getInputStream());
                            errorMessageId = getHeaderValueInContent("Message-ID", content);
                        }
                        
                        // response for reception proof
                        if (bodyPart.getContentType().startsWith("message/disposition-notification")) {
                            String content = getStreamContent(bodyPart.getInputStream());
                            originalMessageID = getHeaderValueInContent("Original-Message-ID", content);
                        }
                    }
                    
                    // si on a eu un un message message/delivery-status et un
                    // message/rfc822
                    if (StringUtils.isNotEmpty(errorMessage) && StringUtils.isNotEmpty(errorMessageId)) {
                        // put this message into file
                        if (log.isInfoEnabled()) {
                            log.info("Error message received for message " + errorMessageId);
                            log.info(" error description : " + errorMessage);
                        }
                        
                        Sending sending = getSendingByMessageID(errorMessageId, proxy);
                        if (sending != null) {
                            sending.setStatus(VradiConstants.SendingStatus.ERROR
                                .getValue());
                            proxy.store(sending);
                            
                            // FIXME EC20100506 mark all mails as read
                            markAsRead = true;
                        }
                        else {
                            if (log.isWarnEnabled()) {
                                log.warn("Can't find sending for message id : " + errorMessageId);
                            }
                        }
                    }
                    
                    // cas de l'accusé de reception
                    else if (StringUtils.isNotEmpty(originalMessageID)) {
                        // put this message into file
                        if (log.isInfoEnabled()) {
                            log.info("Proof notification received for message " + originalMessageID);
                        }
                        
                        Sending sending = getSendingByMessageID(originalMessageID, proxy);
                        if (sending != null) {
                            if (sending.getStatus() == VradiConstants.SendingStatus.WAITING_RECEPTION_PROOF.getValue()) {
                                sending.setStatus(VradiConstants.SendingStatus.RECEIVED.getValue());
                                sending.setReceptionDate(new Date());
                                proxy.store(sending);
                                setValidEmailForUser(sending);
                                
                                // FIXME EC20100506 mark all mails as read
                                markAsRead = true;
                            }
                            else {
                                if (log.isWarnEnabled()) {
                                    log.warn("Received a reception proof for a sending with non reception proof waiting status");
                                }
                            }
                        }
                        else {
                            if (log.isWarnEnabled()) {
                                log.warn("Can't find sending for message id : " + originalMessageID);
                            }
                        }
                    }

                    // mark message as READ
                    // FIXME EC20100506 mark all mails as read
                    message.setFlag(Flags.Flag.SEEN, markAsRead);
                }
                else {
                    if (log.isWarnEnabled()) {
                        log.warn("Mail inbox contains non multipart mails !");
                    }
                }
            }

            folder.close(true);
            store.close();

        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't recieve messages", e);
            }
            throw new VradiException("Can't recieve messages", e);
        }

    }
    
    /**
     * Set {@link User#validEmail} attribute for User associated with sending.
     * 
     * @param sending
     */
    protected void setValidEmailForUser(Sending sending) {
        
        String userId = sending.getUser();
        User user = proxy.restore(User.class, userId);
        
        // update valid email field only is necessary
        if (user != null && !user.getValidEmail()) {
            if (log.isInfoEnabled()) {
                log.info("User email " + user.getEmail() + " has been validated");
            }
            user.setValidEmail(true);
            proxy.store(user);
        }
    }

    /**
     * Convert an input stream to String.
     * 
     * @param inputStream
     * @return
     * @throws IOException 
     */
    protected String getStreamContent(InputStream inputStream) throws IOException {   
        StringBuilder sb = new StringBuilder();
        String line;
        try {
            BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream, "UTF-8"));
            while ((line = reader.readLine()) != null) {
                sb.append(line).append("\n");
            }
        } finally {
            inputStream.close();
        }
        return sb.toString();
    }

    /**
     * Find a header definition in mail content.
     * 
     * Exemple:
     * <pre>
     * Original-Message-ID: <1833582695.1.1272991628810.JavaMail.chatellier@genux>
     * Diagnostic-Code: smtp; 550 5.1.1 <toto@codelutin.com>: Recipient address
     *      rejected: User unknown in local recipient table
     * Message-ID: <804992706.5.1272991629788.JavaMail.chatellier@genux>
     * </pre>
     * @param headerName
     * @param content
     * @return
     */
    protected String getHeaderValueInContent(String headerName, String content) {
        String result = null;
        
        String headerText = headerName + ":";
        int firstIndex = content.indexOf(headerText) + headerText.length();
        int lastIndex = content.indexOf("\n", firstIndex);
        if (lastIndex == -1) {
            // case EOF
            lastIndex = content.length();
        }

        result = content.substring(firstIndex, lastIndex);
        result = result.trim();
        
        // TODO EC-20100505 multilines header
        
        return result;
    }

    /**
     * Find first sending entity with given messageID.
     * 
     * @param messageID message id to find sending
     * @param proxy wikitty proxy
     * @return {@link Sending} or {@code null} if none found
     */
    protected static Sending getSendingByMessageID(String messageID, WikittyProxy proxy) {

        Sending result = null;

        // find sending with wikitty criteria
        Search search = Search.query();
        search.eq(Element.ELT_EXTENSION, Sending.EXT_SENDING).eq(
                Sending.FQ_FIELD_MESSAGEID, messageID);
        Criteria criteria = search.criteria();

        PagedResult<Sending> sendingsResult = proxy.findAllByCriteria(
                Sending.class, criteria);

        List<Sending> sendingList = sendingsResult.getAll();
        if (sendingList.size() == 1) {
            result = sendingList.get(0);
        }

        return result;
    }

    /**
     * Called every periodically to manage new mail on server.
     * 
     * @see java.util.TimerTask#run()
     */
    @Override
    public void run() {

        try {
            if (log.isInfoEnabled()) {
                log.info("Checking for notification and error emails...");
            }
            receiveMails();

        } catch (VradiException eee) {
            if (log.isErrorEnabled()) {
                log.error("Mailing task can't run", eee);
            }
            throw new RuntimeException("Mailing task can't run", eee);
        }
        
    }
    
    
}
