/*
 * #%L
 * SGQ :: Business
 * $Id: ProductDAOImpl.java 400 2013-06-23 12:38:25Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-1.1.1/sgq-business/src/main/java/com/herbocailleau/sgq/entities/ProductDAOImpl.java $
 * %%
 * Copyright (C) 2012 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.entities;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.nuiton.topia.TopiaException;

import com.herbocailleau.sgq.business.model.ProductSearchModel;

public class ProductDAOImpl<E extends Product> extends ProductDAOAbstract<E> {

    /**
     * Extrait des produits les valeurs distinctes des categories de produit.
     * 
     * @return les categories utilisées par les produits
     * @throws TopiaException 
     */
    public List<String> findDistinctCategories() throws TopiaException {
        String query = "select distinct " + Product.PROPERTY_CATEGORY +
            " from " + Product.class.getName() +
            " order by " + Product.PROPERTY_CATEGORY;

        List<String> result = context.findAll(query);
        return result;
    }
    /**
     * Retourne l'ensemble des produits correspondant au filtre.
     * 
     * @param search search model
     * @return
     * @throws TopiaException
     */
    public List<Product> findAllModel(ProductSearchModel search) throws TopiaException {
        return findAllModel(search, 0, -1);
    }

    /**
     * Retourne l'ensemble des produits correspondant au filtre.
     * 
     * @param search search model
     * @param offset offset
     * @param limit limit (use -1 for no limit)
     * @return
     * @throws TopiaException
     */
    public List<Product> findAllModel(ProductSearchModel search, int offset, int limit) throws TopiaException {
        String queryPrefix = "select P";
        String querySuffix = " order by P." + Product.PROPERTY_CODE + " ASC";

        List<Product> results = (List<Product>)performQueryWithFilter(search, queryPrefix, querySuffix, offset, limit);
        return results;
    }

    /**
     * Retourne le nombre de résultats total correspondant au filtre de recherche.
     * 
     * @param search search model
     * @return
     * @throws TopiaException
     */
    public long findAllCount(ProductSearchModel search) throws TopiaException {
        String queryPrefix = "select count(P)";

        Number result = (Number)performQueryWithFilter(search, queryPrefix, "", -1, -1);
        return result.longValue();
    }

    /**
     * Construit la requete réutilisable avec des bout de requete (avant/après)
     * suivant ce que doit effectivement retourner la requete.
     * 
     * @param search le filtre de recherche
     * @param queryPrefix prefix
     * @param querySuffix suffix
     * @param offset offset (use -1 for no limit)
     * @param limit offset (use -1 for no limit)
     * @return le resultat (le type dépend du prefix)
     * @throws TopiaException 
     */
    protected Object performQueryWithFilter(ProductSearchModel search, String queryPrefix,
            String querySuffix, int offset, int limit) throws TopiaException {

        String query = queryPrefix;
        query += " from " + Product.class.getName() + " P";
        query += " where 1=1";

        List<Object> params = new ArrayList<Object>();

        // name field
        if (StringUtils.isNotBlank(search.getName())) {
            query += " and " + BatchDAOImpl.getFieldLikeInsensitive("P.name", ":name");
            params.add("name");
            params.add("%" + StringUtils.stripAccents(search.getName()) + "%");
        }

        // code field
        if (StringUtils.isNotBlank(search.getCode())) {
            query += " and " + BatchDAOImpl.getFieldLikeInsensitive("P.code", ":code");
            params.add("code");
            params.add("%" + StringUtils.stripAccents(search.getCode()) + "%");
        }

        // category
        if (StringUtils.isNotBlank(search.getCategory())) {
            query += " and " + BatchDAOImpl.getFieldLikeInsensitive("P.category", ":category");
            params.add("category");
            params.add("%" + StringUtils.stripAccents(search.getCategory()) + "%");
        }

        // latin name
        if (StringUtils.isNotBlank(search.getLatinName())) {
            query += " and " + BatchDAOImpl.getFieldLikeInsensitive("P.latinName", ":latinName");
            params.add("latinName");
            params.add("%" + StringUtils.stripAccents(search.getLatinName()) + "%");
        }

        // product status
        if (CollectionUtils.isNotEmpty(search.getProductStatus())) {
            int index = 0;
            query += " and (1=1";
            for (ProductStatus status : search.getProductStatus()) {
                if (status == null) { // cas de l'option "vide"
                    query += " AND P.productStatus IS EMPTY";
                } else {
                    query += " AND :status" + index + " member of P.productStatus";
                    params.add("status" + index);
                    params.add(status);
                    index++;
                }
            }

            query += ")";
        }

        // analyze types
        if (CollectionUtils.isNotEmpty(search.getAnalyzeTypes())) {
            int index = 0;
            query += " and (";
            Iterator<AnalyzeType> itAnalyzeType = search.getAnalyzeTypes().iterator();
            while (itAnalyzeType.hasNext()) {
                AnalyzeType analyzeType = itAnalyzeType.next();
                if (analyzeType == null) { // cas de l'option "vide"
                    query += " P." + Product.PROPERTY_ANALYZE_TYPE + " IS EMPTY";
                } else {
                    query += " :analyzeType" + index + " member of P." + Product.PROPERTY_ANALYZE_TYPE;
                    params.add("analyzeType" + index);
                    params.add(analyzeType);
                    index++;
                }
                if (itAnalyzeType.hasNext()) {
                    query += search.isAnalyzeTypeOrOperator() ? " OR" : " AND";
                }
            }

            query += ")";
        }

        // product achived
        if (!search.isArchived()) {
            query += " AND " + Product.PROPERTY_ARCHIVED + " = false";
        }

        // add final suffix to query
        query += querySuffix;

        // perform query
        Object result = null;
        if (offset == -1) {
            result = context.findUnique(query, params.toArray());
        } else if (limit == -1) {
            result = context.findAll(query, params.toArray());
        } else {
            result = context.find(query, offset, offset + limit - 1, params.toArray());
        }
        return result;
    }

    /**
     * Test is product is used by any batch in system.
     * 
     * @param product product to test
     * @return {@code true} is product is used
     * @throws TopiaException 
     */
    public boolean isUsedByAnyBatch(Product product) throws TopiaException {
        String query = "select count(*) FROM " + ProductPlace.class.getName() +
                " where " + ProductPlace.PROPERTY_PRODUCT + " = :product";

        Number result = (Number)context.findUnique(query, "product", product);
        
        if (result.longValue() == 0) {
            query = "select count(*) FROM " + Batch.class.getName() +
                " where " + Batch.PROPERTY_PRODUCT + " = :product";
            result = (Number)context.findUnique(query, "product", product);
        }

        return result.longValue() > 0;
    }

    /**
     * Test is product is used by non expired batch.
     * 
     * @param product product to test
     * @return {@code true} is product is used
     * @throws TopiaException 
     */
    public boolean isUsedByNonExpiredBatch(Product product) throws TopiaException {
        String query = "select count(*) FROM " + Batch.class.getName() +
            " where " + Batch.PROPERTY_PRODUCT + " = :product" +
            " AND " + Batch.PROPERTY_EXPIRED_DATE + " = null";
        Number result = (Number)context.findUnique(query, "product", product);

        return result.longValue() > 0;
    }

} //ProductDAOImpl<E extends Product>
