package com.cybelia.sandra;

import com.cybelia.sandra.entities.Camion;
import com.cybelia.sandra.entities.Eleveur;
import com.cybelia.sandra.entities.Etape;
import com.cybelia.sandra.entities.InfoAccess;
import com.cybelia.sandra.entities.Tour;
import com.cybelia.sandra.entities.UserIndicateurs;
import com.cybelia.sandra.entities.UserIndicateursDAO;
import com.cybelia.sandra.entities.notifier.Queue;
import com.cybelia.sandra.entities.notifier.QueueDAO;
import com.cybelia.sandra.ibu.IbuFileComparator;
import com.cybelia.sandra.ibu.MigrationContext;
import com.cybelia.sandra.ibu.csv.bean.Ibu;
import com.cybelia.sandra.ibu.csv.reader.CSVReaderIbu;
import com.cybelia.sandra.notifier.SandraNotifier;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.io.FileExistsException;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.taas.entities.TaasUser;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.FileUtil;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.StatefulJob;

import static com.cybelia.sandra.ibu.csv.CSVReaderGeneric.DELIM;
import static org.nuiton.i18n.I18n._;

/**
 * @author sletellier
 */
public class SchedulerJobFactory {

    /**
     * Logger.
     */
    public static final Log log = LogFactory.getLog(SchedulerJobFactory.class);

    public enum SchedulerJobClass {
        FIND_UNCONNECTED_CAMION(FindUnconnectedCamion.class, FindUnconnectedCamion.PATTERN),
        NO_COORD_GPS_VERIFIER(NoCoordGPSVerfier.class, NoCoordGPSVerfier.PATTERN),
        NO_LOCKED_GPS_VERIFIER(NotLockedGPSVerifier.class, NotLockedGPSVerifier.PATTERN),
        CLEAN_UP_NOTIFICATION(CleanUpNotification.class, CleanUpNotification.PATTERN),
        RAZ_NUMBER_OF_CONNECTION(RazNumberOfConnection.class, RazNumberOfConnection.PATTERN),
        SEND_STATS_NOTIFICATION(SendStatsNotification.class, SendStatsNotification.PATTERN),
        FIND_DANGEROUS_BREEDER(FindDangerousBreeder.class, FindDangerousBreeder.PATTERN),
        IBU_VERIFIER(IbuVerifier.class, IbuVerifier.PATTERN);

        protected Class<? extends SchedulerJob> schedulerClass;
        protected String pattern;

        SchedulerJobClass(Class<? extends SchedulerJob> schedulerClass, String pattern) {
            this.schedulerClass = schedulerClass;
            this.pattern = pattern;
        }

        public Class<? extends SchedulerJob> getSchedulerClass() {
            return schedulerClass;
        }

        public String getPattern() {
            return pattern;
        }
    }

    public static abstract class SchedulerJob implements StatefulJob {

        public abstract void doAction(TopiaContext transaction) throws Exception;

        public abstract String getPattern();

        @Override
        public void execute(JobExecutionContext context) throws JobExecutionException {
            TopiaContext transaction = null;
            try {
                transaction = beginTransaction();

                doAction(transaction);

            } catch (Exception eee) {
                log.error("Failed to execute scheduler " + _(getPattern()), eee);
            } finally {
                try {
                    closeTransaction(transaction);
                } catch (TopiaException eee) {
                    log.error("Failed to close transaction", eee);
                }
            }
        }
    }

    public static class FindUnconnectedCamion extends SchedulerJob {

        public static final String PATTERN = "sandra.scheduler.unconnectedcamion";

        @Override
        public void doAction(TopiaContext transaction) throws Exception {
            log.info("findUnconnectedCamion");

            // Creation de la date -3j
            Date now = new Date();
            Calendar cLess3 = Calendar.getInstance();

            cLess3.setTime(now);
            cLess3.add(Calendar.DAY_OF_YEAR, -3);

            List<Camion> camions = getAllCamionLessSynchDate(transaction, cLess3.getTime(), true);

            if (!camions.isEmpty()) {

                // Sort camion with code
                Collections.sort(camions, CAMION_CODE_COMPARATOR);

                // Envoie de la notification
                String camionUnconnectedString = SandraHelper.convertEntitiesToListForNotifier(camions);

                // Send notification
                new SandraNotifier().notifyEvent(
                        "unconnectedCamion",
                        SandraSchedulerConfigHelper.getSandraName(SandraConfig.getConfig()),
                        SandraSchedulerConfigHelper.getSandraUrl(SandraConfig.getConfig()),
                        camionUnconnectedString);
            } else {
                log.warn("No camion connected last 3 days, abording sending notification");
            }
        }

        @Override
        public String getPattern() {
            return PATTERN;
        }
    }

    public static class NoCoordGPSVerfier extends SchedulerJob {

        public static final String PATTERN = "sandra.scheduler.findnocoordgps";

        @Override
        public void doAction(TopiaContext transaction) throws Exception {


            log.info("getAllEleveurWhithNoGPS");
            List<String[]> datas = new ArrayList<String[]>();

            // Recuperation des eleveurs sans GPS
            String hql = "SELECT eleveur FROM " + Eleveur.class.getName() + " eleveur" +
                    " WHERE tomtomgps IS null";

            List<Eleveur> eleveurs = transaction.findAll(hql);

            // Recuperation de la derniere livraison
            hql = "SELECT etape.topiaId, etape.eleveur.topiaId, etape.tour.topiaId, tour.camion.topiaId" +
                    " FROM " + Tour.class.getName() + " tour, " +
                    Etape.class.getName() + " etape, " +
                    Camion.class.getName() + " camion" +
                    " WHERE tour = etape.tour" +
                    " AND tour.camion = camion" +
                    " AND etape.eleveur = :eleveur" +
                    " AND camion.code != 'GUI'" +
                    " AND tour.dateLivraison >= :date" +
                    " GROUP BY camion.code, etape.topiaId, etape.eleveur.topiaId," +
                    " etape.tour.topiaId, tour.camion.topiaId, tour.dateLivraison" +
                    " ORDER BY camion.code, etape.eleveur.topiaId, tour.dateLivraison DESC";

            // Pour tous les eleveurs sans coordonees GPS
            for (Eleveur eleveur : eleveurs) {
                List<Object[]> result = transaction.find(hql, 0, 0, "eleveur", eleveur, "date", SandraHelper.getLastWeekDate());

                if (!result.isEmpty()) {
                    datas.add(SandraHelper.convertToStringArray(result.get(0)));
                }
            }
            if (log.isInfoEnabled()) {
                log.info("Found " + datas.size() + " eleveurs with no GPS");
            }

            List<String[]> eleveurNoGPS = sortDataByCodeCamion(transaction, datas);
            // etape.topiaid, etape.eleveur, etape.tour, tour.camion

            log.info("Sending noCoordGPS : " + eleveurNoGPS.size());

            if (!eleveurNoGPS.isEmpty()) {

                // Send notification
                EmailHelper.sendNotification("noCoordGPS", eleveurNoGPS);
            }
        }

        @Override
        public String getPattern() {
            return PATTERN;
        }
    }

    public static class NotLockedGPSVerifier extends SchedulerJob {

        public static final String PATTERN = "sandra.scheduler.notLockedCoordGPS";

        @Override
        public void doAction(TopiaContext transaction) throws Exception {
            log.info("getAllEleveurWhithNotLockedGPS");

            List<String[]> datas = new ArrayList<String[]>();

            // Recuperation des eleveurs sans GPS
            String hql = "select eleveur from " + Eleveur.class.getName() + " eleveur" +
                    " WHERE tomtomgps IS NOT NULL AND nbtomtomgpsmodif = 0";

            List<Eleveur> eleveurs = transaction.findAll(hql);

            // Recuperation de la derniere livraison
            hql = "SELECT etape.topiaId, etape.eleveur.topiaId, etape.tour.topiaId, tour.camion.topiaId" +
                " FROM " + Tour.class.getName() + " tour, " +
                Etape.class.getName() + " etape, " +
                Camion.class.getName() + " camion" +
                " WHERE tour = etape.tour" +
                " AND tour.camion = camion" +
                " AND etape.eleveur = :eleveur" +
                " AND camion.code != 'GUI'" +
                " AND tour.dateLivraison >= :date" +
                " GROUP BY camion.code, etape, etape.eleveur.topiaId, etape.tour.topiaId," +
                " tour.camion.topiaId, tour.camion.topiaId, tour.dateLivraison" +
                " ORDER BY camion.code, etape.eleveur.topiaId, tour.dateLivraison DESC";

            // Pour tous les eleveurs sans coordonees GPS
            for (Eleveur eleveur : eleveurs) {
                List<Object[]> result = transaction.find(hql, 0, 0, "eleveur", eleveur, "date", SandraHelper.getLastWeekDate());
                if (!result.isEmpty()) {
                    datas.add(SandraHelper.convertToStringArray(result.get(0)));
                }
            }
            if (log.isInfoEnabled()) {
                log.info("Found " + datas.size() + " eleveurs with not locked GPS");
            }
            List<String[]> eleveurGPSButNotLocked = sortDataByCodeCamion(transaction, datas);

            log.info("Sending coordGPSExistingButNotLocked : " + eleveurGPSButNotLocked.size());

            if (!eleveurGPSButNotLocked.isEmpty()) {

                // Send notification
                EmailHelper.sendNotification("coordGPSExistingButNotLocked", eleveurGPSButNotLocked);
            }
        }

        @Override
        public String getPattern() {
            return PATTERN;
        }
    }

    public static class CleanUpNotification extends SchedulerJob {

        public static final String PATTERN = "sandra.scheduler.cleanup";

        @Override
        public void doAction(TopiaContext transaction) throws Exception {
            QueueDAO queueDAO = SandraDAOHelper.getQueueDAO(transaction);
            List<Queue> queues = queueDAO.findAll();

            if (log.isDebugEnabled()) {
                log.debug("Deleting " + queues.size() + " queues entry");
            }

            for (Queue q : queues) {
                if (log.isDebugEnabled()) {
                    log.debug("deleting queue : " + q.getTopiaId());
                }
                queueDAO.delete(q);
            }
        }

        @Override
        public String getPattern() {
            return PATTERN;
        }
    }

    public static class RazNumberOfConnection extends SchedulerJob {

        public static final String PATTERN = "sandra.scheduler.RAZNumberOfConnection";

        @Override
        public void doAction(TopiaContext transaction) throws Exception {
            log.info("RAZNumberOfConnection");

            // Send stats before RAZ
            new SendStatsNotification().doAction(transaction);

            StringBuilder hqlUSerIndicator = new StringBuilder();
            hqlUSerIndicator.append("UPDATE ");
            hqlUSerIndicator.append(UserIndicateurs.class.getName());
            hqlUSerIndicator.append(" SET ");
            hqlUSerIndicator.append(UserIndicateurs.PROPERTY_NB_MAJ);
            hqlUSerIndicator.append("=0, ");
            hqlUSerIndicator.append(UserIndicateurs.PROPERTY_NB_GPS_MAJ);
            hqlUSerIndicator.append("=0, ");
            hqlUSerIndicator.append(UserIndicateurs.PROPERTY_NB_SECU_MAJ);
            hqlUSerIndicator.append("=0, ");
            hqlUSerIndicator.append(UserIndicateurs.PROPERTY_NB_SYNCH_GPRS);
            hqlUSerIndicator.append("=0, ");
            hqlUSerIndicator.append(UserIndicateurs.PROPERTY_NB_SYNCH_WIFI);
            hqlUSerIndicator.append("=0, ");
            hqlUSerIndicator.append(UserIndicateurs.PROPERTY_NB_SYNCH_KO);
            hqlUSerIndicator.append("=0, ");
            hqlUSerIndicator.append(UserIndicateurs.PROPERTY_NB_SYNCH_AUTO);
            hqlUSerIndicator.append("=0, ");
            hqlUSerIndicator.append(UserIndicateurs.PROPERTY_NB_NOTIFS);
            hqlUSerIndicator.append("=0, ");
            hqlUSerIndicator.append(UserIndicateurs.PROPERTY_NB_OCTETS_RECEIVED_GPRS);
            hqlUSerIndicator.append("=0, ");
            hqlUSerIndicator.append(UserIndicateurs.PROPERTY_NB_OCTETS_SEND_GPRS);
            hqlUSerIndicator.append("=0, ");
            hqlUSerIndicator.append(UserIndicateurs.PROPERTY_NB_OCTETS_RECEIVED_WIFI);
            hqlUSerIndicator.append("=0, ");
            hqlUSerIndicator.append(UserIndicateurs.PROPERTY_NB_OCTETS_SEND_WIFI);
            hqlUSerIndicator.append("=0");

            int result = transaction.execute(hqlUSerIndicator.toString());
            log.info(result + " UserIndicateurs raz");

            StringBuilder hqlUser = new StringBuilder();
            hqlUser.append("UPDATE ");
            hqlUser.append(TaasUser.class.getName());
            hqlUser.append(" SET ");
            hqlUser.append(TaasUser.NUMBER_OF_CONNECTION);
            hqlUser.append("=0");

            result = transaction.execute(hqlUser.toString());
            log.info(result + " user raz");
        }

        @Override
        public String getPattern() {
            return PATTERN;
        }
    }

    public static class SendStatsNotification extends SchedulerJob {

        public static final String PATTERN = "sandra.scheduler.sendStatsNotification";
        public static final String CSV_HEADER =
                "Camion" + DELIM +
                "Clients livrés" + DELIM +
                "Géopositionnés" + DELIM +
                "Géopositionnés et verrouillés" + DELIM +
                "Niveau de sécurité non renseigné" + DELIM +
                "Niveau de sécurité vert" + DELIM +
                "Niveau de sécurité orange" + DELIM +
                "Niveau de sécurité rouge" + DELIM +
                "Niveau de sécurité noir" + DELIM +
                "Renseigné pour accès" + DELIM +
                "Renseigné pour équipement" + DELIM +
                "Silo livrés" + DELIM +
                "Silo géopositionnés" + DELIM +
                "Silo géopositionnés et verrouillés" + DELIM +
                "Silo niveau de sécurité non renseigné" + DELIM +
                "Silo niveau de sécurité vert" + DELIM +
                "Silo niveau de sécurité orange" + DELIM +
                "Silo niveau de sécurité rouge" + DELIM +
                "Silo niveau de sécurité noir" + DELIM +
                "Silo renseigné pour accès" + DELIM +
                "Silo renseigné pour équipement" + DELIM +
                "Dernière synchro en wifi" + DELIM +
                "Synchros en automatique" + DELIM +
                "Notifications" + DELIM +
                "Synchro complète en gprs" + DELIM +
                "Maj de coord GPS" + DELIM +
                "Maj de niveau de sécurité" + DELIM +
                "Maj réalisées";

        @Override
        public void doAction(TopiaContext transaction) throws Exception {
            Date dateLessOneMonth = DateUtils.addMonths(new Date(), -1);

            // Recuperation des camions concernes trie par societe et par code camion
            List<Camion> allCamionLessSynchDate =
                    getAllCamionLessSynchDate(transaction,
                                              dateLessOneMonth,
                                              false,
                                              Camion.PROPERTY_SOCIETE,
                                              Camion.PROPERTY_CODE);

            log.info("allCamionLessNoTourFromDate size : " + allCamionLessSynchDate.size());

            // Recuperation des statistiques
            ListMultimap<UserIndicateurs, Integer> stats = getAllStats(transaction, allCamionLessSynchDate);

            log.info("stats size : " + stats.size());

            // Building csv
            StringBuilder builder = new StringBuilder();
            builder.append(CSV_HEADER).append("\n");

            // Iter on camions
            for (UserIndicateurs userIndicateurs : stats.keySet()) {
                builder.append(userIndicateurs.getTaasUser().getLogin()).append(DELIM);
                List<Integer> integers = stats.get(userIndicateurs);
                builder.append(StringUtils.join(integers, DELIM));

                SimpleDateFormat dateLivraisonPatern = new SimpleDateFormat("dd/MM/yyyy");
                Date lastSynch = userIndicateurs.getLastSynch();
                String lastSynchAsString = "jamais";
                if (lastSynch != null) {
                    lastSynchAsString = dateLivraisonPatern.format(lastSynch);
                }
                builder.append(DELIM);
                builder.append(lastSynchAsString);
                builder.append(DELIM);
                builder.append(userIndicateurs.getNbSynchAuto());
                builder.append(DELIM);
                builder.append(userIndicateurs.getNbNotifs());
                builder.append(DELIM);
                builder.append(userIndicateurs.getNbSynchGPRS());
                builder.append(DELIM);
                builder.append(userIndicateurs.getNbGpsMaj());
                builder.append(DELIM);
                builder.append(userIndicateurs.getNbSecuMaj());
                builder.append(DELIM);
                builder.append(userIndicateurs.getNbMaj());
                builder.append("\n");
            }

            // Envoi de la notification
            ApplicationConfig config = SandraConfig.getConfig();
            new SandraNotifier().notifyEvent("userIndicatorStats",
                    SandraSchedulerConfigHelper.getSandraName(config),
                    SandraSchedulerConfigHelper.getSandraUrl(config),
                    builder.toString());
        }

        @Override
        public String getPattern() {
            return PATTERN;
        }
    }

    public static class FindDangerousBreeder extends SchedulerJob {

        public static final String PATTERN = "sandra.scheduler.findDangerousBreeder";

        @Override
        public void doAction(TopiaContext transaction) throws Exception {

            // Recuperation des eleveurs dangereux
            String hql = "SELECT DISTINCT eleveur.topiaId FROM " + Eleveur.class.getName() + " eleveur, " + InfoAccess.class.getName() +
                    " access WHERE access.niveauSecurite > 1 " +
                    "AND (access.eleveur = eleveur.topiaId OR access.topiaId = eleveur.accesEleveur)" +
                    " GROUP BY access.niveauSecurite, eleveur.topiaId" +
                    " ORDER BY eleveur.topiaId";

            List<String> dangerousBreedersAndSilos = transaction.findAll(hql);

            // Convertie la list en string interpretable par le service de notification
            String dangerousBreedersAndSilosString = SandraHelper.convertToListForNotifier(dangerousBreedersAndSilos);

            // Envoi de la notification
            ApplicationConfig config = SandraConfig.getConfig();
            new SandraNotifier().notifyEvent("dangerousBreeder",
                    SandraSchedulerConfigHelper.getSandraName(config),
                    SandraSchedulerConfigHelper.getSandraUrl(config),
                    dangerousBreedersAndSilosString);
        }

        @Override
        public String getPattern() {
            return PATTERN;
        }
    }

    public static class IbuVerifier extends SchedulerJob {

        public static final String PATTERN = "sandra.scheduler.ibuVerifier";

        @Override
        public void doAction(TopiaContext transaction) throws Exception {

            // Starting ibu verifier
            // Recuperation des repertoires de travails
            ApplicationConfig config = SandraConfig.getConfig();
            String ibuDonePath = SandraSchedulerConfigHelper.getIbuDonePath(config);
            String ibuOldPath = SandraSchedulerConfigHelper.getIbuOldPath(config);

            String ibuPatternFile = SandraSchedulerConfigHelper.getPatternFile(config);

            // Recupertation des fichiers
            File ibuDonePathFile = new File(ibuDonePath);
            List<File> files = FileUtil.find(ibuDonePathFile, ibuPatternFile, false);
            Collections.sort(files, new IbuFileComparator(ibuPatternFile));

            ListMultimap<String, String> commandeInError = ArrayListMultimap.create();

            boolean inError = false;
            for (File f : files) {
                String fileName = f.getName();

                // Move file to old
                String old = ibuOldPath + File.separator + fileName;
                File oldFile = new File(old);
                try {
                    FileUtils.moveFile(f, oldFile);
                } catch (FileExistsException eee) {
                    log.warn("File '" + fileName + "' is already in olds");

                    // Delete existing before move
                    FileUtils.deleteQuietly(oldFile);
                    FileUtils.moveFile(f, oldFile);
                }

                boolean fileInError = false;

                try {

                    log.info("Verifing : " + fileName);

                    // Verify
                    Reader fileReader = new FileReader(oldFile);
                    CSVReaderIbu csvReaderIbu = new CSVReaderIbu();
                    List<Ibu> ibus = csvReaderIbu.parse(new MigrationContext(), fileReader);
                    for (Ibu ibu : ibus) {

                        if (ibu != null) {

                            // Get all criteria to verify
                            String commandeCode = ibu.getCommandeCode();
                            int produitOrdreChargement = ibu.getProduitOrdreChargement();
                            int produitOrdre = ibu.getProduitOrdre();
                            Date tourDateLivraison = ibu.getTourDateLivraison();
                            String camionCode = ibu.getCamionCode();
                            int tourNumero = ibu.getTourNumero();
                            String usineCode = ibu.getUsineCode();

                            // Verify that number exist in base
                            String hql = "SELECT count(etape) FROM " + Tour.class.getName() + " tour " +
                                         "JOIN tour.camion AS camion " +
                                         "JOIN tour.etapes AS etape " +
                                         "JOIN etape.produits AS produit " +
                                         "JOIN produit.commande AS commande " +
                                         "JOIN produit.usine AS usine WHERE " +
                                         "commande.numero = :commandeCode AND " +
                                         "etape.ordre = :produitOrdreChargement AND " +
                                         "produit.ordre = :produitOrdre AND " +
                                         "tour.dateLivraison = :tourDateLivraison AND " +
                                         "camion.code = :camionCode AND " +
                                         "tour.numero = :tourNumero AND " +
                                         "usine.code = :usineCode";

                            Object[] args = new Object[]{
                                    "commandeCode", commandeCode,
                                    "produitOrdreChargement", produitOrdreChargement,
                                    "produitOrdre", produitOrdre,
                                    "tourDateLivraison", tourDateLivraison,
                                    "camionCode", camionCode,
                                    "tourNumero", tourNumero,
                                    "usineCode", usineCode};

                            List result = transaction.findAll(hql, args);

                            Object o = result.get(0);
                            int etapesFound = Integer.valueOf("" + o);

                            if (etapesFound == 0) {
                                log.warn("Commande " + commandeCode + " is in error on file : " + fileName);

                                if (log.isDebugEnabled()) {
                                    log.debug("query : " + hql);
                                    log.debug("Args : " + Arrays.toString(args));
                                }

                                String value = ibu.getSocieteCode() + " - " + commandeCode;
                                if (!commandeInError.containsValue(value)) {
                                    commandeInError.put(fileName, value);
                                }

                                // Mark as error
                                inError = true;
                                fileInError = true;
                            }
                        }
                    }
                } catch (Exception eee) {
                    log.error("Failed to verify : " + fileName, eee);
                } finally {

                    // If file is in error, move it to error
                    if (fileInError) {
                        String ibuVerifierErrorPath = SandraSchedulerConfigHelper.getIbuVerifierErrorPath(config);
                        String errorPath = ibuVerifierErrorPath + File.separator + fileName + "_verifier.err";
                        try {
                            // Move file in error path
                            FileUtils.moveFile(oldFile, new File(errorPath));
                        } catch (Exception eee) {
                            log.error("Failed to move '" + oldFile.getPath() + "' to '" + errorPath);

                            // Delete existing before move
                            FileUtils.deleteQuietly(oldFile);
                        }
                    }
                }
            }

            // If one or more files are in error, send mail
            if (inError) {

                // Send notification
                String filesInErrorForNotifier = SandraHelper.convertToListForNotifier(commandeInError.keySet());
                String commandeInErrorForNotifier = SandraHelper.convertToListOfListForNotifier(commandeInError);

                log.info("Mail will be send because some files are in error : " + filesInErrorForNotifier);

                new SandraNotifier().notifyEvent("commandesInError",
                        SandraSchedulerConfigHelper.getSandraName(config),
                        filesInErrorForNotifier,
                        commandeInErrorForNotifier);

            }
        }

        @Override
        public String getPattern() {
            return PATTERN;
        }
    }

    protected static TopiaContext beginTransaction() throws TopiaException, IOException {
        TopiaContext ibuRootContext = SandraSchedulerConfigHelper.getSchedulerRootContext();
        return ibuRootContext.beginTransaction();
    }

    protected static void closeTransaction(TopiaContext transaction) throws TopiaException {
        if (transaction != null && !transaction.isClosed()) {
            transaction.commitTransaction();
            transaction.closeContext();
        }
    }

    public static final Comparator<Camion> CAMION_CODE_COMPARATOR = new Comparator<Camion>() {

        @Override
        public int compare(Camion c1, Camion c2) {
            return c1.getCode().compareTo(c2.getCode());
        }
    };

    public static final Comparator<TaasUser> USER_CODE_COMPARATOR = new Comparator<TaasUser>() {

        @Override
        public int compare(TaasUser u1, TaasUser u2) {
            return u1.getLogin().compareTo(u2.getLogin());
        }
    };

    protected static List<String[]> sortDataByCodeCamion(TopiaContext transaction, List<String[]> datas) throws TopiaException {
        List<String[]> datasTrie = new ArrayList<String[]>();

        // Recuperation des camions
        List<Camion> camions = new ArrayList<Camion>();
        for (String[] data : datas) {
            String camionTopiaId = data[3];
            camions.add(SandraDAOHelper.getCamionDAO(transaction).findByTopiaId(camionTopiaId));
        }

        // Trie par code camion
        Collections.sort(camions, CAMION_CODE_COMPARATOR);

        // Trie des donnee par camions
        for (Camion camion : camions) {
            List<String[]> datasCopy = new ArrayList<String[]>(datas);
            for (String[] data : datasCopy) {
                String camionTopiaId = data[3];

                if (camionTopiaId.equals(camion.getTopiaId())) {
                    datasTrie.add(data);
                    datas.remove(data);
                }
            }
        }
        return datasTrie;
    }

    protected static Map<TaasUser, UserIndicateurs> getAllUserIndicateur(TopiaContext context, List<TaasUser> users) throws TopiaException {
        log.info("getAllUserIndicateur");

        Map<TaasUser, UserIndicateurs> result = new HashMap<TaasUser, UserIndicateurs>();

        // Get daos
        UserIndicateursDAO userIndicateursDAO = SandraDAOHelper.getUserIndicateursDAO(context);

        for (TaasUser user : users) {

            UserIndicateurs userIndicateurs = userIndicateursDAO.findByTaasUser(user);

            result.put(user, userIndicateurs);
        }
        return result;
    }

    protected static ListMultimap<UserIndicateurs, Integer> getAllStats(TopiaContext context, List<Camion> camions) throws TopiaException {
        log.info("getAllStats");

        ListMultimap<UserIndicateurs, Integer> result = ArrayListMultimap.create();
        int allCount = camions.size();
        int cnt = 0;
        for (Camion camion : camions) {
            cnt++;
            log.info("Getting all stats for " + camion.getUserIndicateurs().getTaasUser().getLogin() + " : " + cnt + "/" + allCount);
            UserIndicateursStats stats = new UserIndicateursStats(camion);
            List<Integer> value = stats.aquireDatas(context);
            if (value != null) {
                result.putAll(camion.getUserIndicateurs(), value);
            }
        }
        log.info("End of getAllStats");

        return result;
    }

    protected static List<Camion> getAllCamionLessSynchDate(TopiaContext context, Date dateLess, String ... orderBy) throws TopiaException {
        return getAllCamionLessSynchDate(context, dateLess, false, orderBy);
    }

    protected static List<Camion> getAllCamionLessSynchDate(TopiaContext context, Date dateLess, boolean inverse, String ... orderBy) throws TopiaException {

        log.info("getAllCamionLessSynchDate");

        String hql = "SELECT camion FROM " +
               Camion.class.getName() + " camion" +
                " JOIN camion.userIndicateurs AS userIndicateurs" +
                " WHERE camion.actif =" + true;
        if (inverse) {
            hql += " AND userIndicateurs.lastSynch < :dateLess";
        } else {
            hql += " AND userIndicateurs.lastSynch >= :dateLess";
        }

        if (orderBy != null && orderBy.length > 0) {
            hql += " ORDER BY " + StringUtils.join(orderBy, ",");
        }

        List<Camion> camions = context.findAll(hql, "dateLess", dateLess);

        log.info("Total camions founds " + camions.size());
        return camions;
    }

}
