package com.franciaflex.faxtomail.web.job;

/*
 * #%L
 * FaxToMail :: Web
 * $Id: MailFilterJob.java 662 2014-10-03 09:14:56Z kmorin $
 * $HeadURL: http://svn.codelutin.com/faxtomail/tags/faxtomail-1.0/faxtomail-ui-web/src/main/java/com/franciaflex/faxtomail/web/job/MailFilterJob.java $
 * %%
 * Copyright (C) 2014 Mac-Groupe, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import gui.ava.html.Html2Image;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.List;
import java.util.Properties;
import java.util.regex.Pattern;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.mail.Address;
import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Part;
import javax.mail.Session;
import javax.mail.Store;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.MimeUtility;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.decorator.Decorator;
import org.nuiton.util.StringUtil;
import org.quartz.DisallowConcurrentExecution;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.xhtmlrenderer.pdf.ITextRenderer;

import com.franciaflex.faxtomail.FaxToMailApplicationContext;
import com.franciaflex.faxtomail.FaxToMailConfiguration;
import com.franciaflex.faxtomail.persistence.entities.Attachment;
import com.franciaflex.faxtomail.persistence.entities.AttachmentFile;
import com.franciaflex.faxtomail.persistence.entities.AttachmentFileImpl;
import com.franciaflex.faxtomail.persistence.entities.AttachmentImpl;
import com.franciaflex.faxtomail.persistence.entities.Client;
import com.franciaflex.faxtomail.persistence.entities.DemandStatus;
import com.franciaflex.faxtomail.persistence.entities.Email;
import com.franciaflex.faxtomail.persistence.entities.EmailAccount;
import com.franciaflex.faxtomail.persistence.entities.EmailImpl;
import com.franciaflex.faxtomail.persistence.entities.ExtensionCommand;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailTopiaPersistenceContext;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailUser;
import com.franciaflex.faxtomail.persistence.entities.MailFilter;
import com.franciaflex.faxtomail.persistence.entities.MailFolder;
import com.franciaflex.faxtomail.services.DecoratorService;
import com.franciaflex.faxtomail.services.FaxToMailServiceContext;
import com.franciaflex.faxtomail.services.FaxToMailServiceUtils;
import com.franciaflex.faxtomail.services.service.ClientService;
import com.franciaflex.faxtomail.services.service.ConfigurationService;
import com.franciaflex.faxtomail.services.service.EmailService;
import com.franciaflex.faxtomail.services.service.MailFolderService;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.PageSize;
import com.itextpdf.text.Paragraph;
import com.itextpdf.text.pdf.PdfWriter;

/**
 * @author Kevin Morin (Code Lutin)
 */
@DisallowConcurrentExecution
public class MailFilterJob extends AbstractFaxToMailJob {

    private static final Log log = LogFactory.getLog(MailFilterJob.class);

    protected FaxToMailApplicationContext applicationContext;

    protected FaxToMailConfiguration config;

    protected ConfigurationService configurationService;

    protected EmailService emailService;

    protected MailFolderService mailFolderService;

    protected ClientService clientService;

    protected DecoratorService decoratorService;

    @Override
    public void execute(JobExecutionContext jobExecutionContext) throws JobExecutionException {
        applicationContext = getApplicationContext(jobExecutionContext);
        FaxToMailTopiaPersistenceContext persistenceContext = null; 
        

        try {
            persistenceContext = applicationContext.newPersistenceContext();
            FaxToMailServiceContext serviceContext = applicationContext.newServiceContext(persistenceContext);
            config = serviceContext.getApplicationConfig();
        
            if (log.isInfoEnabled()) {
                log.info("Running MailFilterJob at " + serviceContext.getNow());
            }

            emailService = serviceContext.getEmailService();
            mailFolderService = serviceContext.getMailFolderService();
            configurationService = serviceContext.getConfigurationService();
            clientService = serviceContext.getClientService();
            decoratorService = serviceContext.getDecoratorService();

            Collection<EmailAccount> emailAccounts = configurationService.getEmailAccounts();
    
            for (EmailAccount account : emailAccounts) {
                checkEmails(account);
            }
            
            if (log.isDebugEnabled()) {
                log.debug("MailFilterJob ended at " + serviceContext.getNow());
            }
        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't run quartz job", ex);
            }
        } finally {
            if (persistenceContext != null) {
                persistenceContext.close();
            }
        }
    }

    /**
     * Checks the emails of the account
     * @param account
     */
    public void checkEmails(EmailAccount account) {
        Properties properties = new Properties();
        
        switch (account.getProtocol()) {
        case IMAPS:
            properties.setProperty("mail.imap.ssl.enable", "true");
        case IMAP:
            properties.setProperty("mail.store.protocol", "imap");
            properties.setProperty("mail.imap.host", account.getHost());
            properties.setProperty("mail.imap.port", String.valueOf(account.getPort()));
            properties.setProperty("mail.imap.connectiontimeout", "2000");
            break;
        case POP3S:
            properties.setProperty("mail.pop3.ssl.enable", "true");
        case POP3:
            properties.setProperty("mail.store.protocol", "pop3");
            properties.setProperty("mail.pop3.host", account.getHost());
            properties.setProperty("mail.pop3.port", String.valueOf(account.getPort()));
            properties.setProperty("mail.pop3.connectiontimeout", "2000");
            break;
        }

        Session session = Session.getInstance(properties);
        Store store = null;
        Folder defaultFolder = null;
        Folder inbox = null;

        try {
            store = session.getStore();
            store.connect(account.getLogin(), account.getPassword());
            defaultFolder = store.getDefaultFolder();
            inbox = defaultFolder.getFolder("INBOX");
            checkEmailsOfFolder(account, inbox);

        } catch (Exception e) {
            log.error("Error while getting emails from the mailbox", e);

        } finally {
            close(inbox);
            close(defaultFolder);
            try {
                if (store != null && store.isConnected()) {
                    store.close();
                }
            } catch (MessagingException e) {
                log.error("Error while closing the store", e);
            }
        }

    }

    protected void close(Folder folder) {
        if (folder != null && folder.isOpen()) {
            try {
                boolean expunge = config.isMailExpunge();
                folder.close(expunge);
            } catch (Exception e) {
                log.error("Error while closing the folder", e);
            }
        }
    }

    /**
     * Check the emails of teh folder, create the emails in the database and delete the email in the folder.
     * 
     * @param emailAccount email account currently checked
     * @param folder the folder to check
     */
    protected void checkEmailsOfFolder(EmailAccount emailAccount, Folder folder) {
        
        int importedCount = 0;

        try {
            folder.open(Folder.READ_WRITE);

            int count = folder.getMessageCount();
            int unread = folder.getUnreadMessageCount();

            if (log.isDebugEnabled()) {
                log.debug(count + " messages, " + unread + " unread");
            }

            for (int i = 0 ; i < count ; i++) {
                try {
                    Email email = new EmailImpl();
                    email.setFax(emailAccount.isFaxAccountType());

                    int messageNumber = count - i;
                    Message message = folder.getMessage(messageNumber);

                    Charset charset = FaxToMailServiceUtils.getCharset(message);

                    List<String> modifiedProperties = new ArrayList<>();

                    if (log.isDebugEnabled()) {
                        log.debug(String.format("Message %d/%d : %s", i, count,  message.getSubject()));
                    }

                    List<Address> recipients = new ArrayList<>();
                    Address[] toRecipients = message.getRecipients(Message.RecipientType.TO);
                    if (toRecipients != null) {
                        recipients.addAll(Arrays.asList(toRecipients));
                    }
                    Address[] ccRecipients = message.getRecipients(Message.RecipientType.CC);
                    if (ccRecipients != null) {
                        recipients.addAll(Arrays.asList(ccRecipients));
                    }

                    MailFilter filter = null;
                    for (Address address : recipients) {

                        String recipient = address.toString();

                        // some recipient are like "toto tutu<toto.tutu73@gmail.com>"
                        // the regex is to extract email address from it
                        recipient = recipient.replaceFirst("^.*<(.*)>$", "$1");
                        recipient = recipient.toLowerCase();

                        Collection<MailFilter> filters = mailFolderService.getFiltersForRecipient(recipient);

                        if (CollectionUtils.isNotEmpty(filters)) {
                            filter = filters.iterator().next();
                            email.setRecipient(recipient);
                            modifiedProperties.add(Email.PROPERTY_RECIPIENT);
                            break;
                        }

                    }
                    if (filter == null) {
                        if (log.isDebugEnabled()) {
                            log.debug(" ==> No filter found for this message");
                            if (log.isTraceEnabled()) {
                                for (Address address : recipients) {
                                    log.trace(" - for recipient " + address.toString());
                                }
                            }
                        }
                        
                        //  on garde le mail sur le serveur pour le traiter suite à l'ajout d'un futur filtre
                        continue;
                    }

                    // find company of the filter folder
                    MailFolder filterFolder = filter.getMailFolder();
                    while (!filterFolder.isUseCurrentLevelCompany() && filterFolder.getParent() != null) {
                        filterFolder = filterFolder.getParent();
                    }
                    String company = filterFolder.getCompany();

                    Address[] addresses = message.getFrom();
                    if (addresses != null && addresses.length > 0) {
                        String sender = addresses[0].toString();
                        // some sender are like "toto tutu<toto.tutu73@gmail.com>"
                        // the regex is to extract email address from it
                        sender = sender.replaceFirst("^.*<(.*)>$", "$1");
                        sender = sender.toLowerCase();

                        // Identification du client en fonction du numéro de fax appelant ou de l'adresse e-mail émettrice
                        Client client = clientService.getClientForEmailAddress(sender, email, company);
                        modifiedProperties.add(Email.PROPERTY_SENDER);

                        String object;
                        if (client != null) {
                            //TODO echatellier : a valider que l'on affiche bien le premier mail du client
                            String contact = sender;
                            if (email.isFax()) {
                                String faxNumber = contact.substring(0, contact.indexOf('@')).replaceAll(" ", "");
                                // NumberUtils.isNumber peut echouer (notation octal)
                                if (StringUtils.isNumeric(faxNumber)) {
                                    contact = StringUtils.leftPad(faxNumber, 10, '0');
                                }
                            }

                            List<String> objectItems = new ArrayList<>();
                            if (StringUtils.isNotEmpty(client.getCaracteristic1())) {
                                objectItems.add(client.getCaracteristic1());
                            }
                            if (StringUtils.isNotEmpty(client.getCaracteristic2())) {
                                objectItems.add(client.getCaracteristic2());
                            }
                            if (StringUtils.isNotEmpty(client.getCaracteristic3())) {
                                objectItems.add(client.getCaracteristic3());
                            }
                            if (StringUtils.isNotEmpty(client.getCode())) {
                                objectItems.add(client.getCode());
                            }
                            if (StringUtils.isNotEmpty(client.getName())) {
                                objectItems.add(client.getName());
                            }
                            objectItems.add(contact);
                            objectItems.add(DateFormat.getDateInstance(DateFormat.MEDIUM).format(new Date()));

                            object = StringUtils.join(objectItems, " / ");

                            modifiedProperties.add(Email.PROPERTY_CLIENT);

                        } else {
                            object = t("faxtomail.email.object.noClient");
                        }
                        email.setObject(object);
                        email.setClient(client);
                        modifiedProperties.add(Email.PROPERTY_OBJECT);

                        MailFolder mailFolder = null;
                        if (!filter.isFilterFolderPriority() && client != null) {
                            FaxToMailUser personInCharge = client.getPersonInCharge();
                            if (personInCharge != null) {
                                mailFolder = mailFolderService.getFolderForFaxToMailUser(personInCharge);
                            }
                        }

                        if (mailFolder == null) {
                            // to default folder
                            mailFolder = filter.getMailFolder();
                        }

                        email.setMailFolder(mailFolder);
                        modifiedProperties.add(Email.PROPERTY_MAIL_FOLDER);

                        // if client is null and folder
                        if (client == null && emailAccount.isRejectAllowed()) {
                            Boolean reject = null;
                            String rejectMessage = null;
                            String senderEmail = null;

                            // find reject conf
                            MailFolder rejectMailFolder = mailFolder;
                            while (!Boolean.FALSE.equals(reject) && rejectMailFolder != null
                                    && (reject == null
                                            || !rejectMailFolder.isUseCurrentLevelRejectResponseMessage()
                                            || !rejectMailFolder.isUseCurrentLevelRejectResponseMailAddress())) {

                                if (reject == null) {
                                    reject = rejectMailFolder.getRejectUnknownSender();
                                }
                                if (rejectMessage == null && rejectMailFolder.isUseCurrentLevelRejectResponseMessage()) {
                                    rejectMessage = rejectMailFolder.getRejectResponseMessage();
                                }
                                if (senderEmail == null && rejectMailFolder.isUseCurrentLevelRejectResponseMailAddress()) {
                                    senderEmail = rejectMailFolder.getRejectResponseMailAddress();
                                }
                                rejectMailFolder = rejectMailFolder.getParent();
                            }

                            if (Boolean.TRUE.equals(reject)) {
                                
                                if (StringUtils.isAnyBlank(senderEmail, rejectMessage)) {
                                    if (log.isWarnEnabled()) {
                                        log.warn("Can't send reject message due to invalid configuration");
                                    }
                                } else {
                                    // unknown client -> message rejected
                                    String recipient = email.getSender();
                                    if (email.isFax()) {
                                        recipient = FaxToMailServiceUtils.addFaxDomainToFaxNumber(recipient, mailFolder);
                                    }
                                    emailService.rejectEmail(senderEmail, recipient, t("faxtomail.email.subject.re", message.getSubject()), rejectMessage);
    
                                    // important, delete mail
                                    deleteMail(message);
                                }

                                continue;
                            }
                        }

                    }

                    Date receivedDate = new Date();
                    email.setReceptionDate(receivedDate);
                    modifiedProperties.add(Email.PROPERTY_RECEPTION_DATE);

                    Date now = new Date();

                    Decorator<Date> dateDecorator = decoratorService.getDecoratorByType(Date.class, DecoratorService.DATE);
                    String projectRef = t("faxtomail.email.projectReference.default", dateDecorator.toString(now));
                    email.setProjectReference(projectRef);
                    modifiedProperties.add(Email.PROPERTY_PROJECT_REFERENCE);

                    email.setDemandStatus(DemandStatus.UNTREATED);
                    modifiedProperties.add(Email.PROPERTY_DEMAND_STATUS);

                    StringBuilder emailSource = new StringBuilder();
                    Enumeration<String> headerLines = ((MimeMessage)message).getAllHeaderLines();
                    while (headerLines.hasMoreElements()) {
                        String headerLine = headerLines.nextElement();
                        emailSource.append(headerLine).append("\n");
                    }
                    String originalContent = IOUtils.toString(message.getInputStream(), charset);
                    emailSource.append("\n").append(originalContent);
                    email.setOriginalEmail(emailSource.toString());

                    List<Attachment> attachments = new ArrayList<>();
                    if (message.isMimeType("multipart/*")) {

                        // manage boundary id
                        String htmlContent = decomposeMultipartEmail(attachments, message, email, emailService);
                        if (htmlContent != null)  {
                            if (log.isDebugEnabled()) {
                                log.debug("Converting html content to pdf : " + message.getSubject());
                            }
                            Attachment attachment = convertHTMLToPdf(attachments, htmlContent, t("faxtomail.email.content.attachment.htmlfileName"));
                            if (attachment != null) {
                                //remove text plain attachement if exists, to avoid having twice the mail content in the attachments
                                String plainTextFileName = t("faxtomail.email.content.attachment.plainfileName") + ".pdf";
                                for (Attachment a : attachments) {
                                    if (plainTextFileName.equals(a.getOriginalFileName())) {
                                        attachments.remove(a);
                                        break;
                                    }
                                }
                                attachments.add(attachment);
                            }
                        }
                    } else {
                        // convertit le contenu texte en PDF
                        String content = IOUtils.toString(message.getInputStream(), charset);
                        Attachment attachment = convertTextToPdf(content, t("faxtomail.email.content.attachment.plainfileName"));
                        attachments.add(0, attachment);
                    }

                    emailService.saveEmail(email,
                            attachments,
                            null,
                            email.getClient() != null ? email.getClient().getCode() : null,
                            null,
                            modifiedProperties.toArray(new String[modifiedProperties.size()]));
                    importedCount++;

                    if (log.isDebugEnabled()) {
                        log.debug(" ==> Message placé dans le dossier " + email.getMailFolder().getName());
                    }

                    // important, delete mail
                    deleteMail(message);

                } catch (Exception e) {
                    log.error("Error while reading the email", e);
                }
            }
            if (log.isDebugEnabled()) {
                log.debug("End of emails");
            }

        } catch (Exception e) {
            log.error("Error while reading the emails", e);
        }

        // usefull log info (do not remove)
        if (importedCount > 0 && log.isInfoEnabled()) {
            log.info(String.format("Imported %d mail for account %s@%s", importedCount, emailAccount.getLogin(), emailAccount.getHost()));
        }
    }

    /**
     * Supprime le mail sur le serveur (si nécéssaire).
     * 
     * @param message message to delete
     * @throws MessagingException 
     */
    protected void deleteMail(Message message) throws MessagingException {
        // suppression des mails sur le serveur distant (non actif par default)
        if (config.isMailDelete()) {
            message.setFlag(Flags.Flag.DELETED, true);
        }
    }

    /**
     * Decompose a multipart part.
     * - sets the email content if the part contains a text bodypart
     * - adds attachments to the email
     *
     * @param part the part to decompose
     * @param email the email object to fill with the attachments or content
     * @param emailService email service
     * @throws Exception
     */
    protected String decomposeMultipartEmail(List<Attachment> attachments, Part part, Email email, EmailService emailService) throws Exception {
        String result = null;

        
        DataSource dataSource = part.getDataHandler().getDataSource();
        MimeMultipart mimeMultipart = new MimeMultipart(dataSource);
        int multiPartCount = mimeMultipart.getCount();

        for (int j = 0; j < multiPartCount; j++) {
            MimeBodyPart bp = (MimeBodyPart) mimeMultipart.getBodyPart(j);

            // if it is a text part, then this is the email content
            String disposition = bp.getDisposition();
            if (bp.isMimeType("text/*") && !Part.ATTACHMENT.equals(disposition)) {

                Charset charset = FaxToMailServiceUtils.getCharset(bp);
                String content = IOUtils.toString(bp.getInputStream(), charset);

                if (bp.isMimeType("text/plain")) {
                    Attachment attachment = convertTextToPdf(content, t("faxtomail.email.content.attachment.plainfileName"));
                    attachments.add(attachment);

                } else {
                    result = content;
                }

            // if it is multipart part, decompose it
            } else if (bp.isMimeType("multipart/*")) {
                String htmlContent = decomposeMultipartEmail(attachments, bp, email, emailService);
                if (htmlContent != null) {
                    result = htmlContent;
                }

            // else, this is an attachment
            } else {
                String fileName = bp.getFileName();

                // parse Content-ID (content identifier in html mail content)
                String[] headers = bp.getHeader("Content-ID");
                String contentID = null;
                if (headers != null) {
                    contentID = headers[0];
                    contentID = contentID.replaceFirst("^<(.*)>$", "$1");
                }

                // remove the guillemets between the id

                if (fileName == null && contentID == null) {
                    break;
                } else if (fileName == null) {
                    fileName = contentID;
                }

                try {
                    fileName = MimeUtility.decodeText(fileName);
                    if (log.isDebugEnabled()) {
                        log.debug("FileName : " + fileName + ", Content-ID : " + contentID);
                    }
                } catch (UnsupportedEncodingException ex) {
                    // don't care, use filename raw value
                    if (log.isWarnEnabled()) {
                        log.warn("Can't debug email file name", ex);
                    }
                }

                DataHandler dh = bp.getDataHandler();

                // create new attachment
                Attachment attachment = new AttachmentImpl();
                attachment.setAddedByUser(false);
                AttachmentFile attachmentFile = emailService.getAttachmentFileFromStream(dh.getInputStream());
                attachmentFile.setFilename(fileName);
                attachment.setOriginalFile(attachmentFile);
                attachment.setContentId(contentID);

                // convert attachment if defined by admin
                convertIfNecessary(attachment);

                // save attachment
                attachments.add(attachment);
            }
        }
        return result;
    }

    /**
     * Retourne un input stream sur une piece jointe convertie ou pas.
     * 
     * @param attachment to convert
     * @return attachment
     * @throws IOException 
     */
    protected Attachment convertIfNecessary(Attachment attachment) throws IOException {
        
        // get file extension
        String filename = attachment.getOriginalFileName();
        String extension = FilenameUtils.getExtension(filename);

        if (StringUtils.isNotBlank(extension)) {
            ExtensionCommand command = configurationService.getExtensionCommand(extension);

            // si une extension est configurée avec une commande non vide
            if (command != null && StringUtils.isNotBlank(command.getConvertToPdfCommand())) {
                attachment = convertToPdf(attachment, command);
            }
        }
        return attachment;
    }

    /**
     * Convert attachment to pdf.
     * 
     * @param attachment attachment to convert
     * @param extensionCommand command
     * @return
     * @throws IOException 
     */
    protected Attachment convertToPdf(Attachment attachment, ExtensionCommand extensionCommand) throws IOException {
        
        // get file extension
        String filename = attachment.getOriginalFileName();
        String basename = FilenameUtils.getBaseName(filename);
        String extension = FilenameUtils.getExtension(filename);

        // copy file to temp file
        File file = File.createTempFile("faxtomail-" + basename, "." + extension);
        file.deleteOnExit();
        // on creer un nouveau nom de fichier ou seulement le nom de l'extension change
        // cela permet de fonctionner avec openoffice par exemple où il n'est pas possible
        // de specifier le nom du fichier de sortie
        String fullname = StringUtils.removeEnd(file.getAbsolutePath(), "." + extension);
        File outfile = new File(fullname + ".pdf");
        outfile.deleteOnExit();
        FileUtils.writeByteArrayToFile(file, attachment.getOriginalFile().getContent());

        // get process command
        String command = extensionCommand.getConvertToPdfCommand();
        String[] args = StringUtil.split(command, " ");
        List<String> comArgs = new ArrayList<String>();
        for (String arg : args) {
            String localArg = arg;
            localArg = localArg.replace("%f", file.getAbsolutePath());
            localArg = localArg.replace("%o", outfile.getAbsolutePath());
            comArgs.add(localArg);
        }
        ProcessBuilder pb = new ProcessBuilder(comArgs);
        // run process
        if (log.isDebugEnabled()) {
            log.debug("Convert attachment with command : " + comArgs);
        }
        try {
            Process process = pb.start();
            process.waitFor();

            // read output to save into attachment
            byte[] outContent = FileUtils.readFileToByteArray(outfile);
            
            // on va dire que normalement un pdf ca fait plus de 1 octet
            if (outContent.length >= 1) {
                AttachmentFile editedFile = new AttachmentFileImpl();
                editedFile.setContent(outContent);
                editedFile.setFilename(basename + "-converted.pdf");
                attachment.setEditedFile(editedFile);
            } else if (log.isWarnEnabled()) {
                log.warn("Can't convert extension to pdf (content is empty)");
            }

        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Cannot run convert command", e);
            }
        } catch (InterruptedException e) {
            if (log.isErrorEnabled()) {
                log.error("Cannot run convert command", e);
            }
        }

        // cleanup
        file.delete();
        outfile.delete();

        return attachment;
    }

    protected Attachment convertTextToPdf(String content, String name) throws IOException, DocumentException {
        //File target = File.createTempFile("faxtomail-", ".tmp");
        //target.deleteOnExit();

        Document document = new Document();
        //FileOutputStream fos = new FileOutputStream(target);
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        PdfWriter writer = PdfWriter.getInstance(document, os);
        writer.open();

        document.setPageSize(PageSize.A4);
        document.open();

        document.add(new Paragraph(content));

        document.close();
        writer.close();

        // convert content to blob
        AttachmentFile attachmentFileNew = new AttachmentFileImpl();
        attachmentFileNew.setContent(os.toByteArray());
        attachmentFileNew.setFilename(name + ".pdf");

        Attachment attachment = new AttachmentImpl();
        attachment.setOriginalFile(attachmentFileNew);
        attachment.setOriginalFileName(name + ".pdf");
        attachment.setAddedByUser(false);
        
        //target.delete();

        return attachment;
    }

    /**
     * Convert html content to image.
     * 
     * @param attachments attachment content to link content to
     * @param content html content
     * @param name result attachment name
     * @return image attachment or {@code null} if content can't be converted
     * @throws IOException
     */
    protected Attachment convertHTMLToPdf(List<Attachment> attachments, String content, String name) throws IOException {


        Attachment result = null;
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        Collection<File> fileToDelete = new ArrayList<File>();
        try {
            content = content.replaceAll("<meta (.*?)>(</meta>)?", "");

            for (Attachment attachment : attachments) {
                String key = attachment.getContentId();
                if (key == null) {
                    key = attachment.getOriginalFileName();
                }

                // get file content
                AttachmentFile attachmentFile = attachment.getOriginalFile();
                File file = attachmentFile.getFile();
                fileToDelete.add(file);
    
                // replace the inline attachments with the extracted attachment file url
                // match les patterns:
                // <td background="cid:bg.gif" height="52">
                // <img border=0 src="cid:bg.gif" />
                // <img src='cid:5e9ef859-ea65-4f9b-a9fa-30d4a2c5837c'
                content = content.replaceAll("(\\w+)=([\"'])cid:" + Pattern.quote(key) + "([\"'])", "$1=$2file://" + file.getAbsolutePath() + "$3");
    
                if (log.isDebugEnabled()) {
                    log.debug("Mapping attachment id " + key + " to file " + file.getAbsolutePath());
                }
            }

            Html2Image image = Html2Image.fromHtml(content);
            //ImageRenderer imageRenderer = image.getImageRenderer();
            //imageRenderer.saveImage(os, true);
            org.w3c.dom.Document doc = image.getParser().getDocument();

            // convert to pdf
            ITextRenderer renderer = new ITextRenderer();
            renderer.setDocument(doc, null);
            renderer.layout();
            renderer.createPDF(os);
            
            // convert content to blob
            AttachmentFile attachmentFileNew = new AttachmentFileImpl();
            attachmentFileNew.setContent(os.toByteArray());
            attachmentFileNew.setFilename(name + ".pdf");
    
            result = new AttachmentImpl();
            result.setOriginalFile(attachmentFileNew);
            result.setOriginalFileName(name + ".pdf");
            result.setAddedByUser(false);
        } catch (OutOfMemoryError er) {
            // certains mails (spam) très compliqué ne peuvent pas être convertit en images
            if (log.isWarnEnabled()) {
                log.warn("Can't convert html content to pdf", er);
            }
        } catch (DocumentException ex) {
            if (log.isWarnEnabled()) {
                log.warn("Can't convert html content to pdf", ex);
            }
        } finally {
            // on supprime les fichiers avant le delete on exit pour eviter que le disque ne se remplissent trop vite
            for (File file : fileToDelete) {
                file.delete();
            }
        }

        return result;
    }
}
