/*
 * *##% 
 * JAXX Runtime
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.runtime.validator.swing;

import jaxx.runtime.validator.swing.ui.AbstractBeanValidatorUI;
import jaxx.runtime.validator.swing.ui.IconValidationUI;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.jxlayer.JXLayer;

import javax.swing.JComponent;
import javax.swing.SwingUtilities;
import java.awt.Container;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import jaxx.runtime.validator.BeanValidator;
import jaxx.runtime.validator.BeanValidatorField;

/**
 * La surcharge de {@link jaxx.runtime.validator.BeanValidator} pour les ui swing
 * <p/>
 * /**
 * <p/>
 * Permet d'ajouter facilement le support de la validation des champs d'un
 * bean et de le relier a une interface graphique.
 * Utilise xwork pour la validation et JXLayer pour la visualisation.
 * <p/>
 * <p/>
 * Le mieux pour son integration dans Jaxx est de faire de la generation pour
 * force la compilation du code suivant:
 * <p/>
 * <pre>
 * myValidor.getBean().get<field>();
 * </pre>
 * <p/>
 * et ceci pour chaque field ajoute a la map fieldRepresentation. De cette
 * facon meme si le champs field est en texte on a une verification de son
 * existance a la compilation.
 * <p/>
 * <p/>
 * La representation en tag pourrait etre
 * <pre>
 * &lt;validator id="myValidator" beanClass="{Personne.class}" errorList="$list"&gt;
 *   &lt;field name="name" component="$name"/&gt;
 *   &lt;field name="firstName" component="$firstName"/&gt;
 *   &lt;field name="birthDate" component="$birthDate"/&gt;
 * &lt;/validator>
 * &lt;validator beanClass="{Personne.class}" autoField="true" errorList="$list"&gt;
 *   &lt;fieldRepresentation name="name" component="$lastName"/&gt;
 * &lt;/validator&gt;
 * </pre>
 * <p/>
 * dans le premier exemple on fait un mapping explicite des champs, mais on voit
 * que le nom du composant graphique est le meme que celui du champs. Pour eviter
 * de longue saisie, il est possible d'utiliser le flag <b>autoField</b>
 * qui pour chaque champs du ayant une methode get du bean recherche un composant
 * avec cet Id. Il est aussi possible de surcharge un champs explicitement
 * comme ici name, dans le cas ou le composant qui porterait ce nom serait
 * utilise pour autre chose.
 * <p/>
 * <p/>
 * Il faut un handler particulier pour ce composant car les attributs
 * <b>beanClass</b> et <b>autoField</b> ne sont present que dans le XML jaxx et
 * servent a la generation. Il faut aussi prendre en compte les elements
 * fieldRepresentation fils du tag validator.
 * <p/>
 * <p/>
 * Voici ce que pourrait etre le code genere par jaxx
 * <pre>
 * // declaration du bean
 * BeanValidator<beanClass> $myValidator;
 * // init du bean
 * protected void createMyValidator() {
 *   $myValidator = new BeanValidator<beanClass>();
 *   // genere seulement si autoField = true
 *   for (Method m : beanClass.getMethod()) {
 *     if (m.getName().startsWith("get")) {
 *       String fieldName = m.getName().substring(3).toLowerCase();
 *       $myValidator.setFieldRepresentation(fieldName, $objectMap.get(fieldName));
 *     }
 *   }
 *   // pour chaque tag fieldRepresentation
 *   myValidator.setFieldRepresentation("name", $lastName);
 *   // si beanClass est specifie et n'est pas Object, on force l'acces au champs
 *   // pour validation a la compilation
 *   $myValidator.getBean().getName();
 *   $objectMap.put("myValidator", $myValidator);
 * }
 * </pre>
 *
 * @param <B> le type de bean a valider
 * @author poussin
 * @author chemit
 * @version 1.0 
 */
public class SwingValidator<B> extends BeanValidator<B> {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private final Log log = LogFactory.getLog(SwingValidator.class);
    static private final Class<? extends AbstractBeanValidatorUI> DEFAULT_UI_CLASS = IconValidationUI.class;
    /** permet de faire le lien en un champs du bean et l'objet qui permet de l'editer */
    protected Map<String, JComponent> fieldRepresentation;
    /** Object servant a contenir la liste des erreurs */
    protected SwingValidatorMessageListModel errorListModel;
    /** Object servant a contenir la liste des erreurs */
    protected SwingValidatorMessageTableModel errorTableModel;
    /** ui renderer class */
    protected Class<? extends AbstractBeanValidatorUI> uiClass;

    public SwingValidator(Class<B> beanClass, String contextName) {
        super(beanClass, contextName);
        fieldRepresentation = new HashMap<String, JComponent>();
    }

    /**
     * To reload a bean in the validator.
     *
     * This method is used to reload ui, since some editors
     * could not exist when validator is init, so some messages
     * should not be attached to an editor.
     */
    public void reloadBean() {
        B b = getBean();
        if (b != null) {
            setBean(null);
            setBean(b);
        }
    }

    public JComponent getFieldRepresentation(String fieldname) {
        return fieldRepresentation.get(fieldname);
    }

    public Class<? extends AbstractBeanValidatorUI> getUiClass() {
        return uiClass;
    }

    public void setErrorListModel(SwingValidatorMessageListModel errorListModel) {
        this.errorListModel = errorListModel;
        if (errorListModel != null) {
            // register the validator in the  model list
            errorListModel.registerValidator(this);
        }
    }

    public void setErrorTableModel(SwingValidatorMessageTableModel errorTableModel) {
        this.errorTableModel = errorTableModel;
        if (errorTableModel != null) {
            // register the validator in the  model table
            errorTableModel.registerValidator(this);
        }
    }

    public void setUiClass(Class<? extends AbstractBeanValidatorUI> uiClass) {
        this.uiClass = uiClass;
    }

    @Override
    public void setContextName(String contextName) {
        /*Map<ValidatorField<B>, List<ValidatorErrorListener>> oldListeners = new HashMap<ValidatorField<B>, List<ValidatorErrorListener>>();

        for (ValidatorField<B> field : fields) {
        ValidatorErrorListener[] listeners = field.getValidatorErrorListeners();
        List<ValidatorErrorListener> toReinject = new ArrayList<ValidatorErrorListener>();
        for (ValidatorErrorListener listener : listeners) {
        if (listener instanceof AbstractBeanValidatorUI) {
        // this listener will be reinject via installUIs method
        continue;
        }
        toReinject.add(listener);
        }
        oldListeners.put(field, toReinject);
        }*/
        super.setContextName(contextName);
        // must reinstall ui
        installUIs();
        // reinject none ui listeners
        /*for (Entry<ValidatorField<B>, List<ValidatorErrorListener>> entry : oldListeners.entrySet()) {
        ValidatorField<B> field = getField(entry.getKey().getName());
        for (ValidatorErrorListener listener : entry.getValue()) {
        field.addValidatorErrorListener(listener);
        }
        }
        oldListeners.clear();*/
    }

    /**
     * Permet d'indiquer le composant graphique responsable de l'affichage
     * d'un attribut du bean
     *
     * @param fieldname the field name in the bean
     * @param c         the editor component for the field
     */
    public void setFieldRepresentation(String fieldname, JComponent c) {
        BeanValidatorField<B> field = getField(fieldname);
        if (field == null) {
            // no field registred in the validator
            log.warn("the field '" + fieldname + "' is not defined in validator (no rules on it)");
            return;
        }
        fieldRepresentation.put(fieldname, c);
    }

    public void setFieldRepresentation(Map<String, JComponent> fieldRepresentation) {
        for (Map.Entry<String, JComponent> e : fieldRepresentation.entrySet()) {
            setFieldRepresentation(e.getKey(), e.getValue());
        }
    }

    @Override
    public SwingValidator<?> getParentValidator() {
        return (SwingValidator<?>) super.getParentValidator();
    }

    public void setParentValidator(SwingValidator<?> parentValidator) {
        super.setParentValidator(parentValidator);
    }

    /** install ui on required components */
    public void installUIs() {
        SwingUtilities.invokeLater(new Runnable() {

            @Override
            public void run() {
                if (uiClass == null) {
                    // use the default one
                    uiClass = DEFAULT_UI_CLASS;
                }
                for (Entry<String, JComponent> entry : fieldRepresentation.entrySet()) {
                    try {
                        setMessageRepresentation(entry.getKey(), null, entry.getValue(), uiClass);
                    } catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                }
            }
        });
    }

    protected void setMessageRepresentation(String fieldname, JComponent old, JComponent c, Class<? extends AbstractBeanValidatorUI> uiClass)
            throws InvocationTargetException, IllegalAccessException, InstantiationException, NoSuchMethodException {
        if (old == c) {
            // same component, nothing to do
            return;
        }
        BeanValidatorField<B> field = getField(fieldname);

        if (field == null) {
            // this case should not appear since fieldName has already been check in method addFieldRepresentation
            return;
        }
        if (old != null) {
            // suppression du jxlayer sous l'ancien composant
            Container container = old.getParent();
            if (container instanceof JXLayer<?>) {
                JXLayer<?> jx = (JXLayer<?>) container;
                Object ui = jx.getUI();
                if (ui != null && ui instanceof AbstractBeanValidatorUI) {
                    removeBeanValidatorListener((AbstractBeanValidatorUI) ui);
                }

                jx.setUI(null);
            }
        }
        if (c != null) {
            // ajout du jxlayer sous ce composant
            Container container = c.getParent();
            if (container instanceof JXLayer<?>) {
                Constructor<? extends AbstractBeanValidatorUI> cons = uiClass.getConstructor(BeanValidatorField.class);
                AbstractBeanValidatorUI ui = cons.newInstance(field);
                ui.setEnabled(true);
                JXLayer<JComponent> jx = (JXLayer<JComponent>) container;
                addBeanValidatorListener(ui);
                jx.setUI(ui);
            }
        }
    }
}
