/*
 * #%L
 * Pollen :: Services
 * $Id: PollenMigrationCallbackV1_3.java 3698 2012-09-23 13:01:38Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.2/pollen-services/src/main/java/org/chorem/pollen/entities/migration/PollenMigrationCallbackV1_3.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.entities.migration;

import com.google.common.base.Function;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.tuple.Pair;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.business.persistence.ChoiceType;
import org.chorem.pollen.business.persistence.PollType;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.framework.TopiaSQLQuery;
import org.nuiton.topia.migration.TopiaMigrationCallbackByClassNG;
import org.nuiton.util.Version;
import org.nuiton.util.VersionUtil;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;

/**
 * Migration for version {@code 1.3}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PollenMigrationCallbackV1_3 extends TopiaMigrationCallbackByClassNG.MigrationCallBackForVersion {

    @Override
    public Version getVersion() {
        return VersionUtil.valueOf("1.3");
    }

    @Override
    protected void prepareMigrationScript(TopiaContextImplementor tx,
                                          List<String> queries,
                                          boolean showSql,
                                          boolean showProgression) throws TopiaException {

        // change choice type to use enumeration
        migrateChoiceTypes(tx, queries);

        // change poll type to use enumeration
        migratePollTypes(tx, queries);

        // change vote counting type to use enumeration
        migrateVoteCountingTypes(tx, queries);

        // change all Boolean properties to boolean
        migrateToPrimitiveTypes(queries);

        // add foreign key indexes
        addForeignKeyIndexes(queries);

        // add author column for Comment
        addCommentAuthorColumn(queries);
    }

    private void addCommentAuthorColumn(List<String> queries) {

        queries.add("ALTER TABLE comment ADD COLUMN author VARCHAR(255);");

        queries.add("UPDATE comment AS c SET author = (SELECT p.votingId FROM pollAccount p WHERE c.pollAccount = p.topiaId);");

        queries.add("ALTER TABLE comment ALTER COLUMN author TYPE VARCHAR(255);");
        queries.add("UPDATE comment  SET author = 'anonymous'  WHERE author IS NULL;");
        queries.add("ALTER TABLE comment ALTER COLUMN author SET NOT NULL;");
    }

    private void addForeignKeyIndexes(List<String> queries) {

        queries.add("CREATE INDEX idx_PollAccount_pollsCreated ON poll(creator);");
        queries.add("CREATE INDEX idx_PollAccount_comment ON comment(pollAccount);");
        queries.add("CREATE INDEX idx_PollAccount_vote ON vote(pollAccount);");
        queries.add("CREATE INDEX idx_Poll_vote ON vote(poll);");
        queries.add("CREATE INDEX idx_Poll_choice ON choice(poll);");
        queries.add("CREATE INDEX idx_Poll_result ON result(poll);");
        queries.add("CREATE INDEX idx_Poll_comment ON comment(poll);");
        queries.add("CREATE INDEX idx_Poll_preventRule ON preventRule(poll);");
        queries.add("CREATE INDEX idx_Poll_votingList ON votingList(poll);");
        queries.add("CREATE INDEX idx_UserAccount_favoriteList ON personList(owner);");
        queries.add("CREATE INDEX idx_UserAccount_pollAccount ON pollAccount(userAccount);");
        queries.add("CREATE INDEX idx_PersonList_pollAccount ON pollAccount(personList);");
    }

    private void migrateToPrimitiveTypes(List<String> queries) throws TopiaException {

        migrateBoolean(queries, "useraccount", "administrator");
        migrateBoolean(queries, "choice", "validate");
        migrateBoolean(queries, "personToList", "hasVoted");
        migrateBoolean(queries, "poll", "closed");
        migrateBoolean(queries, "poll", "choiceAddAllowed");
        migrateBoolean(queries, "poll", "anonymousVoteAllowed");
        migrateBoolean(queries, "poll", "anonymous");
        migrateBoolean(queries, "poll", "publicResults");
        migrateBoolean(queries, "poll", "continuousResults");
        migrateBoolean(queries, "preventRule", "repeated");
        migrateBoolean(queries, "preventRule", "active");
        migrateBoolean(queries, "preventRule", "oneTime");
        migrateBoolean(queries, "result", "byGroup");
        migrateBoolean(queries, "vote", "anonymous");

        migrateNumber(queries, "poll", "maxChoiceNb");
        migrateNumber(queries, "preventRule", "sensibility");

        migrateNumber(queries, "PersonToList", "weight");
        migrateNumber(queries, "VotingList", "weight");
        migrateNumber(queries, "Vote", "weight");
    }

    public static final String UPDATE_PRIMITIVE_TO_DEFAULT_VALUE = "UPDATE %1$s SET %2$s = %3$s where %2$s IS NULL;";

    public static final String ADD_PRIMITIVE_NOT_NULL_CONSTRAINST = "ALTER TABLE %s ALTER COLUMN %s SET NOT NULL;";

    public static final String ADD_PRIMITIVE_DEFAULT_VALUE = "ALTER TABLE %s ALTER COLUMN %s SET DEFAULT %s;";

    private void migrateBoolean(List<String> queries, String tableName, String field) {

        Object defaultValue = false;
        queries.add(String.format(UPDATE_PRIMITIVE_TO_DEFAULT_VALUE, tableName, field, defaultValue));
        queries.add(String.format(ADD_PRIMITIVE_NOT_NULL_CONSTRAINST, tableName, field));
        queries.add(String.format(ADD_PRIMITIVE_DEFAULT_VALUE, tableName, field, defaultValue));

    }

    private void migrateNumber(List<String> queries, String tableName, String field) {

        Object defaultValue = 0;
        queries.add(String.format(UPDATE_PRIMITIVE_TO_DEFAULT_VALUE, tableName, field, defaultValue));
        queries.add(String.format(ADD_PRIMITIVE_NOT_NULL_CONSTRAINST, tableName, field));
        queries.add(String.format(ADD_PRIMITIVE_DEFAULT_VALUE, tableName, field, defaultValue));
    }

    private void migrateChoiceTypes(TopiaContextImplementor tx,
                                    List<String> queries) throws TopiaException {

        // build the new mapping between old choice type id associated to new choicetype enum ordinal
        Map<String, Integer> mapping = getMapping(tx, "choicetype", ChoiceType.values());

        replaceColumn(tx,
                      "poll",
                      "fk3497bf5b0b3601",
                      "choicetype",
                      "choicetype",
                      mapping,
                      queries);

        // drop choicetype table
        queries.add("DROP TABLE choicetype;");
    }

    private void migratePollTypes(TopiaContextImplementor tx,
                                  List<String> queries) throws TopiaException {

        // build the new mapping between old choice type id associated to new choicetype enum ordinal
        Map<String, Integer> mapping = getMapping(tx, "polltype", PollType.values());

        replaceColumn(tx,
                      "poll",
                      "fk3497bf978bfd",
                      "polltype",
                      "polltype",
                      mapping,
                      queries);

        // drop pollType table
        queries.add("DROP TABLE polltype;");
    }

    enum VoteCountingType {
        NORMAL,
        PERCENTAGE,
        CONDORCET,
        NUMBER
    }

    private void migrateVoteCountingTypes(TopiaContextImplementor tx,
                                          List<String> queries) throws TopiaException {


        // build the new mapping between old id associated to new enum ordinal
        Map<String, Integer> mapping = getMapping(tx, "votecounting", VoteCountingType.values());

        replaceColumn(tx,
                      "poll",
                      "fk3497bf7d358045",
                      "votecounting",
                      "votecountingtype",
                      mapping,
                      queries);

        replaceColumn(tx,
                      "result",
                      "fkc84dc81d7d358045",
                      "votecounting",
                      "votecountingtype",
                      mapping,
                      queries);

        // drop votecounting table
        queries.add("DROP TABLE votecounting;");
    }

    private <E extends Enum<E>> Map<String, Integer> getMapping(TopiaContextImplementor tx, String tableName, E... enums) throws TopiaException {

        List<Pair<String, String>> entitiesFromDb =
                new TopiaIdSQLQueryToPair("select topiaId, name from " + tableName).findMultipleResult(tx);

        Map<String, Pair<String, String>> entityByTopiaIdAndName =
                Maps.uniqueIndex(entitiesFromDb, PAIR_TO_VALUE);

        // build the new mapping between old id associated to new enum ordinal
        Map<String, Integer> result = Maps.newHashMap();

        for (Enum choiceType : enums) {

            // find it in existing mapping
            String name = choiceType.name();
            Pair<String, String> choiceTypePair = entityByTopiaIdAndName.get(name);
            if (choiceTypePair == null) {

                // this can not happen
                throw new PollenTechnicalException(
                        "Could not find in db to migrate choicetype with name " + name);
            }

            String id = choiceTypePair.getKey();
            result.put(id, choiceType.ordinal());
        }
        return result;
    }

    private void replaceColumn(TopiaContextImplementor tx,
                               String tableName, String fk,
                               String oldColumn,
                               String newColum,
                               Map<String, Integer> mapping,
                               List<String> queries) throws TopiaException {

        List<Pair<String, String>> rowsToReplace =
                new TopiaIdSQLQueryToPair("select topiaId, " + oldColumn + " from " + tableName).findMultipleResult(tx);

        queries.add("ALTER TABLE " + tableName + " DROP CONSTRAINT " + fk + ";");
        queries.add("ALTER TABLE " + tableName + " DROP COLUMN " + oldColumn + ";");
        queries.add("ALTER TABLE " + tableName + " ADD COLUMN " + newColum + " INT NOT NULL DEFAULT 0;");
        queries.add("ALTER TABLE " + tableName + " ALTER COLUMN " + newColum + " DROP NOT NULL;");

        String updateRequest = "update " + tableName + " set " + newColum + " = %s where topiaid = '%s';";
        for (Pair<String, String> pollByChoice : rowsToReplace) {

            String tableRowId = pollByChoice.getKey();
            String typeId = pollByChoice.getValue();

            Integer ordinalValue = mapping.get(typeId);
            if (ordinalValue == null) {

                // use a 0 value
                ordinalValue = 0;
            }
            String query = String.format(updateRequest, ordinalValue, tableRowId);
            queries.add(query);
        }
    }

    private static class TopiaIdSQLQueryToPair extends TopiaSQLQuery<Pair<String, String>> {

        protected final String request;

        private TopiaIdSQLQueryToPair(String request) {
            this.request = request;
        }

        @Override
        protected PreparedStatement prepareQuery(Connection connection) throws SQLException {
            PreparedStatement prepareStatement = connection.prepareStatement(request);
            return prepareStatement;
        }

        @Override
        protected Pair<String, String> prepareResult(ResultSet set) throws SQLException {
            Pair<String, String> result = Pair.of(set.getString(1), set.getString(2));
            return result;
        }
    }

    public static final Function<Pair<String, String>, String> PAIR_TO_VALUE = new Function<Pair<String, String>, String>() {
        @Override
        public String apply(Pair<String, String> input) {
            return input.getValue();
        }
    };

}
