/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.services;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.converters.DataChoiceConverter;
import org.chorem.pollen.business.dto.ChoiceDTO;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.ChoiceDAO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.utils.ContextUtil;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

/**
 * Gestion des choix d'un sondage.
 *
 * @author enema
 * @version $Id: ServiceChoiceImpl.java 2862 2010-02-09 15:31:45Z jruchaud $
 */
public class ServiceChoiceImpl implements
        org.chorem.pollen.business.services.ServiceChoice {
    private TopiaContext rootContext = ContextUtil.getInstance().getContext();
    private ChoiceDAO choiceDAO = null;
    private DataChoiceConverter converter = new DataChoiceConverter();

    /** log. */
    private static final Log log = LogFactory.getLog(ServiceChoiceImpl.class);

    public ServiceChoiceImpl() {
    }

    @Override
    public String createChoice(ChoiceDTO choice) {
        TopiaContext transaction = null;
        String topiaId = "";
        try {
            transaction = rootContext.beginTransaction();

            choiceDAO = PollenModelDAOHelper.getChoiceDAO(transaction);

            Choice choiceEntity = choiceDAO.create();
            converter.setTransaction(transaction);
            converter.populateChoiceEntity(choice, choiceEntity);

            topiaId = choiceEntity.getTopiaId();
            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entity created: " + topiaId);
            }

            return topiaId;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return "";
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public List<Choice> createChoices(List<ChoiceDTO> choices, String pollId,
            TopiaContext transaction) {
        if (choices.isEmpty() || pollId.length() == 0) {
            return null;
        }

        List<Choice> result = null;
        try {
            choiceDAO = PollenModelDAOHelper.getChoiceDAO(transaction);
            converter.setTransaction(transaction);

            result = new ArrayList<Choice>();
            for (ChoiceDTO choice : choices) {
                Choice choiceEntity = choiceDAO.create();
                choice.setId(choiceEntity.getTopiaId());
                choice.setPollId(pollId);
                converter.populateChoiceEntity(choice, choiceEntity);
                result.add(choiceEntity);
            }

            return result;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        }
    }

    @Override
    public boolean updateChoice(ChoiceDTO choiceDTO) {
        TopiaContext transaction = null;
        boolean resu = false;
        try {
            transaction = rootContext.beginTransaction();

            choiceDAO = PollenModelDAOHelper.getChoiceDAO(transaction);

            Choice choiceEntity = choiceDAO.findByTopiaId(choiceDTO.getId());
            if (choiceEntity != null) {
                converter.setTransaction(transaction);
                converter.populateChoiceEntity(choiceDTO, choiceEntity);

                choiceDAO.update(choiceEntity);
                transaction.commitTransaction();
                resu = true;
            }

            if (log.isDebugEnabled()) {
                log.debug("Entity updated: " + choiceDTO.getId());
            }

            return resu;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return false;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public boolean deleteChoice(String choiceId) {
        TopiaContext transaction = null;
        try {
            transaction = rootContext.beginTransaction();

            choiceDAO = PollenModelDAOHelper.getChoiceDAO(transaction);

            Choice choiceEntity = choiceDAO.findByTopiaId(choiceId);

            choiceDAO.delete(choiceEntity);
            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entity deleted: " + choiceId);
            }

            return true;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return false;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public ChoiceDTO findChoiceById(String choiceId) {
        TopiaContext transaction = null;
        ChoiceDTO result = null;
        try {
            transaction = rootContext.beginTransaction();

            choiceDAO = PollenModelDAOHelper.getChoiceDAO(transaction);

            Choice choiceEntity = choiceDAO.findByTopiaId(choiceId);

            if (choiceEntity != null) {
                converter.setTransaction(transaction);
                result = converter.createChoiceDTO(choiceEntity);
            }

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entity found: "
                        + ((result == null) ? "null" : result.getId()));
            }

            return result;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public List<ChoiceDTO> findChoicesByName(String choiceName, String pollId) {
        TopiaContext transaction = null;
        List<ChoiceDTO> results = null;
        List<Choice> choiceEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            choiceDAO = PollenModelDAOHelper.getChoiceDAO(transaction);

            HashMap<String, Object> properties = new HashMap<String, Object>();
            properties.put("pollId", pollId);
            properties.put("name", choiceName);

            choiceEntities = choiceDAO.findAllByProperties(properties);

            converter.setTransaction(transaction);
            results = converter.createChoiceDTOs(choiceEntities);

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public List<ChoiceDTO> selectChoices(Map<String, Object> properties) {
        TopiaContext transaction = null;
        List<ChoiceDTO> results = null;
        List<Choice> choiceEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            choiceDAO = PollenModelDAOHelper.getChoiceDAO(transaction);

            if (properties == null) {
                choiceEntities = choiceDAO.findAll();
                if (log.isWarnEnabled()) {
                    log
                            .warn("Attention : tous les choix ont été sélectionnés !");
                }
            } else {
                choiceEntities = choiceDAO.findAllByProperties(properties);
            }
            converter.setTransaction(transaction);
            results = converter.createChoiceDTOs(choiceEntities);

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }
}