/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.services;

import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.converters.DataUserConverter;
import org.chorem.pollen.business.dto.UserDTO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.UserAccountDAO;
import org.chorem.pollen.business.utils.ContextUtil;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

/**
 * Implémentation du service de gestion des utilisateurs.
 *
 * @author amine
 * @author rannou
 * @version $Id: ServiceUserImpl.java 2737 2009-08-20 13:00:52Z nrannou $
 */
public class ServiceUserImpl implements ServiceUser {
    private TopiaContext rootContext = ContextUtil.getInstance().getContext();
    private UserAccountDAO userDAO = null;
    private DataUserConverter converter = new DataUserConverter();

    /** log. */
    private static final Log log = LogFactory.getLog(ServiceUserImpl.class);

    public ServiceUserImpl() {
    }

    @Override
    public String createUser(UserDTO user, String password) {
        TopiaContext transaction = null;
        String topiaId = null;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);
            if (userDAO.findByLogin(user.getLogin()) != null
                    || (!"".equals(user.getEmail()) && userDAO.findByEmail(user
                            .getEmail()) != null)) {
                return null;
            }

            UserAccount userEntity = userDAO.create();
            converter.populateUserEntity(user, userEntity, password);

            topiaId = userEntity.getTopiaId();
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entity created: " + topiaId);
            }

            return topiaId;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        }
    }

    @Override
    public boolean deleteUser(String login) {
        TopiaContext transaction = null;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);

            UserAccount userEntity = userDAO.findByLogin(login);

            userDAO.delete(userEntity);
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entity deleted: " + userEntity.getTopiaId());
            }

            return true;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return false;
        }
    }

    @Override
    public boolean deleteUser(UserDTO user) {
        TopiaContext transaction = null;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);

            UserAccount userEntity = userDAO.findByTopiaId(user.getId());
            userDAO.delete(userEntity);
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entity deleted: " + userEntity.getTopiaId());
            }

            return true;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return false;
        }
    }

    @Override
    public UserDTO findUserById(String userId) {
        TopiaContext transaction = null;
        UserDTO result = null;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);

            UserAccount userEntity = userDAO.findByTopiaId(userId);

            if (userEntity != null) {
                converter.setTransaction(transaction);
                result = converter.createUserDTO(userEntity);
            }

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entity found: "
                        + ((result == null) ? "null" : result.getId()));
            }

            return result;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        }
    }

    @Override
    public List<UserDTO> selectUsers(Map<String, Object> properties) {
        TopiaContext transaction = null;
        List<UserDTO> results = null;
        List<UserAccount> userEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);

            if (properties == null) {
                userEntities = userDAO.findAll();
                if (log.isWarnEnabled()) {
                    log
                            .warn("Attention : tous les utilisateurs ont été sélectionnés !");
                }
            } else {
                userEntities = userDAO.findAllByProperties(properties);
            }

            results = converter.createUserDTOs(userEntities);

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        }
    }

    @Override
    public boolean updateUser(UserDTO user) {
        TopiaContext transaction = null;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);

            UserAccount userEntity = userDAO.findByTopiaId(user.getId());

            converter.setTransaction(transaction);
            converter.populateUserEntity(user, userEntity);

            userDAO.update(userEntity);
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entity updated: " + user.getId());
            }

            return true;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return false;
        }

    }

    @Override
    public boolean updatePasswordUser(UserDTO user, String newPassword) {
        TopiaContext transaction = null;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);

            UserAccount userEntity = userDAO.findByTopiaId(user.getId());

            userEntity.setPassword(newPassword);
            userEntity.update();
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entity updated: " + user.getId());
            }

            return true;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return false;
        }
    }

    @Override
    public boolean isUserLoginExisting(String login) {
        TopiaContext transaction = null;
        boolean userExists = true;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);
            if (userDAO.findByLogin(login) == null) {
                userExists = false;
            }

            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entity exists for the login " + login + ": "
                        + userExists);
            }
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return true;
        }
        return userExists;
    }

    @Override
    public boolean isUserEmailExisting(String email) {
        TopiaContext transaction = null;
        boolean userExists = true;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);
            if (userDAO.findByEmail(email) == null) {
                userExists = false;
            }

            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entity exists for the email " + email + ": "
                        + userExists);
            }
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return true;
        }
        return userExists;
    }
}