/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.converters;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.VotingListDTO;
import org.chorem.pollen.business.persistence.PersonToList;
import org.chorem.pollen.business.persistence.PersonToListDAO;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountDAO;
import org.chorem.pollen.business.persistence.PollDAO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.business.persistence.VoteDAO;
import org.chorem.pollen.business.persistence.VotingList;
import org.chorem.pollen.business.persistence.VotingListDAO;
import org.nuiton.topia.TopiaException;

/**
 * Gestion de la conversion des dtos VotingList en entité et vice-versa.
 *
 * @author enema
 * @version $Id: DataVotingListConverter.java 2615 2009-07-01 13:47:45Z nrannou $
 */
public class DataVotingListConverter extends DataConverter {

    /**
     * Création d'une liste à partir d'un dto liste.
     *
     * @param votingListDTO le dto votingList
     * @param eVotingList l'entité votingList
     */
    public void populateVotingListEntity(VotingListDTO votingListDTO,
            VotingList eVotingList) throws TopiaException {
        eVotingList.setName(votingListDTO.getName());
        eVotingList.setWeight(votingListDTO.getWeight());

        if (votingListDTO.getPollId() != null) {
            if (votingListDTO.getPollId().length() > 0) {
                PollDAO pollDAO = PollenModelDAOHelper
                        .getPollDAO(super.transaction);
                Poll pollEntity = pollDAO.findByTopiaId(votingListDTO
                        .getPollId());
                eVotingList.setPoll(pollEntity);
            }
        }
        if (votingListDTO.getVoteId() != null) {
            if (votingListDTO.getVoteId().length() > 0) {
                VoteDAO voteDAO = PollenModelDAOHelper
                        .getVoteDAO(super.transaction);
                Vote voteEntity = voteDAO.findByTopiaId(votingListDTO
                        .getVoteId());
                eVotingList.setVote(voteEntity);
            }
        }
    }

    /**
     * Création d'un dto votingList à partir d'une entité.
     *
     * @param eVotingList l'entité votingList
     * @return le dto votingList
     */
    public VotingListDTO createVotingListDTO(VotingList eVotingList) {
        VotingListDTO dto = new VotingListDTO();
        dto.setId(eVotingList.getTopiaId());
        dto.setName(eVotingList.getName());
        dto.setWeight(eVotingList.getWeight());

        dto.setPollAccountDTOs(this.getPollAccountDTOs(eVotingList
                .getPollAccountPersonToList()));

        if (eVotingList.getPoll() != null) {
            dto.setPollId(eVotingList.getPoll().getTopiaId());
        }
        if (eVotingList.getVote() != null) {
            dto.setVoteId(eVotingList.getVote().getTopiaId());
        }

        return dto;
    }

    /**
     * Retourne une liste de dtos votingList à partir d'une liste d'entités.
     *
     * @param lVotingLists liste de entités votingList
     * @return la liste des dtos votingList
     */
    public List<VotingListDTO> createVotingListDTOs(
            List<VotingList> lVotingLists) {
        List<VotingListDTO> results = new ArrayList<VotingListDTO>();
        VotingListDTO dto;
        for (VotingList v : lVotingLists) {
            dto = this.createVotingListDTO(v);
            results.add(dto);
        }
        return results;
    }

    /**
     * Retourne une liste de pollAccount correspondant aux personnes de la
     * liste.
     *
     * @param persons les personnes
     * @return liste des comptes
     */
    private List<PollAccountDTO> getPollAccountDTOs(
            Collection<PersonToList> persons) {

        List<PollAccountDTO> accounts = new ArrayList<PollAccountDTO>();
        DataPollAccountConverter converter = new DataPollAccountConverter();

        for (PersonToList p : persons) {
            accounts.add(converter.createPollAccountDTO(p.getPollAccount()));
        }

        return accounts;
    }

    /**
     * Mise à jour des PersonToList d'une entité votingList à partir d'un dto
     * votingList.
     *
     * @param votingListDTO Le dto votingList.
     * @param eVotingList L'entité votingList.
     */
    public void populatePersonVotingList(VotingListDTO votingListDTO,
            VotingList eVotingList) throws TopiaException {

        // Mise à jour des identifiants de liste dans les pollAccounts
        for (PollAccountDTO pollAccountDTO : votingListDTO.getPollAccountDTOs()) {
            pollAccountDTO.setVotingListId(eVotingList.getTopiaId());
        }

        // Mise à jour des PersonToList correspondants à la liste de votants
        eVotingList.setPollAccountPersonToList(this
                .getPersonToLists(votingListDTO.getPollAccountDTOs()));
    }

    /**
     * Retourne la liste des PersonToList à partir d'une liste de dtos
     * votingList.
     *
     * @param pollAccountDTOs La liste des dtos votingList
     * @return la liste d'entités PersonToList
     */
    private List<PersonToList> getPersonToLists(
            List<PollAccountDTO> pollAccountDTOs) throws TopiaException {
        List<PersonToList> liste = new ArrayList<PersonToList>();

        for (PollAccountDTO pollAccountDTO : pollAccountDTOs) {
            PersonToListDAO personToListDAO = PollenModelDAOHelper
                    .getPersonToListDAO(super.transaction);
            PersonToList personToListEntity = personToListDAO.create();
            personToListEntity.setWeight(pollAccountDTO.getWeight());

            // Ajout de la votingList
            VotingListDAO votingListDAO = PollenModelDAOHelper
                    .getVotingListDAO(super.transaction);
            VotingList votingListEntity = votingListDAO
                    .findByTopiaId(pollAccountDTO.getVotingListId());
            personToListEntity.setVotingList(votingListEntity);

            // Ajout du pollAccount
            PollAccountDAO pollAccountDAO = PollenModelDAOHelper
                    .getPollAccountDAO(super.transaction);
            PollAccount pollAccountEntity = pollAccountDAO
                    .findByTopiaId(pollAccountDTO.getId());
            personToListEntity.setPollAccount(pollAccountEntity);

            liste.add(personToListEntity);
        }

        return liste;
    }
}
