/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business;

import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.converters.DataVoteConverter;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.VoteDTO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.business.persistence.VoteDAO;
import org.chorem.pollen.business.utils.ContextUtil;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

/**
 * Implémentation du service de gestion des votes.
 *
 * @author rannou
 * @version $Id: ServiceVoteImpl.java 2621 2009-07-03 15:09:23Z nrannou $
 */
public class ServiceVoteImpl implements ServiceVote {
    private TopiaContext rootContext = ContextUtil.getInstance().getContext();
    private TopiaContext transaction = null;
    private VoteDAO voteDAO = null;
    private DataVoteConverter converter = new DataVoteConverter();

    /** log. */
    private static final Log log = LogFactory.getLog(ServiceVoteImpl.class);

    public ServiceVoteImpl() {
    }

    @Override
    public String createVote(VoteDTO voteDTO, PollAccountDTO pollAccountDTO) {
        String topiaId = null;
        try {
            transaction = rootContext.beginTransaction();

            // création du compte associé au vote s'il n'existe pas
            // sinon mise à jour du compte
            ServicePollAccount spa = new ServicePollAccountImpl();
            if (pollAccountDTO != null) {
                PollAccountDTO account = spa.findPollAccountById(pollAccountDTO
                        .getId());
                if (account != null) {
                    log.debug("Compte associé au vote : " + account.getId());
                    spa.updatePollAccount(pollAccountDTO);
                    voteDTO.setPollAccountId(account.getId());
                } else {
                    log.debug("Nouveau compte associé au vote");
                    voteDTO.setPollAccountId(spa
                            .createPollAccount(pollAccountDTO));
                }
                voteDTO.setVotingListId(pollAccountDTO.getVotingListId());
                voteDTO.setWeight(pollAccountDTO.getWeight());
                log.debug("Poids du vote : " + voteDTO.getWeight());
            } else {
                voteDTO.setPollAccountId(null);
            }

            voteDAO = PollenModelDAOHelper.getVoteDAO(transaction);

            Vote voteEntity = voteDAO.create();
            converter.setTransaction(transaction);
            converter.populateVoteEntity(voteDTO, voteEntity);

            transaction.commitTransaction();

            converter.populateChoiceVote(voteDTO, voteEntity);

            topiaId = voteEntity.getTopiaId();
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity created: " + topiaId);
            }

            return topiaId;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public boolean updateVote(VoteDTO voteDTO) {
        try {
            transaction = rootContext.beginTransaction();

            voteDAO = PollenModelDAOHelper.getVoteDAO(transaction);
            Vote voteEntity = voteDAO.findByTopiaId(voteDTO.getId());
            converter.setTransaction(transaction);
            converter.populateVoteEntity(voteDTO, voteEntity);
            voteEntity.update();

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity updated: " + voteDTO.getId());
            }

            return true;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }
    }

    @Override
    public boolean deleteVote(VoteDTO voteDTO) {
        try {
            transaction = rootContext.beginTransaction();

            voteDAO = PollenModelDAOHelper.getVoteDAO(transaction);
            Vote voteEntity = voteDAO.findByTopiaId(voteDTO.getId());
            voteEntity.delete();
            // TODO : supprimer ce vote dans les autres tables ou cela se fait tout seul ? (DELETE ou DELETE CASCADE ?)

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity deleted: " + voteDTO.getId());
            }

            return true;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }
    }

    @Override
    public List<VoteDTO> selectVotes(Map<String, Object> properties) {
        List<VoteDTO> results = null;
        List<Vote> voteEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            voteDAO = PollenModelDAOHelper.getVoteDAO(transaction);

            if (properties == null) {
                voteEntities = voteDAO.findAll();
                if (log.isWarnEnabled()) {
                    log
                            .warn("Attention : tous les votes ont été sélectionnés !");
                }
            } else {
                voteEntities = voteDAO.findAllByProperties(properties);
            }

            converter.setTransaction(transaction);
            results = converter.createVoteDTOs(voteEntities);

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    /**
     * Méthode exécutée lorsqu'une exception est détéctée.
     *
     * @param e l'exception
     */
    private void doCatch(TopiaException e) {

        // rollback de la transaction courante
        try {
            if (transaction != null) {
                transaction.rollbackTransaction();
                transaction.closeContext();
            }
        } catch (TopiaException ex) {
            ex.printStackTrace();
        }
        e.printStackTrace();
    }
}