/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business;

import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.converters.DataUserConverter;
import org.chorem.pollen.business.dto.UserDTO;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountDAO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.UserAccountDAO;
import org.chorem.pollen.business.utils.ContextUtil;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

/**
 * Implémentation du service de gestion des utilisateurs.
 *
 * @author amine
 * @author rannou
 * @version $Id: ServiceUserImpl.java 2621 2009-07-03 15:09:23Z nrannou $
 */
public class ServiceUserImpl implements ServiceUser {
    private TopiaContext rootContext = ContextUtil.getInstance().getContext();
    private TopiaContext transaction = null;
    private UserAccountDAO userDAO = null;
    private DataUserConverter converter = new DataUserConverter();

    /** log. */
    private static final Log log = LogFactory.getLog(ServiceUserImpl.class);

    public ServiceUserImpl() {
    }

    @Override
    public String createUser(UserDTO user, String password) {
        String topiaId = null;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);
            UserAccount userEntity = userDAO.findByLogin(user.getLogin());

            if (userEntity == null) {
                userEntity = userDAO.create();
                converter.populateUserEntity(user, userEntity, password);

                topiaId = userEntity.getTopiaId();
                transaction.commitTransaction();
                transaction.closeContext();

                if (log.isInfoEnabled()) {
                    log.info("Entity created: " + topiaId);
                }
            }

            return topiaId;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public boolean deleteUser(String login) {
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);

            UserAccount userEntity = userDAO.findByLogin(login);

            userDAO.delete(userEntity);
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity deleted: " + userEntity.getTopiaId());
            }

            return true;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }
    }

    @Override
    public boolean deleteUser(UserDTO user) {
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);

            UserAccount userEntity = userDAO.findByTopiaId(user.getId());
            userDAO.delete(userEntity);
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity deleted: " + userEntity.getTopiaId());
            }

            return true;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }
    }

    @Override
    public UserDTO findUserById(String userId) {
        UserDTO result = null;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);

            UserAccount userEntity = userDAO.findByTopiaId(userId);

            if (userEntity != null) {
                converter.setTransaction(this.transaction);
                result = converter.createUserDTO(userEntity);
            }

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity found: "
                        + ((result == null) ? "null" : result.getId()));
            }

            return result;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public List<UserDTO> selectUsers(Map<String, Object> properties) {
        List<UserDTO> results = null;
        List<UserAccount> userEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);

            if (properties == null) {
                userEntities = userDAO.findAll();
                if (log.isWarnEnabled()) {
                    log
                            .warn("Attention : tous les utilisateurs ont été sélectionnés !");
                }
            } else {
                userEntities = userDAO.findAllByProperties(properties);
            }

            results = converter.createUserDTOs(userEntities);

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public boolean updateUser(UserDTO user) {
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);

            UserAccount userEntity = userDAO.findByTopiaId(user.getId());

            converter.setTransaction(this.transaction);
            converter.populateUserEntity(user, userEntity);

            userDAO.update(userEntity);
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity updated: " + user.getId());
            }

            return true;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }

    }

    @Override
    public boolean addPollAccountToUser(String login, String pollAccountId) {
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);

            UserAccount userEntity = userDAO.findByLogin(login);
            PollAccountDAO pollAccountDAO = PollenModelDAOHelper
                    .getPollAccountDAO(transaction);
            PollAccount pollAccountEntity = pollAccountDAO
                    .findByTopiaId(pollAccountId);
            userEntity.addPollAccount(pollAccountEntity);
            userEntity.update();
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity updated: " + userEntity.getTopiaId());
            }

            return true;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }
    }

    @Override
    public boolean updatePasswordUser(UserDTO user, String newPassword) {
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);

            UserAccount userEntity = userDAO.findByTopiaId(user.getId());

            userEntity.setPassword(newPassword);
            userEntity.update();
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity updated: " + user.getId());
            }

            return true;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }
    }

    /**
     * Méthode exécutée lorsqu'une exception est détéctée.
     *
     * @param e l'exception
     */
    private void doCatch(TopiaException e) {

        // rollback de la transaction courante
        try {
            if (transaction != null) {
                transaction.rollbackTransaction();
                transaction.closeContext();
            }
        } catch (TopiaException ex) {
            ex.printStackTrace();
        }
        e.printStackTrace();
    }
}