/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.converters.DataPollConverter;
import org.chorem.pollen.business.dto.ChoiceDTO;
import org.chorem.pollen.business.dto.PollDTO;
import org.chorem.pollen.business.dto.PreventRuleDTO;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollDAO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.persistence.PreventRule;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.UserAccountDAO;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.business.persistence.VoteDAO;
import org.chorem.pollen.business.utils.ContextUtil;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

/**
 * Implémentation du service de gestion des sondages.
 *
 * @author Erwan Nema
 * @author rannou
 * @version $Id: ServicePollImpl.java 2621 2009-07-03 15:09:23Z nrannou $
 */
public class ServicePollImpl implements ServicePoll {
    private TopiaContext rootContext = ContextUtil.getInstance().getContext();
    private TopiaContext transaction = null;
    private PollDAO pollDAO = null;
    private DataPollConverter converter = new DataPollConverter();

    /** log. */
    private static final Log log = LogFactory.getLog(ServicePollImpl.class);

    public ServicePollImpl() {
    }

    @Override
    public String createPoll(PollDTO pollDTO) {
        String topiaId = "";
        try {
            transaction = rootContext.beginTransaction();

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);

            Poll pollEntity = pollDAO.create();

            if (log.isDebugEnabled()) {
                log.debug("service " + pollDTO.getVoteCounting() + " "
                        + pollDTO.getPollType());
            }

            converter.setTransaction(transaction);
            converter.populatePollEntity(pollDTO, pollEntity);

            // Création du pollAccount associé au sondage
            ServicePollAccount spa = new ServicePollAccountImpl();
            if (log.isDebugEnabled()) {
                log.debug("userId : " + pollDTO.getUserId());
            }
            PollAccount pollAccountEntity = spa.createPollAccount(pollDTO
                    .getCreatorId(), pollDTO.getCreatorEmail(), pollDTO
                    .getUserId());
            if (log.isDebugEnabled()) {
                log
                        .debug("pollAccountEntity created: "
                                + pollAccountEntity != null);
            }
            pollEntity.setCreator(pollAccountEntity);

            topiaId = pollEntity.getTopiaId();

            // Création des choix du sondage
            ServiceChoice sChoice = new ServiceChoiceImpl();
            pollEntity.setChoice(sChoice.createChoices(pollDTO.getChoiceDTOs(),
                    topiaId, transaction));

            // Création des commentaires du sondage
            ServiceComment sComment = new ServiceCommentImpl();
            pollEntity.setComment(sComment.createComments(pollDTO
                    .getCommentDTOs(), topiaId, transaction));

            // Création des listes de votants du sondage
            ServiceList sList = new ServiceListImpl();
            pollEntity.setVotingList(sList.createVotingLists(pollDTO
                    .getVotingListDTOs(), topiaId, transaction));

            // Création des règle de notification pour le sondage
            ServicePreventRule sPreventRule = new ServicePreventRuleImpl();
            pollEntity.setPreventRule(sPreventRule.createPreventRules(pollDTO
                    .getPreventRuleDTOs(), topiaId, transaction));

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("creator after creation: " + pollEntity.getCreator());
            }
            if (log.isInfoEnabled()) {
                log.info("Entity created: " + topiaId);
            }

            return topiaId;
        } catch (TopiaException e) {
            doCatch(e);
            return "";
        }
    }

    @Override
    public boolean updatePoll(PollDTO pollDTO) {
        try {
            transaction = rootContext.beginTransaction();

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);

            Poll pollEntity = pollDAO.findByTopiaId(pollDTO.getId());

            converter.setTransaction(transaction);
            converter.populatePollEntity(pollDTO, pollEntity);

            pollDAO.update(pollEntity);

            // mise à jour ou création des choix
            ServiceChoice sChoice = new ServiceChoiceImpl();
            for (ChoiceDTO choiceDTO : pollDTO.getChoiceDTOs()) {
                boolean updated = sChoice.updateChoice(choiceDTO);
                if (!updated) {
                    choiceDTO.setId(sChoice.createChoice(choiceDTO));
                }

                if (log.isDebugEnabled()) {
                    log.debug("Choice " + choiceDTO.getName() + " ("
                            + choiceDTO.getId() + ") updated: " + updated);
                }
            }

            // mise à jour ou création des règles de notification
            ServicePreventRule sPreventRule = new ServicePreventRuleImpl();
            for (PreventRuleDTO preventRuleDTO : pollDTO.getPreventRuleDTOs()) {
                boolean updated = sPreventRule
                        .updatePreventRule(preventRuleDTO);
                if (!updated) {
                    preventRuleDTO.setId(sPreventRule
                            .createPreventRule(preventRuleDTO));
                }

                if (log.isDebugEnabled()) {
                    log.debug("PreventRule (" + preventRuleDTO.getId()
                            + ") updated: " + updated);
                }
            }

            // suppression des règles de notification
            boolean valid = false;
            Iterator<PreventRule> it = pollEntity.getPreventRule().iterator();
            while (it.hasNext()) {
                PreventRule currentRule = it.next();
                for (PreventRuleDTO preventRuleDTO : pollDTO
                        .getPreventRuleDTOs()) {
                    if (currentRule.getTopiaId().equals(preventRuleDTO.getId())) {
                        valid = true;
                    }
                }
                if (!valid) {
                    sPreventRule.deletePreventRule(currentRule.getTopiaId());
                }
            }

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity updated: " + pollDTO.getId());
            }

            return true;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }
    }

    @Override
    public boolean deletePoll(String pollId) {
        try {
            transaction = rootContext.beginTransaction();

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);

            Poll pollEntity = pollDAO.findByTopiaId(pollId);

            pollDAO.delete(pollEntity);
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity deleted: " + pollId);
            }

            return true;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }
    }

    @Override
    public PollDTO findPollById(String pollId) {
        PollDTO result = null;
        try {
            transaction = rootContext.beginTransaction();

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);

            Poll pollEntity = pollDAO.findByTopiaId(pollId);

            if (pollEntity != null) {
                converter.setTransaction(transaction);
                result = converter.createPollDTO(pollEntity);
            }

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity found: "
                        + ((result == null) ? "null" : result.getId()));
            }

            return result;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public PollDTO findPollByPollId(String pollId) {
        PollDTO result = null;
        try {
            transaction = rootContext.beginTransaction();

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);

            Poll pollEntity = pollDAO.findByPollId(pollId);

            if (pollEntity != null) {
                converter.setTransaction(transaction);
                result = converter.createPollDTO(pollEntity);
            }

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity found: "
                        + ((result == null) ? "null" : result.getId()));
            }

            return result;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public List<PollDTO> findPollsByName(String pollName) {
        List<PollDTO> results = null;
        List<Poll> pollEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);

            pollEntities = pollDAO.findAllByTitle(pollName);

            converter.setTransaction(transaction);
            results = converter.createPollDTOs(pollEntities);

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public List<PollDTO> findPollsByUser(String userId) {
        List<PollDTO> results = null;
        UserAccountDAO userDAO = null;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);
            UserAccount user = userDAO.findByTopiaId(userId);

            List<Poll> pollEntities = new ArrayList<Poll>();
            for (PollAccount pollAccount : user.getPollAccount()) {
                pollEntities.addAll(pollAccount.getPollsCreated());
            }

            converter.setTransaction(transaction);
            results = converter.createPollDTOs(pollEntities);

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public List<PollDTO> findRunningPolls(boolean withEndDate) {
        List<PollDTO> results = null;
        List<Poll> pollEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            // requête de sélection des sondages
            if (withEndDate) {
                pollEntities = transaction
                        .find("from "
                                + Poll.class.getName()
                                + " as poll where poll.endDate is not null and poll.endDate > current_timestamp()");
            } else {
                pollEntities = transaction
                        .find("from "
                                + Poll.class.getName()
                                + " as poll where poll.endDate is null or poll.endDate > current_timestamp()");
            }

            converter.setTransaction(transaction);
            results = converter.createPollDTOs(pollEntities);

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public List<PollDTO> selectPolls(Map<String, Object> properties) {
        List<PollDTO> results = null;
        List<Poll> pollEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);

            if (properties == null) {
                pollEntities = pollDAO.findAll();
                if (log.isWarnEnabled()) {
                    log
                            .warn("Attention : tous les sondages ont été sélectionnés !");
                }
            } else {
                pollEntities = pollDAO.findAllByProperties(properties);
            }
            converter.setTransaction(transaction);
            results = converter.createPollDTOs(pollEntities);

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public String moderate(String pollId) {
        this.deletePoll(pollId);
        return "the poll has been deleted";
    }

    @Override
    public boolean addVoteToPoll(String pollId, String voteId) {
        try {
            transaction = rootContext.beginTransaction();

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);
            Poll pollEntity = pollDAO.findByTopiaId(pollId);
            VoteDAO voteDAO = PollenModelDAOHelper.getVoteDAO(transaction);
            Vote voteEntity = voteDAO.findByTopiaId(voteId);

            if (log.isDebugEnabled()) {
                log.debug(pollEntity + " " + voteEntity);
            }

            pollEntity.addVote(voteEntity);
            pollEntity.update();

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity updated: " + pollId);
            }

            return true;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }
    }

    /**
     * Méthode exécutée lorsqu'une exception est détéctée.
     *
     * @param e l'exception
     */
    private void doCatch(TopiaException e) {

        // rollback de la transaction courante
        try {
            if (transaction != null) {
                transaction.rollbackTransaction();
                transaction.closeContext();
            }
        } catch (TopiaException ex) {
            ex.printStackTrace();
        }
        e.printStackTrace();
    }
}