/* *##%
 * Copyright (c) 2009 ruchaud. All rights reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty.jpa;

import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.UpdateResponse;
import org.sharengo.wikitty.Wikitty;
import org.sharengo.wikitty.WikittyException;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyExtensionStorage;
import org.sharengo.wikitty.WikittyStorage;
import org.sharengo.wikitty.WikittyTransaction;
import org.sharengo.wikitty.WikittyUtil;
import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.PersistenceUnit;
import javax.persistence.Query;
import org.springframework.beans.factory.annotation.Autowired;

/**
 *
 * @author ruchaud
 * @version $Revision: 1 $
 *
 * Last update: $Date: 2010-04-16 10:29:38 +0200 (ven., 16 avril 2010) $
 * by : $Author: echatellier $
 */
public class WikittyStorageJPA implements WikittyStorage {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyStorageJPA.class);

    /** storage for extension */
    @Autowired
    protected WikittyExtensionStorage extensionStorage;

    @PersistenceUnit
    protected EntityManagerFactory entityManagerFactory;

    public WikittyStorageJPA() {
    }

    @Override
    public UpdateResponse store(WikittyTransaction transaction,
            Collection<Wikitty> wikitties,
            boolean disableAutoVersionIncrement) throws WikittyException {
        EntityManager entityManager = WikittyJPAUtil.getEntityManager(entityManagerFactory, transaction);

        UpdateResponse result = new UpdateResponse();
        for (Wikitty wikitty : wikitties) {
            String wikittyId = wikitty.getId();

            String actualVersion = null;
            String requestedVersion = wikitty.getVersion();

            // test if wikitty version is more recent that database version
            Wikitty wikittyFound = entityManager.find(Wikitty.class, wikittyId);
            if(wikittyFound != null) {
                actualVersion = wikittyFound.getVersion();
            }

            // compute new version, but not change wikitty during prepare
            String newVersion = null;
            if (disableAutoVersionIncrement) {
                if (actualVersion == null) { //no version in place
                    if (requestedVersion == null) { //no version requested
                        newVersion = WikittyUtil.DEFAULT_VERSION;
                    } else { //version requested
                        newVersion = requestedVersion;
                    }
                } else { //version in place is not null
                    if (requestedVersion == null) { //no version requested
                        newVersion = WikittyUtil.incrementMajorRevision(actualVersion);
                    } else if (WikittyUtil.versionEquals(actualVersion, requestedVersion)) { //same version
                        // wikitty is not modified, do nothing
                        continue;
                    } else if (WikittyUtil.versionGreaterThan(requestedVersion, actualVersion)) { //requested version is newer
                        newVersion = requestedVersion;
                    } else { //requested version is obsolete
                        throw new WikittyException(String.format(
                                "Your wikitty '%s' is obsolete (saving: '%s'; existing: '%s')", wikittyId, requestedVersion, actualVersion));
                    }
                }
            } else {
                if (WikittyUtil.versionEquals(actualVersion, requestedVersion)) {
                    // wikitty is not modified, do nothing
                    continue;
                } else if (WikittyUtil.versionGreaterThan(actualVersion, requestedVersion)) {
                    throw new WikittyException(String.format(
                            "Your wikitty '%s' is obsolete",wikittyId));
                } else {
                    newVersion = WikittyUtil.incrementMajorRevision(actualVersion);
                }
            }

            for (WikittyExtension ext : wikitty.getExtensions()) {
                String extName = ext.getName();
                for (String fieldName : ext.getFieldNames()) {
                    FieldType fieldType = ext.getFieldType(fieldName);

                    Object value = wikitty.getFieldAsObject(extName, fieldName);
                    if(fieldType.isNotNull() && value == null) {
                        throw new WikittyException(String.format(
                                    "Field %s in extension %s can't be null",
                                    fieldName,extName));
                    }
                }
            }

            // FIXME poussin 20090831 perhaps we must not modify wikitty, and user must use UpdateResponse to update their objects
            wikitty.setVersion(newVersion);
            wikitty.clearDirty();

            if(actualVersion == null) {
                entityManager.persist(wikitty);
            } else {
                entityManager.merge(wikitty);
            }
            
            result.addVersionUpdate(wikitty.getId(), newVersion);
        }
        
        return result;
    }

    @Override
    public UpdateResponse delete(WikittyTransaction transaction, Collection<String> ids) throws WikittyException {
        try {
            EntityManager entityManager = WikittyJPAUtil.getEntityManager(entityManagerFactory, transaction);
            UpdateResponse result = new UpdateResponse();
            Date now = new Date();

            for (String id : ids) {
                if (!exists(transaction, id)) {
                    throw new WikittyException(String.format(
                            "Wikitty with id '%s' don't exists", id));
                }
                // addVersionUpdate delete date field
                Wikitty wikitty = entityManager.find(Wikitty.class, id);
                wikitty.setDeleteDate(now);
                entityManager.persist(wikitty);

                result.addDeletionDateUpdate(id, now);
            }

            return result;
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    @Override
    public boolean exists(WikittyTransaction transaction, String id) {
        EntityManager entityManager = WikittyJPAUtil.getEntityManager(entityManagerFactory, transaction);
        Wikitty wikitty = entityManager.find(Wikitty.class, id);
        boolean result = wikitty != null;
        return result;
    }

    @Override
    public boolean isDeleted(WikittyTransaction transaction, String id) {
        EntityManager entityManager = WikittyJPAUtil.getEntityManager(entityManagerFactory, transaction);
        Wikitty wikitty = entityManager.find(Wikitty.class, id);
        Date deleteDate = wikitty.getDeleteDate();
        boolean result = deleteDate != null;
        return result;
    }

    @Override
    public Wikitty restore(WikittyTransaction transaction, String id, String ... fqFieldNames) {
        EntityManager entityManager = WikittyJPAUtil.getEntityManager(entityManagerFactory, transaction);
        Wikitty wikitty = entityManager.find(Wikitty.class, id);
        if(wikitty == null) {
            throw new WikittyException(String.format(
                "The wikitty not exsist with id '%s'", id));
        }
        return wikitty;
    }

    @Override
    public void scanWikitties(WikittyTransaction transaction, Scanner scanner) {
        EntityManager entityManager = WikittyJPAUtil.getEntityManager(entityManagerFactory, transaction);
        Query query = entityManager.createQuery("SELECT we FROM Wikitty we");
        List<Wikitty> results = query.getResultList();

        Iterator<Wikitty> iterator = results.iterator();
        while (iterator.hasNext()) {
            Wikitty wikitty = iterator.next();
            scanner.scan(wikitty);
        }
    }

    @Override
    public void clear(WikittyTransaction transaction) {
        // Nothing do
    }
}
