package org.sharengo.wikitty.jpa;

/* *##%
 * Copyright (c) 2009 ruchaud. All rights reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.PersistenceUnit;
import javax.persistence.Query;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.UpdateResponse;
import org.sharengo.wikitty.WikittyException;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyExtensionStorage;
import org.sharengo.wikitty.WikittyTransaction;
import org.sharengo.wikitty.WikittyUtil;

/**
 *
 * @author ruchaud
 * @version $Revision: 1 $
 *
 * Last update: $Date: 2010-04-16 10:29:38 +0200 (ven., 16 avril 2010) $
 * by : $Author: echatellier $
 */
public class WikittyExtensionStorageJPA implements WikittyExtensionStorage {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyExtensionStorageJPA.class);

    @PersistenceUnit
    protected EntityManagerFactory entityManagerFactory;

    /** cache for last extension version; key: extName value: extVersion */
    protected Map<String, String> lastVersion = null;

    public WikittyExtensionStorageJPA() {
    }

    @Override
    public UpdateResponse store(WikittyTransaction transaction,
            Collection<WikittyExtension> extensions) throws WikittyException {
        EntityManager entityManager = WikittyJPAUtil.getEntityManager(entityManagerFactory, transaction);
        for (WikittyExtension ext : extensions) {
           // extension id is extension name with version
           String extId = ext.getId();
           String extName = ext.getName();
           String extVersion = ext.getVersion();

           String extLastVersion = getLastVersion(transaction, extName);
           // else no permitted to update with the less version
           if(WikittyUtil.versionGreaterThan(extVersion, extLastVersion)) {
               entityManager.persist(ext);
               getLastVersion(transaction).put(extName, extVersion);
           } else {
               if(!WikittyUtil.versionEquals(extVersion, extLastVersion)) {
                    throw new WikittyException(String.format(
                            "Your extension '%s' is obsolete", extId));
               }
           }
        }

        // nothing to put in result
        UpdateResponse result = new UpdateResponse();
        return result;
    }

    @Override
    public boolean exists(WikittyTransaction transaction, String id) {
        EntityManager entityManager = WikittyJPAUtil.getEntityManager(entityManagerFactory, transaction);
        WikittyExtension extensionEntity = entityManager.find(WikittyExtension.class, id);
        boolean result = extensionEntity != null;
        return result;
    }

    @Override
    public List<String> getAllExtensionIds(WikittyTransaction transaction) {
        EntityManager entityManager = WikittyJPAUtil.getEntityManager(entityManagerFactory, transaction);
        Query query = entityManager.createQuery("SELECT we FROM WikittyExtension we");
        List<WikittyExtension> extensions = query.getResultList();

        List<String> result = new ArrayList<String>();
        for (WikittyExtension extension : extensions) {
            String name = extension.getName();
            String version = extension.getVersion();
            String id = WikittyExtension.computeId(name, version);
            result.add(id);
        }
        return result;
    }

    @Override
    public List<String> getAllExtensionsRequires(
            WikittyTransaction transaction, String extensionName) {
        EntityManager entityManager = WikittyJPAUtil.getEntityManager(entityManagerFactory, transaction);
        Query query = entityManager.createQuery("SELECT we FROM WikittyExtension we where requires=:extensionName");
        query.setParameter("extensionName", extensionName);
        List<WikittyExtension> extensions = query.getResultList();

        List<String> result = new ArrayList<String>();
        for (WikittyExtension extension : extensions) {
            String name = extension.getName();
            String version = extension.getVersion();
            String id = WikittyExtension.computeId(name, version);
            result.add(id);
        }
        return result;
    }
    
    /**
     * return last version available for specified extension name
     * @param name name of extension
     * @return last version availble for this version, or null if extension
     * doesn't exist
     */
    @Override
    public String getLastVersion(WikittyTransaction transaction, String extName) {
        String result = getLastVersion(transaction).get(extName);
        log.debug("LastVersion=" + result + " for " + extName);
        return result;
    }

    protected Map<String, String> getLastVersion(WikittyTransaction transaction) {
        if(lastVersion == null) {
            // create cache for futur call
            // FIXME: jru 28/12/2009 just one query to return the last and remove cache
            lastVersion = new HashMap<String, String>();

            EntityManager entityManager = WikittyJPAUtil.getEntityManager(entityManagerFactory, transaction);
            Query query = entityManager.createQuery("SELECT we FROM WikittyExtension we");
            List<WikittyExtension> results = query.getResultList();

            Iterator<WikittyExtension> iterator = results.iterator();
            while (iterator.hasNext()) {
                WikittyExtension extensionEntity = iterator.next();
                String name = extensionEntity.getName();
                String version = extensionEntity.getVersion();
                String prevVersion = lastVersion.get(name);
                if (prevVersion == null ||
                        WikittyUtil.versionGreaterThan(version, prevVersion)) {
                    lastVersion.put(name, version);
                }
            }
        }
        return lastVersion;
    }
    
    @Override
    public WikittyExtension restore(WikittyTransaction transaction, String name, String version)
            throws WikittyException {
        EntityManager entityManager = WikittyJPAUtil.getEntityManager(entityManagerFactory, transaction);

        Query query = entityManager.createQuery(
                "SELECT we FROM WikittyExtension we " +
                "where name=:name and version=:version");
        query.setParameter("name", name);
        query.setParameter("version", version);

        WikittyExtension extension = (WikittyExtension) query.getSingleResult();
        if(extension == null) {
            log.debug("Extension not found");
            return null;
        }
        log.debug("Extension restored=" + extension);
        return extension;
    }

    @Override
    public void clear(WikittyTransaction transaction) {
        // Nothing do
    }
}
