/* *##%
 * Copyright (c) 2009 ruchaud. All rights reserved.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty.jms;

import static org.sharengo.wikitty.jms.WikittyJMSUtil.ACTION_CLEAR;
import static org.sharengo.wikitty.jms.WikittyJMSUtil.ACTION_STORE;
import static org.sharengo.wikitty.jms.WikittyJMSUtil.JNDI_PROVIDER_URL;
import static org.sharengo.wikitty.jms.WikittyJMSUtil.PARAM_EXTENSIONS;
import static org.sharengo.wikitty.jms.WikittyJMSUtil.TOPIC_EXTENSION_STORAGE;

import java.util.ArrayList;
import java.util.Properties;

import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicSession;
import javax.jms.TopicSubscriber;
import javax.naming.NamingException;

import org.apache.activemq.ActiveMQConnectionFactory;
import org.apache.activemq.command.ActiveMQTopic;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.WikittyException;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyExtensionStorage;
import org.sharengo.wikitty.WikittyTransaction;

/**
 * WikittyExtensionSubscriberJMS.
 *
 * @author $Author: echatellier $
 * @version $Revision: 1 $ $Date: 2010-04-16 10:29:38 +0200 (ven., 16 avril 2010) $
 * @since 11 févr. 2010 17:58:34
 */
public class WikittyExtensionSubscriberJMS implements MessageListener {
    private static final Log log = LogFactory.getLog(WikittyExtensionSubscriberJMS.class);

    protected final TopicConnectionFactory connectionFactory;
    protected final TopicConnection topicConnection;
    protected final Topic topic;

    protected final TopicSession topicSession;
    protected final TopicSubscriber topicSubscriber;
    
    protected final WikittyExtensionStorage delegateExtensionStorage;
    
    /**
     * Creates a durable subscriber on a topic named <code>WikittyExtensionStorageTopic</code>.
     * All received messages are dispatched to the specified <code>delegateExtensionStorage</code>.
     * 
     * @param delegateExtensionStorage  an instance of <code>WikittyExtensionStorage</code>
     * @param properties properties passed to <code>InitialContext</code>,
     *         it must contains at least a value for the <code>Context.PROVIDER_URL</code>
     * @throws NamingException
     * @throws JMSException
     */
    public WikittyExtensionSubscriberJMS(WikittyExtensionStorage delegateExtensionStorage,
            Properties properties) throws NamingException, JMSException {
        
        if (delegateExtensionStorage == null) {
            throw new IllegalArgumentException("delegateExtensionStorage is null");
        }
        this.delegateExtensionStorage = delegateExtensionStorage;
        
        topic = new ActiveMQTopic(TOPIC_EXTENSION_STORAGE);

        String provider = properties.getProperty(JNDI_PROVIDER_URL);
        connectionFactory = new ActiveMQConnectionFactory(provider);
        
        topicConnection = connectionFactory.createTopicConnection();
        topicConnection.setClientID("WikittyExtensionSubscriberJMS");
        topicSession = topicConnection.createTopicSession(true, Session.SESSION_TRANSACTED);
        
        topicSubscriber = topicSession.createDurableSubscriber(topic, "WikittyExtensionSubscriberJMS");
        topicSubscriber.setMessageListener(this);
    }
    
    /*
     * Before your application can consume messages,
     * you must call the connection start method
     */
    public void start() throws Exception {
        topicConnection.start();
    }
    
    /*
     * If you want to stop message delivery temporarily
     * without closing the connection, you call the stop method.
     */
    public void stop() throws Exception {
        topicConnection.stop();
    }
    
    /*
     * Closing a connection also closes its sessions and
     * their message producers and message consumers.
     */
    public void close() throws JMSException {
        topicConnection.close();
    }
    
    @Override
    protected void finalize() throws Throwable {
        topicConnection.close();
    }
    
    /*
     * Your onMessage method should handle all exceptions. It must not throw
     * checked exceptions, and throwing a RuntimeException is considered a programming error.
     */
    /**
     * Forwards the message to the <code>delegateExtensionStorage</code> according
     * to its type (TextMessage or ObjectMessage) and its client properties.
     * More precisely, the <code>action</code> client property determines the
     * delegateExtensionStorage's method which will be invoked.
     * 
     * @param message the incoming message
     */
    @Override
    public void onMessage(Message message) {
        final WikittyTransaction transaction = new WikittyTransaction();
        boolean sessionCommit = false;
        
        try {
            transaction.begin();
            
            if (log.isDebugEnabled()) {
                WikittyJMSUtil.logMessageHeaders(log, message);
            }
            
            if (message instanceof ObjectMessage) {
                WikittyActionMessage actionMessage =
                        WikittyActionMessage.createReceiveMessage((ObjectMessage) message);
                onActionMessage(transaction, actionMessage);
                
            } else {
                throw new WikittyException("Unsupported message: " +
                        message.getClass().getName());
            }
            
            transaction.commit();
            sessionCommit = true;
            
        } catch (Exception e) {
            log.error("Exception while message dispatch", e);
            
            try {
                transaction.rollback();
                
            } catch (Exception e1) {
                log.error("Exception while transaction rollback", e1);
            }
            
            try {
                topicSession.rollback();
                
            } catch (JMSException e1) {
                log.error("JMSException while session rollback", e1);
            }
        }
    
        if (sessionCommit) {
            try {
                topicSession.commit();
            } catch (JMSException e) {
                log.error("JMSException while session commit", e);
            }
        }
    }
    
    protected void onActionMessage(WikittyTransaction transaction, WikittyActionMessage message) {
        try {
            String action = message.getAction();
            
            if (ACTION_STORE.equals(action)) {
                actionStore(transaction, message);
                
            } else if (ACTION_CLEAR.equals(action)) {
                actionClear(transaction, message);
                
            } else {
                throw new UnsupportedOperationException("Unsupported action: " + action);
            }
            
        } catch (Exception e) {
            throw new WikittyException(e);
        }
    }
    
    protected void actionStore(WikittyTransaction transaction, WikittyActionMessage message)
            throws JMSException {
        ArrayList<WikittyExtension> extensions =
                (ArrayList<WikittyExtension>) message.getParameter(PARAM_EXTENSIONS);

        delegateExtensionStorage.store(transaction, extensions);
    }

    protected void actionClear(WikittyTransaction transaction, WikittyActionMessage message) {
        delegateExtensionStorage.clear(transaction);
    }
}
