/* *##%
 * Copyright (c) 2009 ruchaud. All rights reserved.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty.jms;

import static org.sharengo.wikitty.jms.WikittyJMSUtil.ACTION_CLEAR;
import static org.sharengo.wikitty.jms.WikittyJMSUtil.ACTION_STORE;
import static org.sharengo.wikitty.jms.WikittyJMSUtil.JNDI_PROVIDER_URL;
import static org.sharengo.wikitty.jms.WikittyJMSUtil.PARAM_EXTENSIONS;
import static org.sharengo.wikitty.jms.WikittyJMSUtil.TOPIC_EXTENSION_STORAGE;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Properties;

import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.jms.Topic;
import javax.jms.TopicPublisher;
import javax.jms.XATopicConnection;
import javax.jms.XATopicConnectionFactory;
import javax.jms.XATopicSession;
import javax.naming.NamingException;

import org.apache.activemq.ActiveMQXAConnectionFactory;
import org.apache.activemq.command.ActiveMQTopic;
import org.sharengo.wikitty.UpdateResponse;
import org.sharengo.wikitty.WikittyException;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyExtensionStorage;
import org.sharengo.wikitty.WikittyTransaction;

/**
 * @author ruchaud
 * @version $Revision: 1 $
 *
 * Last update: $Date: 2010-04-16 10:29:38 +0200 (ven., 16 avril 2010) $
 * by : $Author: echatellier $
 */
public class WikittyExtensionStorageJMS implements WikittyExtensionStorage {

    final protected XATopicConnectionFactory connectionFactory;
    final protected Topic topic;

    final protected XATopicConnection topicConnection;
    final protected XATopicSession topicSession;
    final protected TopicPublisher topicPublisher;

    /**
     * Creates a publisher on a topic named <code>WikittyExtensionStorageTopic</code>.
     * Methods invoked on this <code>WikittyExtensionStorage</code> are forwarded to the jms broker.
     * 
     * @param properties properties passed to <code>InitialContext</code>,
     *         it must contains at least a value for the <code>Context.PROVIDER_URL</code>
     * @throws NamingException
     * @throws JMSException
     */

    public WikittyExtensionStorageJMS(Properties properties) throws NamingException, JMSException {
        topic = new ActiveMQTopic(TOPIC_EXTENSION_STORAGE);

        String provider = properties.getProperty(JNDI_PROVIDER_URL);
        connectionFactory = new ActiveMQXAConnectionFactory(provider);
        
        topicConnection = connectionFactory.createXATopicConnection();
        topicSession = topicConnection.createXATopicSession();
        topicPublisher = topicSession.getTopicSession().createPublisher(topic);
    }
    
    /*
     * Closing a connection also closes its sessions and
     * their message producers and message consumers.
     */
    public void close() throws JMSException {
        topicConnection.close();
    }
    
    @Override
    protected void finalize() throws Throwable {
        topicConnection.close();
    }

    /**
     * Forwards the specified <code>extensions</code> to the jms broker.
     * 
     * The message forwarded is an <code>ObjectMessage</code>.
     * 
     * It is composed of:
     * <ul>
     *  <li>the specified <code>extensions</code></li>
     *  <li>a string property: action=<code>store</code></li>
     * </ul>
     * 
     * @param transaction
     * @param extensions
     */
    @Override
    public UpdateResponse store(WikittyTransaction transaction, Collection<WikittyExtension> extensions) {
        try {
            WikittyJMSUtil.addXAResource(transaction, topicSession);

            WikittyActionMessage message = new WikittyActionMessage();
            message.setAction(ACTION_STORE);
            message.addParameter(PARAM_EXTENSIONS, new ArrayList<WikittyExtension>(extensions));
            
            ObjectMessage sendMessage = WikittyActionMessage.createSendMessage(message);
            topicPublisher.publish(sendMessage);

            // nothing to put in result
            UpdateResponse result = new UpdateResponse();
            return result;
        } catch (Exception eee) {
            throw new WikittyException("Error in WikittyStorageJMS", eee);
        }
    }

    /**
     * Forwards a <code>TextMessage</code> to the jms broker.
     * 
     * The message forwarded is a <code>TextMessage</code>.
     * 
     * It is composed of:
     * <ul>
     *  <li>a string property: action=<code>clear</code></li>
     * </ul>
     * 
     * @param transaction
     */
    @Override
    public void clear(WikittyTransaction transaction) {
        try {
            WikittyJMSUtil.addXAResource(transaction, topicSession);

            WikittyActionMessage message = new WikittyActionMessage();
            message.setAction(ACTION_CLEAR);
            
            ObjectMessage sendMessage = WikittyActionMessage.createSendMessage(message);
            topicPublisher.publish(sendMessage);
        } catch (Exception eee) {
            throw new WikittyException("Error in WikittyStorageJMS", eee);
        }
    }

    @Override
    public boolean exists(WikittyTransaction transaction, String id) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public List<String> getAllExtensionIds(WikittyTransaction transaction) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public List<String> getAllExtensionsRequires(
            WikittyTransaction transaction, String extensionName) {
        throw new UnsupportedOperationException("Not supported yet.");
    }
    
    @Override
    public String getLastVersion(WikittyTransaction transaction, String extName) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public WikittyExtension restore(WikittyTransaction transaction, String name, String version) throws WikittyException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

}
