/*
 * #%L
 * Wikitty :: api
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2012 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.storage;

import java.math.BigDecimal;
import java.math.MathContext;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.swing.text.html.parser.DTDConstants;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.query.FacetSortType;
import org.nuiton.wikitty.query.FacetTopic;
import org.nuiton.wikitty.query.WikittyQuery;
import org.nuiton.wikitty.query.WikittyQueryMaker;
import org.nuiton.wikitty.query.WikittyQueryResult;
import org.nuiton.wikitty.query.conditions.Condition;
import org.nuiton.wikitty.query.conditions.Element;
import org.nuiton.wikitty.query.conditions.Select;
import org.nuiton.wikitty.services.WikittyTransaction;

/**
 * Ensemble de methode reutilisable dans differente implantation de
 * {@link WikittySearchEngine}
 *
 * @author poussin
 * @version $Revision$
 *
 * Last update: $Date$
 * by : $Author$
 */
public class WikittySearchEngineHelper {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittySearchEngineHelper.class);

    /**
     * Gere le travail pour les requetes ayant un {@link Select}
     *
     * @param searchEngine le searchEngine a utiliser pour les sous requetes
     * @param transaction la transaction a utiliser
     * @param query la requete qui debute par un {@link Select}
     * @return
     */
    static public WikittyQueryResult<String> findAllByQueryWithSelect(
            WikittySearchEngine searchEngine, WikittyTransaction transaction,
            WikittyQuery query) {

        if (!(query.getCondition() instanceof Select)) {
            throw new WikittyException("Query don't start with Select condition");
        } else {
            // gere les conditions qui commence par select
            // il faut recreer deux query, une pour le select, une pour les facettes s'il y en a
            // car le select est execute via une facette, mais cette facette
            // n'a pas les meme limites que les autres facette

            Select select = (Select)query.getCondition();
            Condition newCond;
            if(WikittyExtension.isFqField(select.getElement().getValue())) {
                String extName = WikittyExtension.extractExtensionName(
                        select.getElement().getValue());
                newCond = new WikittyQueryMaker()
                        .and()
                        .exteq(extName)
                        .condition(select.getSubCondition())
                        .getCondition();
            } else {
                newCond = select.getSubCondition();
            }

            // copy de la query pour les facettes
            WikittyQuery queryFacet = query.copy();
            queryFacet.setCondition(newCond);
            queryFacet.setLimit(0);

            // copy de la query pour le select
            // on part de facet qui a deja la bonne condition
            WikittyQuery querySelect = queryFacet.copy();
            querySelect.setLimit(0);
            // ne surtout pas mettre 0, sinon toutes les valeurs possibles sont
            // retournee pas seulement celle qui satisfont la contrainte
            querySelect.setFacetMinCount(1);
            querySelect.setFacetLimit(Integer.MAX_VALUE);
            // on force le sort pour toujours utiliser le meme
            querySelect.setFacetSort(FacetSortType.name);
            // on supprime toutes les facettes, et on ajoute la notre
            querySelect.setFacetQuery();
            querySelect.setFacetField(select.getElement());

            // execution des requetes
            WikittyQueryResult<String> resultFacet =
                    searchEngine.findAllByQuery(transaction, queryFacet);
            WikittyQueryResult<String> resultSelect =
                    searchEngine.findAllByQuery(transaction, querySelect);

            // creation des resultats via la facette select
            List<FacetTopic> topics = resultSelect.getFacets().get(select.getElement().getValue());

            List<String> selectList = new ArrayList<String>(topics.size());
            if (query.getFirst() < topics.size()) {
                // il faut que le premier demande soit inferieur a la taille,
                // sinon on ne fait rien
                for (FacetTopic topic : topics) {
                    selectList.add(topic.getTopicName());
                }
            }

            boolean sortDesc = query.getSortDescending().contains(select.getElement());
            // tri selon l'ordre demande
            if (sortDesc) {
                // par defaut la facette est deja trie par ordre alphabetique
                // donc il n'y a qu'a l'inverser pour avoir l'ordre inverse
                Collections.reverse(selectList);
            }

            // on ne garde que ce qui est demande
            if (query.getFirst() < topics.size()) {
                int first = query.getFirst();
                int last = Math.min(topics.size(), query.getFirst() + query.getLimit());
                selectList = selectList.subList(first, last);
            }

            // gestion des agregats
            if (select.getAggregate() != null) {
                switch(select.getAggregate()) {
                    case AVG: {
                        // convert all to number
                        BigDecimal result = new BigDecimal(0);
                        for (String s : selectList) {
                            BigDecimal v = WikittyUtil.toBigDecimal(s);
                            result = result.add(v);
                        }
                        result = result.divide(new BigDecimal(selectList.size()));
                        selectList = new ArrayList<String>();
                        selectList.add(WikittyUtil.toString(result));
                    }
                        break;
                    case COUNT: {
                        // convert all to number
                        BigDecimal result = new BigDecimal(selectList.size());
                        selectList = new ArrayList<String>();
                        selectList.add(WikittyUtil.toString(result));
                    }
                        break;
                    case MAX: {
                        if (!selectList.isEmpty()) {
                            String result;
                            if (sortDesc) {
                                result = selectList.get(0);
                            } else {
                                result = selectList.get(selectList.size()-1);
                            }
                            selectList = new ArrayList<String>();
                            selectList.add(WikittyUtil.toString(result));
                        }
                    }
                        break;
                    case MIN: {
                        if (!selectList.isEmpty()) {
                            String result;
                            if (sortDesc) {
                                result = selectList.get(selectList.size()-1);
                            } else {
                                result = selectList.get(0);
                            }
                            selectList = new ArrayList<String>();
                            selectList.add(WikittyUtil.toString(result));
                        }
                    }
                        break;
                    case SUM: {
                        // convert all to number
                        BigDecimal result = new BigDecimal(0);
                        for (String s : selectList) {
                            BigDecimal v = WikittyUtil.toBigDecimal(s);
                            result = result.add(v);
                        }
                        selectList = new ArrayList<String>();
                        selectList.add(WikittyUtil.toString(result));
                    }
                        break;
                }
            }

            // fusion des resultats
            WikittyQueryResult<String> result = new WikittyQueryResult<String>(
                    query.getName(),
                    query.getFirst(),
                    topics.size(),
                    resultSelect.getQueryString(),
                    resultFacet.getFacets(),
                    selectList);

            return result;
        }
    }
}
