/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: StorageTest.java 700 2011-01-31 15:25:48Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.0.5/wikitty-api/src/test/java/org/nuiton/wikitty/conform/StorageTest.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.conform;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.junit.Assert;
import org.junit.Test;
import org.nuiton.wikitty.entities.WikittyLabelImpl;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.entities.WikittyLabel;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.entities.WikittyTreeNode;
import org.nuiton.wikitty.entities.WikittyTreeNodeImpl;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.entities.WikittyImpl;
import org.nuiton.wikitty.addons.WikittyLabelUtil;
import org.nuiton.wikitty.WikittyObsoleteException;
import org.nuiton.wikitty.services.WikittyEvent;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.entities.WikittyTreeNodeHelper;
import org.nuiton.wikitty.search.operators.Element;
import org.nuiton.wikitty.search.Search;
import org.nuiton.wikitty.search.TreeNodeResult;

public abstract class StorageTest extends AbstractTestConformance {

    private final static Log log  = LogFactory.getLog(StorageTest.class);

    @Test
    public void testStoreRestoreBasics() throws Exception {
        Wikitty w = createBasicWikitty();
        String id = w.getId();
        w = getProxy().store(w);
        w = getProxy().restore(id);
        assertTrue(w.hasExtension(EXTNAME));
        for ( int i = 0; i < 3; i++ ) {
            assertTrue( w.hasField(EXTNAME, "fieldName" + i));
        }
    }

    @Test
    public void testExtensionMethod() throws Exception {
        {
            getProxy().storeExtension(EXT_TEST);
            List<String> extIds = getProxy().getAllExtensionIds();
            log.debug("extIds: " + extIds);
            assertEquals(1, extIds.size());
            WikittyExtension ext = getProxy().restoreExtension(extIds.get(0));
            assertEquals(EXT_TEST.getName(), ext.getName());
            assertEquals(EXT_TEST.getVersion(), ext.getVersion());
            assertEquals(EXT_TEST, ext);
        }
        {
            // test de la suppression de l'extension
            getProxy().deleteExtension(EXT_TEST.getName());
            List<String> extIds = getProxy().getAllExtensionIds();
            assertEquals(0, extIds.size());
        }
        {
            // test que la suppression echoue si l'extension est utilisee
            getProxy().storeExtension(EXT_TEST);

            Wikitty w = new WikittyImpl();
            w.addExtension(EXT_TEST);
            w = getProxy().store(w);
            
            try {
                getProxy().deleteExtension(EXT_TEST.getName());
                assertTrue(false); // il doit y avoir une exception, donc on ne passe pas la
            } catch (WikittyException eee) {
                assertTrue(true);
            }
            List<String> extIds = getProxy().getAllExtensionIds();
            assertEquals(1, extIds.size());
        }
    }

    @Test
    public void testExtensionRequires() throws Exception {
        getProxy().storeExtension(Arrays.asList(EXT_REQUIRED, EXT_REQUIRES));
        List<String> extIds = getProxy().getAllExtensionsRequires(EXTREQUIRED);
        log.debug("extIds: " + extIds);
        assertEquals(1, extIds.size());
        
        WikittyExtension ext = getProxy().restoreExtension(extIds.get(0));
        assertEquals(EXT_REQUIRES.getName(), ext.getName());
        assertEquals(EXT_REQUIRES.getVersion(), ext.getVersion());
        assertEquals(EXT_REQUIRES, ext);
    }
    
    @Test
    public void testStoreRestoreLists() throws Exception {
        List<Wikitty> wikitties = new ArrayList<Wikitty>();
        List<String> ids = new ArrayList<String>();
        String[] wValues = new String[] {
                // wikitty[ 0 ]
                "fieldName0 = hello," +
                "fieldName1 = 123456," +
                "fieldName2 = " + format("23/01/1982"),
                
                // wikitty[ 1 ]
                "fieldName0 = Bonzai !," +
                "fieldName1 = 111111," +
                "fieldName2 = " + format("26/09/2009"),
                
                // wikitty[ 2 ]
                "fieldName0 = ho ho ho !," +
                "fieldName1 = 987654," +
                "fieldName2 = " + format("25/12/2029")
        };
        for ( String wValue : wValues ) {
            Wikitty w = createWikitty( wValue, EXTNAME, EXT_TEST );
            wikitties.add( w );
            ids.add( w.getId() );
        }
        wikitties = getProxy().storeWikitty(wikitties);
        
        wikitties = getProxy().restore(ids);
        int wIndex = 0;
        for ( String wValue : wValues ) {
            Wikitty w = wikitties.get(wIndex);
            int i = 0;
            for( Entry<String, String> keyValuePair : getKeyPairs(wValue) ) {
                String fieldName = "fieldName" + i;
                assertEquals(fieldName, keyValuePair.getKey());
                String errorMsg = "Error check field '" +  fieldName + "' on wikitty[" + i + "]";
                switch( i ) {
                case 0: // check fieldName0
                    assertEquals( errorMsg,
                            keyValuePair.getValue(), w.getFieldAsString(EXTNAME, fieldName) );
                    break;
                case 1: // check fieldName1
                    assertEquals( errorMsg,
                            Integer.parseInt(keyValuePair.getValue()),
                            w.getFieldAsInt(EXTNAME, fieldName) );
                    break;
                case 2: // check fieldName2
                    assertEquals( errorMsg,
                            WikittyUtil.parseDate(keyValuePair.getValue()),
                            w.getFieldAsDate(EXTNAME, fieldName) );
                    break;
                default:
                    fail( "this structure should only get 3 elements, unexpected : " + keyValuePair.getKey() );
                }
                i++;
            }
            wIndex++;
        }
    }

    @Test
    public void testStoreRestoreNullEmpty() throws Exception {
        Wikitty w = null;
        // should be ignored (?)
        try {
            w = getProxy().store(w);
            fail("store(null) Must throw an exception !");
        } catch (Exception e) {
            // OK !
        }
        
        w = new WikittyImpl();
        String id = w.getId();
        w = getProxy().store(w);
        w = getProxy().restore(id);
        assertEquals( id, w.getId() );
    }

    @Test
    public void testFieldConstraint() throws Exception {
        WikittyExtension ext = new WikittyExtension("TestConstraint", "1", null,
            WikittyUtil.buildFieldMapExtension(
            "String id notNull=true",
            "String ext[0-n] unique=true",
            "String other[0-n] unique=true notNull=true"
        ));

        getProxy().storeExtension(ext);
        Wikitty w = new WikittyImpl();
        w.addExtension(ext);

        try {
            w = getProxy().store(w);
            Assert.fail("not null contraint don't work on String");
        } catch (WikittyException eee) {
            eee.printStackTrace();
            // ok id must not be null
        }

        w.setField("TestConstraint", "id", "toto");
        try {
            w = getProxy().store(w);
            Assert.fail("not null contraint don't work in Collection");
        } catch (WikittyException eee) {
            eee.printStackTrace();
            // ok id must not be null
        }

        w.addToField("TestConstraint", "other", "titi");
        w = getProxy().store(w);

        w.addToField("TestConstraint", "ext", "tata");
        w.addToField("TestConstraint", "ext", "titi");
        w = getProxy().store(w);

        Wikitty w2 = getProxy().restore(w.getId());
        w2.addToField("TestConstraint", "ext", "tata"); // tata ne doit pas s'ajouter
        w2.addToField("TestConstraint", "ext", "toto");

        w2 = getProxy().store(w2);
        Wikitty w3 = getProxy().restore(w.getId());

        HashSet set = new HashSet(Arrays.asList("tata", "titi", "toto"));
        assertEquals(set, w3.getFieldAsSet("TestConstraint", "ext", String.class));
    }

    @Test
    public void testStoreFindAll() throws Exception {

                // create some wikitty to permit search test
        List<Wikitty> wikitties = new ArrayList<Wikitty>();
        String[] wValues = new String[] {
                // wikitty[ 0 ]
                "fieldName0 = table," +
                "fieldName1 = 003309," +
                "fieldName2 = " + format("23/01/1982"),

                // wikitty[ 1 ]
                "fieldName0 = chaise," +
                "fieldName1 = 003309," +
                "fieldName2 = " + format("26/09/2009"),

                // wikitty[ 2 ]
                "fieldName0 = bureau," +
                "fieldName1 = 223322," +
                "fieldName2 = " + format("25/12/2029")
        };
        for ( String wValue : wValues ) {
            Wikitty w = createWikitty( wValue, EXTNAME, EXT_TEST );
            wikitties.add( w );
        }
        wikitties = getProxy().storeWikitty(wikitties);

        // search test
        Wikitty w = new WikittyImpl();
        w.addExtension(EXT_TEST);
        w.setField(EXT_TEST.getName(), "fieldName1", "003309");
        Criteria criteria = Search.query(w).criteria()
                .addSortDescending(EXT_TEST.getName() + ".fieldName0");
        PagedResult<String> resultFind = getProxy().findAllIdByCriteria(criteria);

        assertEquals(2, resultFind.size());
        List<Wikitty> wikittyFind = getProxy().restore(resultFind.getAll());
        assertEquals(wikitties.subList(0, 2), wikittyFind);
    }

    /**
     * Test ignored because WikittyServiceInMemory don't support facets.
     * 
     * @throws Exception
     */
    @Test
    public void testStoreFindAllFacet() throws Exception {

        // create some wikitty to permit search test
        List<Wikitty> wikitties = new ArrayList<Wikitty>();
        String[] wValues = new String[] {
                // wikitty[ 0 ]
                "fieldName0 = table," +
                "fieldName1 = 663300," +
                "fieldName2 = " + format("23/01/1982"),

                // wikitty[ 1 ]
                "fieldName0 = chaise," +
                "fieldName1 = 663300," +
                "fieldName2 = " + format("26/09/2009"),

                // wikitty[ 2 ]
                "fieldName0 = bureau," +
                "fieldName1 = 223322," +
                "fieldName2 = " + format("25/12/2029")
        };
        for ( String wValue : wValues ) {
            Wikitty w = createWikitty( wValue, EXTNAME, EXT_TEST );
            wikitties.add( w );
        }
        wikitties = getProxy().storeWikitty(wikitties);

        // search test
        Wikitty w = new WikittyImpl();
        w.addExtension(EXT_TEST);
        w.setField(EXT_TEST.getName(), "fieldName1", "663300");
        Criteria criteria = Search.query(w).criteria()
                .setFirstIndex(0).setEndIndex(Criteria.ALL_ELEMENTS)
                .addFacetField(EXT_TEST.getName()+".fieldName0")
                .addFacetField(EXT_TEST.getName()+".fieldName1");
        PagedResult<String> resultFind = getProxy().findAllIdByCriteria(criteria);

        List<String> facetNames = new ArrayList<String>(resultFind.getFacetNames());
        Collections.sort(facetNames);
        assertEquals(Arrays.asList(EXT_TEST.getName()+".fieldName0",
                EXT_TEST.getName()+".fieldName1"), facetNames);

        // with must have 2 topic: table and chaise
        assertEquals(2, resultFind.getTopic(EXT_TEST.getName()+".fieldName0").size());
        // with must have 1 topic: 663300
        assertEquals(1, resultFind.getTopic(EXT_TEST.getName()+".fieldName1").size());


        // essai de facettiser sur les extensions
        criteria = Search.query().keyword("*").criteria()
                .setFirstIndex(0).setEndIndex(0)
                .addFacetField(Element.ELT_EXTENSION);
        resultFind = getProxy().findAllIdByCriteria(criteria);
        assertEquals(1, resultFind.getFacetNames().size());
        assertNotNull(resultFind.getFacets().get(Element.ELT_EXTENSION));
    }

    @Test
    public void testStoreFindDelete() throws Exception {

        // create some wikitty to permit search test
        List<Wikitty> wikitties = new ArrayList<Wikitty>();
        String[] wValues = new String[] {
                // wikitty[ 0 ]
                "fieldName0 = table," +
                "fieldName1 = 003301," +
                "fieldName2 = " + format("23/01/1982"),

                // wikitty[ 1 ]
                "fieldName0 = chaiseICI," +
                "fieldName1 = 113312," +
                "fieldName2 = " + format("26/09/2009"),

                // wikitty[ 2 ]
                "fieldName0 = bureau," +
                "fieldName1 = 223323," +
                "fieldName2 = " + format("25/12/2029")
        };
        for ( String wValue : wValues ) {
            Wikitty w = createWikitty( wValue, EXTNAME, EXT_TEST );
            wikitties.add( w );
        }
        Map<String, String> oldIdVersion = new HashMap<String, String>();
        for( Wikitty w : wikitties ) {
            oldIdVersion.put(w.getId(), w.getVersion());
        }
        WikittyEvent response = getWikittyService().store(null, wikitties, false);
        for (Wikitty w : wikitties) {
            response.update(w);
            String v1 = w.getVersion();
            String v2 = oldIdVersion.get( w.getId() );
            assertNotNull( v1 );
            assertNotNull( v2 );
            assertTrue(WikittyUtil.versionGreaterThan(v1, v2));
        }

        // search test
        Wikitty w = new WikittyImpl();
        w.addExtension(EXT_TEST);
        w.setField(EXT_TEST.getName(), "fieldName0", "chaiseICI");
        Criteria criteria = Search.query(w).criteria();
        Wikitty resultFind = getProxy().findByCriteria(criteria);

        Wikitty wikittySource = wikitties.get(1);
        assertEquals(w.getFieldAsString(EXT_TEST.getName(), "fieldName0"),
                resultFind.getFieldAsString(EXT_TEST.getName(), "fieldName0"));

        assertEquals(wikittySource.getId(), resultFind.getId());
        // test equals implantation method
        assertEquals(wikittySource, resultFind);

        assertEquals(wikittySource.getFieldAsString(EXT_TEST.getName(), "fieldName0"),
                resultFind.getFieldAsString(EXT_TEST.getName(), "fieldName0"));
        assertEquals(wikittySource.getFieldAsString(EXT_TEST.getName(), "fieldName1"),
                resultFind.getFieldAsString(EXT_TEST.getName(), "fieldName1"));
        assertEquals(wikittySource.getFieldAsString(EXT_TEST.getName(), "fieldName2"),
                resultFind.getFieldAsString(EXT_TEST.getName(), "fieldName2"));


        // test to find deleted wikitty
        // test if solr index is coherent with database
        Criteria criteria1 = Search.query().eq(Element.ELT_ID, wikittySource.getId()).criteria();
        Wikitty searchedWikitty1 = getProxy().findByCriteria(criteria1);
        Assert.assertNotNull(searchedWikitty1);

        getProxy().delete(wikittySource.getId());
        Wikitty deletedObject = getProxy().restore(wikittySource.getId());
        assertNull(deletedObject);

        // test to find deleted wikitty
        // test if solr index is coherent with database   
        Criteria criteria2 = Search.query().eq(Element.ELT_ID, wikittySource.getId()).criteria();
        Wikitty searchedWikitty2 = getProxy().findByCriteria(criteria2);
        Assert.assertNull(searchedWikitty2);
    }

    /**
     * Test ignored because WikittyServiceInMemory don't support labels.
     * 
     * @throws Exception
     */
    @Test
    public void testStoreAndLabel() throws Exception {

        // create some wikitty to permit search test
        List<Wikitty> wikitties = createSampleWikitty(getWikittyService());
        long ts = new Date().getTime();

        // labelisation test
        Wikitty w1 = wikitties.get(0);
        WikittyLabelUtil.addLabel(getProxy(), w1.getId(), "titi"+ts);
        WikittyLabelUtil.addLabel(getProxy(), w1.getId(), "toto"+ts);

        Wikitty w2 = wikitties.get(1);
        WikittyLabelUtil.addLabel(getProxy(), w2.getId(), "tata"+ts);
        WikittyLabelUtil.addLabel(getProxy(), w2.getId(), "titi"+ts);

        Wikitty w3 = wikitties.get(2);
        WikittyLabelUtil.addLabel(getProxy(), w3.getId(), "tutu"+ts);
        WikittyLabelUtil.addLabel(getProxy(), w3.getId(), "titi"+ts);

        Wikitty wt = WikittyLabelUtil.findByLabel(getProxy(), "toto"+ts );

        assertEquals(w1, wt);

        Set<String> labels = WikittyLabelUtil.findAllAppliedLabels(getProxy(), w2.getId());

        assertEquals(new HashSet(Arrays.asList("tata"+ts, "titi"+ts)), labels);
    }

    @Test
    public void testStoreAndClassification() throws Exception {
        // create some wikitty to permit search test
        List<Wikitty> wikitties = createSampleWikitty(getWikittyService());
        List<Wikitty> wikittyNodes = new ArrayList<Wikitty>();

        WikittyTreeNodeImpl root = new WikittyTreeNodeImpl();
        wikittyNodes.add(root.getWikitty());
        root.setName("MyCategoryRoot");
        for ( int i = 0; i < 3; i++ ) {
            WikittyTreeNodeImpl leaf = new WikittyTreeNodeImpl();
            wikittyNodes.add( leaf.getWikitty() );
            leaf.setName( "cat-"+i );
            leaf.setParent( root.getWikittyId() );
            // root.addChild( leaf.getWikittyId() );

            for ( int j = 0; j < 5; j++ ) {
                WikittyTreeNodeImpl subLeaf = new WikittyTreeNodeImpl();
                subLeaf.setName( "subcat-"+i+"-"+j );
                subLeaf.setParent( leaf.getWikittyId() );
                wikittyNodes.add( subLeaf.getWikitty() );
                // leaf.addChild( subLeaf.getWikittyId() );
            }
        }
        wikittyNodes = getProxy().storeWikitty(wikittyNodes); // store treeNodes.

        Wikitty table = wikitties.get(0);
        Wikitty chaise = wikitties.get(1);
        Wikitty bureau = wikitties.get(2);

        assign( chaise, root, "cat-1/subcat-1-4" );
        assign( table, root, "cat-1" );
        assign( bureau, root, "cat-2/subcat-2-4" );

        TreeNodeResult<WikittyTreeNode> t = getProxy().findTreeNode(
                WikittyTreeNode.class,
                root.getWikitty().getId(), 0, false, null);

        Assert.assertEquals("MyCategoryRoot", t.getObject().getName());


    }
    
    protected void assign(Wikitty wikitty, WikittyTreeNodeImpl root, String path) {
        String[] nodeNames = path.split("/");
        WikittyTreeNodeImpl currentNode = root;
        outerloop : for( String nodeName : nodeNames ) {
            WikittyTreeNodeImpl node = new WikittyTreeNodeImpl();
            node.setName(nodeName);

            Criteria criteria = Search.query(node.getWikitty()).criteria()
                    .setFirstIndex(0).setEndIndex(Criteria.ALL_ELEMENTS);
            List<String> wikittiesId = getProxy().findAllIdByCriteria(criteria).getAll();
            List<Wikitty> wikitties = getProxy().restore(wikittiesId);
            for ( Wikitty child : wikitties ) {
                if (!child.hasExtension(WikittyTreeNode.EXT_WIKITTYTREENODE)) {
                    continue;
                }
                node = new WikittyTreeNodeImpl(child);
                if ( node.getName().equals(nodeName) ) {
                    currentNode = node;
                    continue outerloop;
                }
            }
            fail( "Unable to find node " + nodeName + " inside " + currentNode.getName() );
        }
        currentNode.addAttachment(wikitty.getId());
        getProxy().store(currentNode.getWikitty());
    }

    /**
     * Test la creation d'une entité business par introspection avec un wikitty.
     */
    @Test
    public void testWikittyUtilNewInstanceWithWikitty() {

        Wikitty w = new WikittyImpl();
        w.addExtension(WikittyTreeNodeImpl.extensions);

        WikittyTreeNode treeNode = WikittyUtil.newInstance(
                null, getWikittyService(), WikittyTreeNode.class, w);
        Assert.assertNotNull(treeNode);
    }
    
    /**
     * Test de la methode cast du proxy.
     */
    @Test
    public void testCast() {
        
        WikittyTreeNode node = new WikittyTreeNodeImpl();
        node.setName("nodeName");
        
        // cast, different business object, same wikitty
        WikittyLabel label = getProxy().cast(node, WikittyLabel.class);
        label.addLabels("toto");
        
        label = getProxy().store(label);
        String wikittyId = label.getWikittyId();
        
        WikittyTreeNode node2 = getProxy().restore(WikittyTreeNode.class, wikittyId);
        Assert.assertEquals("nodeName", node2.getName());
        
        WikittyLabel label2 = getProxy().cast(node, WikittyLabel.class);
        List<String> labels = new ArrayList<String>(label2.getLabels());
        Assert.assertFalse(labels.isEmpty());
        Assert.assertEquals("toto", labels.get(0));
    }
    
    /**
     * Test les differents cas du forage de version lors du store.
     */
    @Test
    public void testStorageVersionForce() {

        // store 1
        Wikitty myWikitty = new WikittyImpl();
        myWikitty.addExtension(WikittyTreeNodeImpl.extensions);
        WikittyTreeNodeHelper.setName(myWikitty, "name");
        myWikitty = getProxy().store(myWikitty);
        Assert.assertEquals("1.0", myWikitty.getVersion());
        
        // store 2 : no modification
        myWikitty = getProxy().store(myWikitty);
        Assert.assertEquals("1.0", myWikitty.getVersion());
        
        // store 3 : modification
        WikittyTreeNodeHelper.setName(myWikitty, "new name");
        myWikitty = getProxy().store(myWikitty);
        Assert.assertEquals("2.0", myWikitty.getVersion());
        
        // store 4 : new wikitty with same wikitty id (obsolete)
        Wikitty myNewWikitty = new WikittyImpl(myWikitty.getId());
        myNewWikitty.addExtension(WikittyTreeNodeImpl.extensions);
        WikittyTreeNodeHelper.setName(myNewWikitty, "new wikitty");
        try {
            myWikitty = getProxy().store(myNewWikitty);
            Assert.fail("Test must throw WikittyObsoleteException");
        }
        catch (WikittyObsoleteException ex) {
            if (log.isTraceEnabled()) {
                log.trace("Wikitty obsolete", ex);
            }
        }
        
        // store 4 : same but with force (increased by force)
        WikittyEvent event = getWikittyService().store(
                null, Collections.singletonList(myWikitty), true);
        event.update(myWikitty);
        Assert.assertEquals("3.0", myWikitty.getVersion());
        
        // store 5 : fix version (not increased by force)
        myNewWikitty.setVersion("11.0");
        event = getWikittyService().store(
                null, Collections.singletonList(myNewWikitty), true);
        event.update(myNewWikitty);
        Assert.assertEquals("11.0", myNewWikitty.getVersion());
    }
    
    /**
     * Test qu'une sauvegarde de wikitty précédemment supprimé fonctionne.
     */
    @Test
    public void testStoragePreviouslyDeleted() {

        // store 1
        Wikitty myWikitty = new WikittyImpl();
        myWikitty.addExtension(WikittyTreeNodeImpl.extensions);
        WikittyTreeNodeHelper.setName(myWikitty, "name");

        myWikitty = getProxy().store(myWikitty);

        // delete
        getProxy().delete(myWikitty.getId());
        Wikitty restoredWikitty = getProxy().restore(myWikitty.getId());
        Assert.assertNull(restoredWikitty);

        // store again
        myWikitty = getProxy().store(myWikitty);
        restoredWikitty = getProxy().restore(myWikitty.getId());
        Assert.assertNotNull(restoredWikitty);
    }

    @Test
    public void testBinaryFieldStorage() {
        String extName = "BinaryExt";
        byte[] bytes = "Coucou le monde".getBytes();

        WikittyExtension BinaryExt = new WikittyExtension(extName,
                                     "1.0", // version
                                     null,
                                     WikittyUtil.buildFieldMapExtension( // building field map
                                             "String name unique=\"true\"",
                                             "Binary content"));
        Wikitty w = new WikittyImpl();
        w.addExtension(BinaryExt);
        w.setField(extName, "name", "LeBin");
        w.setField(extName, "content", bytes);

        w = getProxy().store(w);

        Wikitty restoredWikitty = getProxy().restore(w.getId());
        Assert.assertNotNull(restoredWikitty);
        Assert.assertEquals("LeBin", restoredWikitty.getFieldAsString(extName, "name"));
        Assert.assertEquals(bytes, restoredWikitty.getFieldAsBytes(extName, "content"));
    }

    @Test
    public void testStoreUnmodifiedEntity() {
        WikittyLabel wikitty1 = new WikittyLabelImpl();
        WikittyLabel wikitty2 = new WikittyLabelImpl();

        List<WikittyLabel> toStore = new ArrayList<WikittyLabel>();
        Collections.addAll(toStore, wikitty1, wikitty2);
        List<WikittyLabel> stored = getProxy().store(toStore);

        stored.get(0).addLabels("lbl");

        try {
            getProxy().store(toStore);
        } catch (Exception eee) {
            Assert.fail();
        }
    }
}
