package org.nuiton.topia.templates;

/*
 * #%L
 * ToPIA :: Templates
 * $Id: BinderHelperTransformer.java 3082 2014-04-26 17:39:39Z tchemit $
 * $HeadURL: https://svn.nuiton.org/topia/tags/topia-3.0-beta-7/topia-templates/src/main/java/org/nuiton/topia/templates/BinderHelperTransformer.java $
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.models.object.*;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.util.TopiaEntityBinder;
import org.nuiton.topia.persistence.util.TopiaEntityHelper;
import org.nuiton.util.beans.BinderModelBuilder;
import org.nuiton.util.beans.BinderFactory;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;





/**
 * A template to generate a helper for {@link TopiaEntityBinder}.
 * 
 * @author tchemit <chemit@codelutin.com>
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.topia.templates.BinderHelperTransformer"
 * @since 2.3.1
 */
public class BinderHelperTransformer extends ObjectModelTransformerToJava {

    private static final Log log =
            LogFactory.getLog(BinderHelperTransformer.class);


    @Override
    public void transformFromModel(ObjectModel model) {
        ObjectModelClass resultClass;

        List<ObjectModelClass> classes = TopiaGeneratorUtil.getEntityClasses(model, true);

        if (CollectionUtils.isEmpty(classes)) {

            // no entity classes, so no generation
            log.warn("No entity to generate, " + getClass().getName() + " is skipped");
            return;
        }
        
        String packageName = getDefaultPackageName();
        String modelName = model.getName();
        String binderHelperClazzName = modelName + "BinderHelper";
        String daoHelperClazzName = modelName + "DAOHelper";

        String entityEnumName = TopiaGeneratorUtil.getEntityEnumName(model);
        String entityEnumPackage = packageName + "." + entityEnumName;

        resultClass = createClass(binderHelperClazzName, packageName);

        setSuperClass(resultClass, BinderFactory.class);

        addImport(resultClass, TopiaEntityBinder.class);
        addImport(resultClass, TopiaEntityHelper.class);
        addImport(resultClass, TopiaEntity.class);
        addImport(resultClass, BinderModelBuilder.class);
        addImport(resultClass, entityEnumPackage);

        ObjectModelOperation op;

        op = addOperation(resultClass,
                          "getTopiaBinder",
                          "<E extends TopiaEntity> TopiaEntityBinder<E>",
                          ObjectModelJavaModifier.PUBLIC,
                          ObjectModelJavaModifier.STATIC);
        addParameter(op, "Class<E>", "entityClass");
        addParameter(op, "String", "contextName");
        setOperationBody(op, ""
+"\n"
+"        return (TopiaEntityBinder<E>) newBinder(entityClass, entityClass, contextName, TopiaEntityBinder.class);\n"
+"    "
        );

        op = addOperation(resultClass,
                          "getSimpleTopiaBinder",
                          "<E extends TopiaEntity> TopiaEntityBinder<E>",
                          ObjectModelJavaModifier.PUBLIC,
                          ObjectModelJavaModifier.STATIC);
        addParameter(op, "Class<E>", "entityClass");
        setOperationBody(op, ""
+"\n"
+"        return getTopiaBinder(entityClass, \""+modelName+"\");\n"
+"    "
        );

        op = addOperation(resultClass,
                          "registerTopiaBinder",
                          "void",
                          ObjectModelJavaModifier.PUBLIC,
                          ObjectModelJavaModifier.STATIC);
        addParameter(op, "BinderModelBuilder", "builder");
        addParameter(op, "String", "contextName");
        setOperationBody(op, ""
+"\n"
+"       registerBinderModel(builder, contextName);\n"
+"    "
        );

        op = addOperation(resultClass,
                          "registerTopiaBinder",
                          "<E extends TopiaEntity> TopiaEntityBinder<E>",
                          ObjectModelJavaModifier.PUBLIC,
                          ObjectModelJavaModifier.STATIC);
        addParameter(op, "Class<E>", "entityClass");
        addParameter(op, "BinderModelBuilder", "builder");
        addParameter(op, "String", "contextName");
        setOperationBody(op, ""
+"\n"
+"       registerBinderModel(builder, contextName);\n"
+"       return getTopiaBinder(entityClass, contextName);\n"
+"    "
        );

        op = addOperation(resultClass,
                          "copy",
                          "<E extends TopiaEntity> void",
                          ObjectModelJavaModifier.PUBLIC,
                          ObjectModelJavaModifier.STATIC);
        addParameter(op, "String", "contextName");
        addParameter(op, "E", "source");
        addParameter(op, "E", "target");
        addParameter(op, "boolean", "tech");
        setOperationBody(op, ""
+"\n"
+"        Class<E> entityClass = (Class<E>) TopiaEntityHelper.getContractClass("+entityEnumName+".values(), target.getClass());\n"
+"        TopiaEntityBinder<E> binder = getTopiaBinder(entityClass, contextName);\n"
+"        if (binder == null) {\n"
+"            throw new NullPointerException(\"Could not find a simple topia binder of type : \" + target.getClass());\n"
+"        }\n"
+"        binder.load(source, target, tech);\n"
+"    "
        );

        op = addOperation(resultClass,
                          "simpleCopy",
                          "<E extends TopiaEntity> void",
                          ObjectModelJavaModifier.PUBLIC,
                          ObjectModelJavaModifier.STATIC);
        addParameter(op, "E", "source");
        addParameter(op, "E", "target");
        addParameter(op, "boolean", "tech");
        setOperationBody(op, ""
+"\n"
+"        Class<E> entityClass = (Class<E>) TopiaEntityHelper.getContractClass("+entityEnumName+".values(), target.getClass());\n"
+"        TopiaEntityBinder<E> binder = getSimpleTopiaBinder(entityClass);\n"
+"        if (binder == null) {\n"
+"            throw new NullPointerException(\"Could not find a simple topia binder of type : \" + target.getClass());\n"
+"        }\n"
+"        binder.load(source, target, tech);\n"
+"    "
        );

        StringBuilder initCode = new StringBuilder();

        for (ObjectModelClass clazz : classes) {

            String prefix = getConstantPrefix(clazz, "");

            if (StringUtils.isEmpty(prefix)) {

                // no specific prefix, so no prefix
                if (log.isWarnEnabled()) {
                    log.warn("[" + clazz.getName() + "] Will generate constants with NO prefix, not a good idea...");
                }
            }

            setConstantPrefix(prefix);

            generateBinder(modelName, clazz, resultClass, initCode);

        }

        op = addOperation(resultClass, "initBinders", "void", ObjectModelJavaModifier.PROTECTED, ObjectModelJavaModifier.STATIC);
        setOperationBody(op, initCode.toString());

        op = addOperation(resultClass, null, (String) null, ObjectModelJavaModifier.STATIC);
        setOperationBody(op, ""
 +"\n"
+"    initBinders();\n"
+""
        );
    }

    protected void generateBinder(String modelName,
                                  ObjectModelClass clazz,
                                  ObjectModelClass resultClass,
                                  StringBuilder initCode) {

        List<ObjectModelAttribute> list = new ArrayList<ObjectModelAttribute>();
        for (ObjectModelAttribute attr : clazz.getAttributes()) {
            if (!attr.isNavigable()) {
                continue;
            }

            if (GeneratorUtil.isNMultiplicity(attr)) {
                // not dealing with association
                continue;
            }

            list.add(attr);
        }

        String clazzName = clazz.getName();

        if (list.isEmpty()) {
            // no attribute, do nothing
            if (log.isDebugEnabled()) {
                log.debug("no attribute to add in a binder for " + clazzName +
                          ", will not generate it.");
            }
            return;
        }

        if (log.isDebugEnabled()) {
            log.debug("generate simple binder for " + clazzName);
        }
        addImport(resultClass, clazz);
        initCode.append(""
+"\n"
+"        BinderModelBuilder<"+clazzName+", "+clazzName+"> builder"+clazzName+" =\n"
+"            BinderModelBuilder.newEmptyBuilder("+clazzName+".class);\n"
+"        builder"+clazzName+".addSimpleProperties(\n"
+""
        );
        Iterator<ObjectModelAttribute> itr = list.iterator();
        while (itr.hasNext()) {
            ObjectModelAttribute attr = itr.next();
            String attrName;
            if (!attr.hasAssociationClass()) {
                attrName = attr.getName();
            } else {
                attrName = TopiaGeneratorUtil.getAssocAttrName(attr);
            }
            boolean hasNext = itr.hasNext();
            initCode.append(""
+"            "+clazzName+"."+getConstantName(attrName)+""+(hasNext?",\n":"")+""
            );
        }
        initCode.append(""
+"\n"
+"        );\n"
+"        registerTopiaBinder(builder"+clazzName+", \""+modelName+"\");\n"
+"    "
        );
    }
}

