/* *##% 
 * ToPIA :: Persistence
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*/

/* *
 * TopiaContext.java
 *
 * Created: 3 janv. 2006 21:18:34
 *
 * @author poussin
 * @version $Revision: 1718 $
 *
 * Last update: $Date: 2009-12-16 22:31:36 +0100 (mer., 16 déc. 2009) $
 * by : $Author: tchemit $
 */

package org.nuiton.topia;

import org.nuiton.topia.event.TopiaContextListener;
import org.nuiton.topia.event.TopiaEntitiesVetoable;
import org.nuiton.topia.event.TopiaEntityListener;
import org.nuiton.topia.event.TopiaEntityVetoable;
import org.nuiton.topia.event.TopiaTransactionListener;
import org.nuiton.topia.event.TopiaTransactionVetoable;
import org.nuiton.topia.framework.TopiaService;
import org.nuiton.topia.persistence.TopiaEntity;

import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.Reader;
import java.io.Writer;
import java.util.List;

/**
 * @author poussin
 */

public interface TopiaContext {

    /* Adders */
    public void addTopiaEntityListener(TopiaEntityListener listener);

    public void addTopiaEntityListener(
            Class<? extends TopiaEntity> entityClass,
            TopiaEntityListener listener);

    public void addTopiaEntityVetoable(TopiaEntityVetoable vetoable);

    public void addTopiaEntityVetoable(
            Class<? extends TopiaEntity> entityClass,
            TopiaEntityVetoable vetoable);

    public void addTopiaTransactionListener(TopiaTransactionListener listener);

    public void addTopiaTransactionVetoable(TopiaTransactionVetoable vetoable);

    public void addPropertyChangeListener(PropertyChangeListener listener);

    public void addTopiaContextListener(TopiaContextListener listener);

    public void addTopiaEntitiesVetoable(TopiaEntitiesVetoable vetoable);

    /* Removers */
    public void removeTopiaEntityListener(TopiaEntityListener listener);

    public void removeTopiaEntityListener(
            Class<? extends TopiaEntity> entityClass,
            TopiaEntityListener listener);

    public void removeTopiaEntityVetoable(TopiaEntityVetoable vetoable);

    public void removeTopiaEntityVetoable(
            Class<? extends TopiaEntity> entityClass,
            TopiaEntityVetoable vetoable);

    public void removeTopiaTransactionListener(TopiaTransactionListener listener);

    public void removeTopiaTransactionVetoable(TopiaTransactionVetoable vetoable);

    public void removePropertyChangeListener(PropertyChangeListener listener);

    public void removeTopiaContextListener(TopiaContextListener listener);

    public void removeTopiaEntitiesVetoable(TopiaEntitiesVetoable vetoable);

    /**
     * Return true if specific service is available
     *
     * @param <E> type of service
     * @param interfaceService fqn of the service
     * @return the service
     */
    public <E extends TopiaService> boolean serviceEnabled(
            Class<E> interfaceService);

    /**
     * Return the service
     *
     * @param <E> type of service
     * @param interfaceService fqn of the service
     * @return the service
     * @throws TopiaNotFoundException if service is can't be retrieved
     */
    public <E extends TopiaService> E getService(Class<E> interfaceService)
            throws TopiaNotFoundException;

    /**
     * Permet de créer le schema de la base de données
     *
     * @throws TopiaException if any exception
     */
    public void createSchema() throws TopiaException;

    /**
     * Permet d'afficher les requetes SQL de creation de base
     *
     * @throws TopiaException if any exception
     */
    public void showCreateSchema() throws TopiaException;

    /**
     * Permet de mettre à jour le schema de la base de données
     *
     * @throws TopiaException if any exception
     */
    public void updateSchema() throws TopiaException;

    public TopiaContext beginTransaction() throws TopiaException;

    /**
     * applique les modifications apporté a ce context sur la base de données.
     *
     * @throws TopiaException if any exception
     */
    public void commitTransaction() throws TopiaException;

    /**
     * annule les modifications apporté a ce context
     *
     * @throws TopiaException if any exception
     */
    public void rollbackTransaction() throws TopiaException;

    /**
     * Permet de rechercher un entite directement par son TopiaId
     *
     * @param topiaId l'id de l'entite recherche
     * @return l'entite trouvee (ou null si non trouve)
     * @throws TopiaException if any exception
     */
    public TopiaEntity findByTopiaId(String topiaId) throws TopiaException;

    /**
     * Permet de faire une requete HQL hibernate directement sur la base
     *
     * @param hql  la requete a faire
     * @param args les arguments de la requete
     * @return La liste des resultats
     * @throws TopiaException si une erreur survient durant la requete
     */
    public List find(String hql, Object... args) throws TopiaException;

    /**
     * Permet de faire une requete HQL hibernate directement sur la base
     * en precisant la fenetre des elements a remonter avec les parametres <code>startIndex</code>
     * et <code>endIndex</code>.
     *
     * @param hql        la requete a faire
     * @param startIndex la position du premier element a remonter
     * @param endIndex   la position du dernier element a remonter
     * @param args       les arguments de la requete
     * @return La liste des resultats
     * @throws TopiaException si une erreur survient durant la requete
     */
    public List find(String hql, int startIndex, int endIndex, Object... args)
            throws TopiaException;

    /**
     * Execute HQL operation on data (Update, Delete)
     *
     * @param hql  la requete a faire
     * @param args les arguments de la requete
     * @return The number of entities updated or deleted.
     * @throws TopiaException if any exception
     */
    public int execute(String hql, Object... args) throws TopiaException;

    /**
     * Permet d'ajouter dans le TopiaContext une TopiaEntity créé par un
     * autre context.
     *
     * @param e l'entity a ajouter
     * @throws TopiaException if any exception
     */
    public void add(TopiaEntity e) throws TopiaException;

    /**
     * Permet d'importer des données en XML
     *
     * @param xml le flux XML
     * @throws TopiaException si une erreur survient durant l'import
     */
    public void importXML(Reader xml) throws TopiaException;

    /**
     * Permet d'exporter certaines données en XML
     * <p/>
     * <b>Note:</b> Si le parametre <code>entityAndCondition</code> est vide, alors on duplique
     * toutes les entités de la base.
     *
     * @param xml                le flux XML dans lequel il faut ecrire
     * @param entityAndCondition paramètre qui vont par deux, qui represente
     *                           la classe de l'entity a exporter et la condition where que doit
     *                           respecter l'objet pour etre exporter
     *                           (entityClass, condition)
     * @throws TopiaException si une erreur survient durant l'export
     */
    public void exportXML(Writer xml, Object... entityAndCondition)
            throws TopiaException;

    /**
     * Permet de dupliquer de ce context vers un context d'une autre base des
     * données sans modification des entites.
     * <p/>
     * <b>Note:</b> Si le parametre <code>entityAndCondition</code> est vide, alors on duplique
     * toutes les entités de la base.
     * <p/>
     * <b>Note 2:</b> Il se peut que la replication simple ne soit pas suffisante (par example
     * si l'on veut repliquer q'une partie d'une entité), on utilisera donc la seconde méthode
     * {@link #replicateEntities(TopiaContext, List)}.
     *
     * @param dstCtxt            le context de la base destination
     * @param entityAndCondition paramètre qui vont par deux, qui represente
     *                           la classe de l'entity a exporter et la condition where que doit
     *                           respecter l'objet pour etre exporter
     *                           (entityClass, condition)
     * @throws TopiaException           si une erreur pendant la duplication
     * @throws IllegalArgumentException si l'un des context n'est pas ouvert, ou si on essaye de
     *                                  dupliquer dans la même base.
     */
    public void replicate(TopiaContext dstCtxt, Object... entityAndCondition)
            throws TopiaException, IllegalArgumentException;

    /**
     * Permet de dupliquer une entité du type donné vers un autre context.
     *
     * @param dstCtxt le context de la base destination
     * @param entity  l'entité à répliquer
     * @param <T>     le type des entités à répliquer
     * @throws TopiaException           si une erreur pendant la duplication
     * @throws IllegalArgumentException si l'un des context n'est pas ouvert, ou si on essaye de
     *                                  dupliquer dans la même base.
     */
    public <T extends TopiaEntity> void replicateEntity(TopiaContext dstCtxt, T entity)
            throws TopiaException, IllegalArgumentException;

    /**
     * Permet de dupliquer les entités du type donné vers un autre context.
     *
     * @param dstCtxt  le context de la base destination
     * @param entities les entités à répliquer
     * @param <T>      le type des entités à répliquer
     * @throws TopiaException           si une erreur pendant la duplication
     * @throws IllegalArgumentException si l'un des context n'est pas ouvert, ou si on essaye de
     *                                  dupliquer dans la même base.
     */
    public <T extends TopiaEntity> void replicateEntities(TopiaContext dstCtxt, List<T> entities)
            throws TopiaException, IllegalArgumentException;

    /**
     * Sauve la base de données dans un format natif a la base, la
     * representation n'est pas portable d'une base a l'autre. Cette methode
     * ne doit être utilisé que pour un stockage temporaire utile à une
     * application
     *
     * @param file     le nom du fichier ou stocker les informations
     * @param compress si vrai compress le fichier avec gzip
     * @throws TopiaException if any exception
     */
    public void backup(File file, boolean compress) throws TopiaException;

    /**
     * Supprime toutes les tables et autres elements de la database.
     *
     * @param dropDatabase si vrai alors supprime aussi la base de données
     *                     si la base utilise des fichiers les fichiers seront supprimé (ex: h2)
     *                     ou sera fait sur la base (pastgresql)
     * @throws TopiaException if any exception
     */
    public void clear(boolean dropDatabase) throws TopiaException;

    /**
     * l'inverse de la methode {@link #backup(File,boolean)}
     *
     * @param file le fichier ou prendre les informations, il peut-etre
     *             compressé avec gzip ou non.
     * @throws TopiaException if any exception
     */
    public void restore(File file) throws TopiaException;

    /**
     * Ferme le contexte
     *
     * @throws TopiaException if any exception
     */
    public void closeContext() throws TopiaException;

    /**
     * Indique si le contexte a ete ferme
     *
     * @return <code>true</code> si le context est ferme, <code>false</code> autrement
     */
    public boolean isClosed();

} //TopiaContext
