/*
 * *##% 
 * Maven License Plugin
 * Copyright (C) 2008 - 2010 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.license.plugin.repository;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

/**
 * @author chemit
 * @since 1.0.3
 */
public class LicenseRepositoryFactory {

    /**
     * class-path directory where is the licenses repository
     */
    public static final String JAR_LICENSE_REPOSITORY = "/META-INF/licenses";
    /**
     * to use log facility, just put in your code: log.info(\"...\");
     */
    static private final Log log = LogFactory.getLog(LicenseRepositoryFactory.class);

    /**
     * Obtain a new repository.
     *
     * @param useJarRepository flag to use at first a jar repository
     * @param load             flag to load the repository
     * @param extraResolvers   extra baseURLs to defined extra license resolver
     * @return the instanciate and ready to use license repository
     * @throws IOException if any problem while acquiring license repository
     */
    public static LicenseRepository newLicenseRepository(boolean useJarRepository, boolean load, String... extraResolvers) throws IOException {

        List<URL> baseURLs = new ArrayList<URL>();
        if (extraResolvers != null) {
            for (String exUrl : extraResolvers) {
                if (exUrl == null) {
                    // skpi null url
                    continue;
                }
                URL url;
                try {
                    url = new URL(exUrl);
                } catch (Exception e) {
                    if (log.isDebugEnabled()) {
                        log.warn("could not convert url [" + exUrl + "], for reason " + e.getMessage());
                    } else {
                        log.warn("could not convert url [" + exUrl + "], for reason " + e.getMessage());
                    }
                    log.warn("will skip the url [" + exUrl + "]");
                    continue;
                }
                baseURLs.add(url);
            }
        }
        if (!useJarRepository && baseURLs.isEmpty()) {
            log.warn("no repository to create!");
            // no repository to create
            return null;
        }

        LicenseRepository result = null;
        LicenseRepository current = null;
        if (useJarRepository) {
            current = result = newJarLicenseRepository(false);
        }

        for (URL baseURL : baseURLs) {
            if (baseURL != null && !baseURL.toString().trim().isEmpty()) {
                LicenseRepository r = new LicenseRepository();
                r.setBaseURL(baseURL);
                if (current == null) {
                    // no previous repository to chain
                    result = r;

                } else {
                    // chain to previous repository
                    current.appendRepository(r);
                }
                // new repository to chain
                current = r;
            }
        }
        if (load) {
            result.load();
        }
        return result;
    }

    public static LicenseRepository newJarLicenseRepository(boolean load) throws IOException {
        LicenseRepository result = new LicenseRepository();
        // the first repository is always a jar repository
        URL baseURL = LicenseRepositoryFactory.class.getResource(JAR_LICENSE_REPOSITORY);
        result.setBaseURL(baseURL);
        if (load) {
            result.load();
        }
        return result;
    }
}
