/*
 * *##% 
 * Maven License Plugin
 * Copyright (C) 2008 - 2010 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.license.plugin.repository;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;

/**
 *
 * @author chemit
 * @since 1.0.3
 */
public class LicenseRepository {

    public static final String REPOSITORY_DEFINITION_FILE = "licenses.properties";
    /**
     * the base url of the licenses repository
     */
    protected URL baseURL;
    /**
     * next repository (can be {@code null}).
     */
    protected LicenseRepository next;
    protected List<LicenseDefinition> definitions;
    protected final Map<LicenseDefinition, License> cache;
    /**
     * flag to known if repository was init (pass to {@code true}
     * when invoking the method {@link #load()}).
     */
    protected boolean init;

    public LicenseRepository() {
        cache = new HashMap<LicenseDefinition, License>();
    }

    public URL getBaseURL() {
        return baseURL;
    }

    public void setBaseURL(URL baseURL) {
        checkNotInit();
        this.baseURL = baseURL;
    }

    public LicenseRepository appendRepository(LicenseRepository next) {
        LicenseRepository lastRepository = getLastRepository();
        lastRepository.next = next;
        return next;
    }

    public synchronized void reload() throws IOException {
        if (next != null) {
            next.reload();
        }
        init = false;
        cache.clear();
        load();
    }

    public synchronized void load() throws IOException {
        if (!init) {
            if (baseURL == null || baseURL.toString().trim().isEmpty()) {
                throw new IllegalStateException("no baseURL defined in " + this);
            }

            URL definitionURL = getUrl(baseURL, REPOSITORY_DEFINITION_FILE);
            this.definitions = new ArrayList<LicenseDefinition>();

            if (!checkExists(definitionURL)) {
                throw new IllegalArgumentException("no licenses.properties found withurl [" + definitionURL + "] for resolver " + this);
            }
            Properties p = new Properties();
            p.load(definitionURL.openStream());

            for (Entry<Object, Object> entry : p.entrySet()) {
                String licenseName = (String) entry.getKey();
                String licenseDescription = (String) entry.getValue();

                licenseName = licenseName.trim().toLowerCase();

                URL licenseURL = getUrl(baseURL, licenseName);

                LicenseDefinition def = new LicenseDefinition(licenseURL, licenseName, licenseDescription);
                definitions.add(def);
            }
            definitions = Collections.unmodifiableList(definitions);
        }
        if (next != null) {
            next.load();
        }
        // mark repository as available
        init = true;
    }

    public List<LicenseDefinition> getAllDefinitions() {
        LicenseRepository[] repos = getAllRepositories();
        List<LicenseDefinition> result = new ArrayList<LicenseDefinition>(repos.length);
        for (LicenseRepository repo : repos) {
            result.addAll(repo.definitions);
        }
        return result;
    }

    public List<LicenseDefinition> getDefinitions() {
        return definitions;
    }

    public LicenseDefinition getDefinition(String licenseName) {
        checkInit();
        if (licenseName == null || licenseName.trim().isEmpty()) {
            throw new IllegalArgumentException("licenceName can not be null, nor empty");
        }
        licenseName = licenseName.trim().toLowerCase();
        LicenseDefinition definition = null;
        for (LicenseDefinition d : definitions) {
            if (licenseName.equals(d.getName())) {
                definition = d;
                break;
            }
        }
        if (definition == null && next != null) {
            definition = next.getDefinition(licenseName);
        }
        return definition;
    }

    public License getLicense(String licenseName) throws IOException {
        checkInit();
        if (licenseName == null || licenseName.trim().isEmpty()) {
            throw new IllegalArgumentException("licenceName can not be null, nor empty");
        }

        LicenseDefinition definition = getDefinition(licenseName);

        License license = null;
        if (definition != null) {
            synchronized (cache) {
                license = cache.get(definition);
                if (license == null) {

                    URL licenseURL = definition.getLicenseURL();
                    URL headerURL = definition.getHeaderURL();

                    checkExists(licenseURL);
                    checkExists(headerURL);

                    license = new License();
                    license.setName(definition.getName());
                    license.setDescription(definition.getDescription());
                    license.setLicenseURL(licenseURL);
                    license.setHeaderURL(headerURL);
                    cache.put(definition, license);
                }
            }
        }
        return license;
    }

    public static URL getUrl(URL baseUrl, String suffix) throws IllegalArgumentException {
        String url = baseUrl.toString() + "/" + suffix;
        try {
            return new URL(url);
        } catch (MalformedURLException ex) {
            throw new IllegalArgumentException("could not obtain url " + url, ex);
        }
    }

    protected LicenseRepository getLastRepository() {
        LicenseRepository last = next == null ? this : next.getLastRepository();
        return last;
    }

    protected LicenseRepository[] getAllRepositories() {
        List<LicenseRepository> list = new ArrayList<LicenseRepository>();
        LicenseRepository repo = this;
        while (repo != null) {
            list.add(repo);
            repo = repo.next;
        }
        return list.toArray(new LicenseRepository[list.size()]);
    }

    protected boolean checkExists(URL url) throws IOException {
        URLConnection openConnection = url.openConnection();
        return openConnection.getContentLength() > 0;
    }

    protected void checkNotInit() throws IllegalStateException {
        if (init) {
            throw new IllegalStateException("license repository " + this + " was already initialize...");
        }
    }

    protected void checkInit() throws IllegalStateException {
        if (!init) {
            throw new IllegalStateException("repository " + this + " is not init, use the load method before any license request");
        }
    }
}
