/*
 * *##% 
 * Maven helper plugin
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.license.plugin;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.metadata.ArtifactMetadataSource;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.resolver.ArtifactCollector;
import org.apache.maven.artifact.resolver.filter.ArtifactFilter;
import org.apache.maven.artifact.resolver.filter.ScopeArtifactFilter;
import org.apache.maven.model.License;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectBuilder;
import org.apache.maven.project.ProjectBuildingException;
import org.apache.maven.shared.dependency.tree.DependencyNode;
import org.apache.maven.shared.dependency.tree.DependencyTreeBuilder;
import org.apache.maven.shared.dependency.tree.DependencyTreeBuilderException;

import java.io.File;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;
import org.nuiton.plugin.AbstractPlugin;
import org.nuiton.plugin.PluginHelper;

/**
 * Le goal pour copier le fichier THIRD-PARTY.txt (contenant les licenses de toutes les dependances du projet)
 * dans le classpath (et le generer s'il n'existe pas).
 *
 * @author chemit
 * @goal add-third-party
 * @phase generate-resources
 * @requiresDependencyResolution test
 * @requiresProject true
 */
public class AddThirdPartyFilePlugin extends AbstractPlugin {

    private static final String unknownLicenseMessage = "Unknown license";
    /**
     * Dependance du projet.
     *
     * @parameter default-value="${project}"
     * @required
     * @readonly
     * @since 1.0.0
     */
    protected MavenProject project;
    /**
     * Fichier ou ecrire les licences des dependances.
     *
     * @parameter expression="${helper.thirdPartyFilename}" default-value="THIRD-PARTY.txt"
     * @required
     * @since 1.0.0
     */
    protected String thirdPartyFilename;
    /**
     * Repertoire de sortie des classes (classpath).
     *
     * @parameter expression="${helper.outputDirectory}" default-value="target/generated-sources/resources"
     * @required
     * @since 1.0.0
     */
    protected File outputDirectory;
    /**
     * Encoding a utiliser pour lire et ecrire les fichiers.
     *
     * @parameter expression="${helper.encoding}" default-value="${project.build.sourceEncoding}"
     * @required
     * @since 1.0.0
     */
    protected String encoding;
    /**
     * Un flag pour forcer la generation.
     *
     * @parameter expression="${helper.force}"  default-value="false"
     * @since 1.0.0
     */
    protected boolean force;
    /**
     * Un flag pour conserver un backup des fichiers modifies.
     *
     * @parameter expression="${helper.keepBackup}"  default-value="false"
     * @since 1.0.0
     */
    protected boolean keepBackup;
    /**
     * Un flag pour farie une copie nommé dans META-INF (prefixe avec le nom de l'artifact).
     *
     * @parameter expression="${helper.copyToMETA_INF}"  default-value="false"
     * @since 1.0.0
     */
    protected boolean copyToMETA_INF;
    /**
     * Un flag pour activer le mode verbeux.
     *
     * @parameter expression="${helper.verbose}"  default-value="${maven.verbose}"
     * @since 1.0.0
     */
    protected boolean verbose;
    /**
     * Local Repository.
     *
     * @parameter expression="${localRepository}"
     * @required
     * @readonly
     * @since 1.0.0
     */
    protected ArtifactRepository localRepository;
    /**
     * Remote repositories used for the project.
     *
     * @parameter expression="${project.remoteArtifactRepositories}"
     * @required
     * @readonly
     * @since 1.0.0
     */
    protected List<?> remoteRepositories;
    /**
     * Dependency tree builder component.
     *
     * @component
     */
    protected DependencyTreeBuilder dependencyTreeBuilder;
    /**
     * Artifact Factory component.
     *
     * @component
     */
    protected ArtifactFactory factory;
    /**
     * Artifact metadata source component.
     *
     * @component
     */
    protected ArtifactMetadataSource artifactMetadataSource;
    /**
     * Artifact collector component.
     *
     * @component
     */
    protected ArtifactCollector collector;
    /**
     * Maven Project Builder component.
     *
     * @component
     */
    protected MavenProjectBuilder mavenProjectBuilder;
    /**
     * content of third party file (only computed if {@link #force} is active or the
     * {@link #thirdPartyFile} does not exist, or is not up-to-date.
     */
    protected String thirdPartyFileContent;
    protected File thirdPartyFile;

    public AddThirdPartyFilePlugin() {
        super("all files are up-to-date.");
    }

    @Override
    protected boolean checkPackaging() {
        return rejectPackaging(Packaging.pom);
    }
    boolean doGenerate;

    @Override
    protected boolean init() throws Exception {

        doGenerate = true;
        thirdPartyFile = new File(outputDirectory, thirdPartyFilename);

        if (!force) {
            // regenerate only if file exists and is newer than pom file
            doGenerate = !isFileNewerThanPomFile(thirdPartyFile);
        }

        if (doGenerate) {

            // prepare thirdPartyFileContent

            DependencyNode dependencyTreeNode = resolveProject();

            LicenseMap licenseMap = new LicenseMap();

            for (Object o : dependencyTreeNode.getChildren()) {

                buildLicenseMap((DependencyNode) o, licenseMap);
            }

            thirdPartyFileContent = buildGroupedLicenses(licenseMap);

            // log dependencies with no license
            SortedSet<String> dependenciesWithNoLicense = licenseMap.get(unknownLicenseMessage);
            if (dependenciesWithNoLicense != null) {
                for (String dep : dependenciesWithNoLicense) {
                    // no license found for the dependency
                    getLog().warn("no license found for dependency " + dep);
                }
            }
        } else {
            thirdPartyFileContent = PluginHelper.readAsString(thirdPartyFile, encoding);
        }

        return true;
    }

    @Override
    protected void doAction() throws Exception {
        if (doGenerate) {
            if (verbose) {
                getLog().info("writing third-party file : " + thirdPartyFile);
            }
            if (keepBackup && thirdPartyFile.exists()) {
                if (verbose) {
                    getLog().info("backup " + thirdPartyFile);
                }
                thirdPartyFile.renameTo(new File(thirdPartyFile.getAbsolutePath() + '~'));
            }
            writeFile(thirdPartyFile, thirdPartyFileContent, encoding);
        }
        if (copyToMETA_INF) {
            copyFile(thirdPartyFile, new File(outputDirectory, "META-INF" + File.separator + project.getArtifactId() + "-" + thirdPartyFile.getName()));
        }
        addResourceDir(outputDirectory.getAbsolutePath());
    }

    /** @return resolve the dependency tree */
    protected DependencyNode resolveProject() {
        try {
            ArtifactFilter artifactFilter = new ScopeArtifactFilter(Artifact.SCOPE_TEST);
            return dependencyTreeBuilder.buildDependencyTree(project, localRepository, factory,
                    artifactMetadataSource, artifactFilter, collector);
        } catch (DependencyTreeBuilderException e) {
            getLog().error("Unable to build dependency tree.", e);
            return null;
        }
    }

    protected void buildLicenseMap(DependencyNode node, LicenseMap licenseMap) {
        if (node.getState() != DependencyNode.INCLUDED) {
            // this dependency is not included, so do not treate it
            if (verbose) {
                getLog().info("do not include this dependency " + node.toNodeString());
            }
            return;
        }
        Artifact artifact = node.getArtifact();

        if (verbose && getLog().isDebugEnabled()) {
            getLog().debug("treate node " + node.toNodeString());
        }

        if (!Artifact.SCOPE_SYSTEM.equals(artifact.getScope())) {
            try {
                MavenProject artifactProject = getMavenProjectFromRepository(artifact);
                String artifactName = getArtifactName(artifactProject);

                List<?> licenses = artifactProject.getLicenses();

                if (licenses.isEmpty()) {
                    // no license found for the dependency
                    licenseMap.put(unknownLicenseMessage, artifactName);

                } else {
                    for (Object o : licenses) {
                        if (o == null) {
                            getLog().warn("could not acquire the license for " + artifactName);
                            continue;
                        }
                        License license = (License) o;
                        String licenseKey = license.getName();
                        if (license.getName() == null) {
                            licenseKey = license.getUrl();
                        }
                        licenseMap.put(licenseKey, artifactName);
                    }
                }
            } catch (ProjectBuildingException e) {
                getLog().error("ProjectBuildingException error : ", e);
            }
        }
        if (!node.getChildren().isEmpty()) {
            for (Object o : node.getChildren()) {
                buildLicenseMap((DependencyNode) o, licenseMap);
            }
        }
    }

    protected String buildGroupedLicenses(LicenseMap licenseMap) {
        StringBuilder sb = new StringBuilder();
        sb.append("List of third-party dependencies grouped by their license type.");
        for (String licenseName : licenseMap.keySet()) {
            sb.append("\n\n").append(licenseName).append(" : ");

            SortedSet<String> projects = licenseMap.get(licenseName);

            for (String projectName : projects) {
                sb.append("\n  * ").append(projectName);
            }
        }
        return sb.toString();
    }

    protected String getArtifactName(MavenProject artifactProject) {
        StringBuilder sb = new StringBuilder();

        sb.append(artifactProject.getName());
        sb.append(" (");
        sb.append(artifactProject.getGroupId());
        sb.append(":");
        sb.append(artifactProject.getArtifactId());
        sb.append(":");
        sb.append(artifactProject.getVersion());
        sb.append(" - ");
        String url = artifactProject.getUrl();
        sb.append(url == null ? "no url defined" : url);
        sb.append(")");

        return sb.toString();
    }

    /**
     * Get the <code>Maven project</code> from the repository depending the <code>Artifact</code> given.
     *
     * @param artifact an artifact
     * @return the Maven project for the given artifact
     * @throws ProjectBuildingException if any
     */
    protected MavenProject getMavenProjectFromRepository(Artifact artifact)
            throws ProjectBuildingException {

        boolean allowStubModel = false;

        if (!"pom".equals(artifact.getType())) {
            artifact = factory.createProjectArtifact(artifact.getGroupId(), artifact.getArtifactId(),
                    artifact.getVersion(), artifact.getScope());
            allowStubModel = true;
        }

        // TODO: we should use the MavenMetadataSource instead
        return mavenProjectBuilder.buildFromRepository(artifact, remoteRepositories, localRepository,
                allowStubModel);
    }

    protected class LicenseMap extends java.util.TreeMap<String, SortedSet<String>> {

        private static final long serialVersionUID = 864199843545688069L;

        public SortedSet<String> put(String key, String value) {
            // handle multiple values as a set to avoid duplicates
            SortedSet<String> valueList = get(key);
            if (valueList == null) {
                valueList = new TreeSet<String>();
            }
            if (getLog().isDebugEnabled()) {
                getLog().debug("key:" + key + ",value: " + value);
            }
            valueList.add(value);
            return put(key, valueList);
        }
    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    @Override
    public MavenProject getProject() {
        return project;
    }

    @Override
    public void setProject(MavenProject project) {
        this.project = project;
    }
}
