/*
 * #%L
 * Maven helper plugin
 * 
 * $Id: DefaultPlexusCipher.java 701 2010-04-15 14:01:44Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/maven-helper-plugin/tags/maven-helper-plugin-1.2.6/src/main/java/org/nuiton/components/cipher/DefaultPlexusCipher.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.components.cipher;

import org.codehaus.plexus.logging.AbstractLogEnabled;

import java.security.Provider;
import java.security.Security;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.sonatype.plexus.components.cipher.PBECipher;
import org.sonatype.plexus.components.cipher.PlexusCipher;
import org.sonatype.plexus.components.cipher.PlexusCipherException;

/**
 * This class was forked from plexus-cipher:1.4.
 *
 * Since maven 2.2.x can not use this plexus component in a mojo
 * from the original artifact.
 *
 * @plexus.component role="org.sonatype.plexus.components.cipher.PlexusCipher" role-hint="maven-helper-plugin"
 * @author Oleg Gusakov</a>
 */
public class DefaultPlexusCipher
extends AbstractLogEnabled
implements PlexusCipher
{

    private static final Pattern ENCRYPTED_STRING_PATTERN = Pattern.compile( ".*?[^\\\\]?\\{(.*?[^\\\\])\\}.*" );

    private final PBECipher _cipher;

    // ---------------------------------------------------------------
    public DefaultPlexusCipher()
    throws PlexusCipherException
    {
        _cipher = new PBECipher();
    }
    // ---------------------------------------------------------------
    public String encrypt( final String str, final String passPhrase )
    throws PlexusCipherException
    {
        if ( str == null || str.length() < 1 )
        {
            return str;
        }

        return _cipher.encrypt64( str, passPhrase );
    }

    // ---------------------------------------------------------------
    public String encryptAndDecorate( final String str, final String passPhrase )
        throws PlexusCipherException
    {
        return decorate( encrypt( str, passPhrase ) );
    }

    // ---------------------------------------------------------------
    public String decrypt( final String str, final String passPhrase )
        throws PlexusCipherException
    {
        if ( str == null || str.length() < 1 )
        {
            return str;
        }

        return _cipher.decrypt64( str, passPhrase );
    }

    // ---------------------------------------------------------------
    public String decryptDecorated( final String str, final String passPhrase )
        throws PlexusCipherException
    {
        if ( str == null || str.length() < 1 )
        {
            return str;
        }

        if ( isEncryptedString( str ) )
        {
            return decrypt( unDecorate( str ), passPhrase );
        }

        return decrypt( str, passPhrase );
    }

    // ----------------------------------------------------------------------------
    public boolean isEncryptedString( final String str )
    {
        if ( str == null || str.length() < 1 )
        {
            return false;
        }

        Matcher matcher = ENCRYPTED_STRING_PATTERN.matcher( str );

        return matcher.matches() || matcher.find();
    }

    // ----------------------------------------------------------------------------
    // -------------------
    public String unDecorate( final String str )
        throws PlexusCipherException
    {
        Matcher matcher = ENCRYPTED_STRING_PATTERN.matcher( str );

        if ( matcher.matches() || matcher.find() )
        {
            return matcher.group( 1 );
        }
        else
        {
            throw new PlexusCipherException( "default.plexus.cipher.badEncryptedPassword" );
        }
    }

    // ----------------------------------------------------------------------------
    // -------------------
    public String decorate( final String str )
    {
        return ENCRYPTED_STRING_DECORATION_START + ( str == null ? "" : str ) + ENCRYPTED_STRING_DECORATION_STOP;
    }

    // ---------------------------------------------------------------
    // ---------------------------------------------------------------
    // ***************************************************************
    /**
     * Exploratory part. This method returns all available services types
     */
    public static String[] getServiceTypes()
    {
        Set result = new HashSet();

        // All all providers
        Provider[] providers = Security.getProviders();
        for ( int i = 0; i < providers.length; i++ )
        {
            // Get services provided by each provider
            Set keys = providers[i].keySet();
            for ( Iterator it = keys.iterator(); it.hasNext(); )
            {
                String key = (String) it.next();
                key = key.split( " " )[0];

                if ( key.startsWith( "Alg.Alias." ) )
                {
                    // Strip the alias
                    key = key.substring( 10 );
                }
                int ix = key.indexOf( '.' );
                result.add( key.substring( 0, ix ) );
            }
        }
        return (String[]) result.toArray( new String[result.size()] );
    }

    /**
     * This method returns the available implementations for a service type
     */
    public static String[] getCryptoImpls( final String serviceType )
    {
        Set result = new HashSet();

        // All all providers
        Provider[] providers = Security.getProviders();
        for ( int i = 0; i < providers.length; i++ )
        {
            // Get services provided by each provider
            Set keys = providers[i].keySet();
            for ( Iterator it = keys.iterator(); it.hasNext(); )
            {
                String key = (String) it.next();
                key = key.split( " " )[0];

                if ( key.startsWith( serviceType + "." ) )
                {
                    result.add( key.substring( serviceType.length() + 1 ) );
                }
                else if ( key.startsWith( "Alg.Alias." + serviceType + "." ) )
                {
                    // This is an alias
                    result.add( key.substring( serviceType.length() + 11 ) );
                }
            }
        }
        return (String[]) result.toArray( new String[result.size()] );
    }

    // ---------------------------------------------------------------
    public static void main( final String[] args )
    {
//        Security.addProvider( new BouncyCastleProvider() );

        String[] serviceTypes = getServiceTypes();
        if ( serviceTypes != null )
        {
            for ( int i = 0; i < serviceTypes.length; i++ )
            {
                String serviceType = serviceTypes[i];
                String[] serviceProviders = getCryptoImpls( serviceType );
                if ( serviceProviders != null )
                {
                    System.out.println( serviceType + ": provider list" );
                    for ( int j = 0; j < serviceProviders.length; j++ )
                    {
                        String provider = serviceProviders[j];
                        System.out.println( "        " + provider );
                    }
                }
                else
                {
                    System.out.println( serviceType + ": does not have any providers in this environment" );
                }
            }
        }
    }
    //---------------------------------------------------------------
    //---------------------------------------------------------------
}
